/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.util;

import java.io.File;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.common.Environment;

/**
 * Ftp handling utilities.
 * @author b_georges
 */
public class FtpUtils {
	
	private static final Logger logger = Logger.getLogger(FtpUtils.class);

	/**
	 * Transforms File path to FTP path
	 * @param p_oF The file to transform
	 * @return The modified path
	 */
	public static String fileToFtpString(File p_oF) {
		return (null == p_oF) ? null : p_oF.toString().replace("\\", "/");
	}
	
	public static String getLocalDir() {
		String s = trimToNull(Configuration.getFtpLocalDir());
		if (s == null) {
			s = trimToNull(System.getProperty("java.io.tmpdir"));
			if (s == null) {
				s = "/tmp";
				if (logger.isEnabledFor(Level.WARN)) { 
					logger.warn("property [" + Environment.FTP_LOCALDIR + "] and ${java.io.tmpdir} undefined; defaulting to [" + s + "].");
				}
			}
		}
		if (logger.isEnabledFor(Level.WARN)) {
			File f = new File(s);
			if (!f.exists() || !f.isDirectory() || !f.canRead() || !f.canWrite()) {
				StringBuilder sb = new StringBuilder();
				sb.append("getLocalDir() returning [");
				sb.append(s);
				sb.append("] however [");
				sb.append(f.getAbsolutePath());
				sb.append("]");
				if (!f.exists()) {
					sb.append(" does not exist");
				} else if (!f.isDirectory()) {
					sb.append(" is not a directory");
				} else {
					boolean and = false;
					if (!f.canRead()) {
						sb.append(" cannot be read from");
						and = true;
					}
					if (!f.canWrite()) {
						if (and) {
							sb.append(" and");
						}
						sb.append(" cannot be written to");
					}
				}
				sb.append(".");
				logger.warn(sb.toString());
			}
		}
		return s;
	}
	
	public static String getRemoteDir() {
		String s = trimToNull(Configuration.getFtpRemoteDir());
		if (s == null) {
			s = trimToNull(System.getProperty("java.io.tmpdir"));
			if (s == null) {
				s = "/tmp";
				if (logger.isEnabledFor(Level.WARN)) {
					logger.warn("property [" + Environment.FTP_REMOTEDIR + "] and ${java.io.tmpdir} undefined; defaulting to [" + s + "].");
				}
			}
		}
		return s;
	}
	
	public static int getRenameRetry() throws ConfigurationException {
		return getIntProperty(Environment.FTP_RENAME_RETRY, Configuration.getFtpRenameRetry(), 1, 10);
	}
	
	/**
	 * Coarse-grained ftp timeout property in milliseconds (0 = infinite).
	 * The getTimeout...() properties below will default to this value if they are not set.
	 * 
	 * @return timeout
	 * @throws ConfigurationException if a NumberFormatException is encountered
	 */
	public static int getTimeout() throws ConfigurationException {
		return getIntProperty(Environment.FTP_TIMEOUT, Configuration.getFtpTimeout(), 0, 0);
	}
	
	/**
	 * Fine-grained ftp timeout property in milliseconds (0 = infinite) used for org.apache.commons.net.SocketClient.setDefaultTimeout(int):void,
	 * which <i>sets the default timeout to use when opening a socket.  It is used only previous to a
	 * call to connect() and should not be confused with setSoTimeout() which operates on the currently opened socket.</i>
	 * 
	 * @return timeoutDefault, or getTimeout() if not set
	 * @throws ConfigurationException if a NumberFormatException is encountered
	 */
	public static int getTimeoutDefault() throws ConfigurationException {
		return getIntProperty(Environment.FTP_TIMEOUT_DEFAULT, Configuration.getFtpTimeoutDefault(), 0, getTimeout());
	}
	
	/**
	 * Fine-grained ftp timeout property in milliseconds (0 = infinite) used for org.apache.commons.net.ftp.FTPClient.setDataTimeout(int):void,
	 * which <i>sets the timeout to use when reading from the data connection</i>.
	 * 
	 * @return timeoutData, or getTimeout() if not set
	 * @throws ConfigurationException if a NumberFormatException is encountered
	 */
	public static int getTimeoutData() throws ConfigurationException {
		return getIntProperty(Environment.FTP_TIMEOUT_DATA, Configuration.getFtpTimeoutData(), 0, getTimeout());
	}
	
	/**
	 * Fine-grained ftp timeout property in milliseconds (0 = infinite) used for java.net.Socket.setSoTimeout(int):void,
	 * which <i>sets the timeout of a currently open connection</i>.
	 * 
	 * @return timeoutSo, or getTimeout() if not set
	 * @throws ConfigurationException if a NumberFormatException is encountered
	 */
	public static int getTimeoutSo() throws ConfigurationException {
		return getIntProperty(Environment.FTP_TIMEOUT_SO, Configuration.getFtpTimeoutSo(), 0, getTimeout());
	}
	
	private static int getIntProperty(String configName, String configValue, int minValue, int defValue) throws ConfigurationException {
		int retValue;
		String s = trimToNull(configValue);
		if (s != null) {
			try {
				retValue = Integer.parseInt(s);
			} catch (NumberFormatException e) {
				throw new ConfigurationException("could not parse property [" + configName + "] value [" + s + "] to an int.", e);
			}
		} else {
			retValue = defValue;
		}
		if (retValue < minValue) {
			if (logger.isEnabledFor(Level.WARN)) {
				logger.warn("property [" + configName + "] value [" + retValue + "] illegal; defaulting to [" + defValue + "].");
			}
			retValue = defValue;
		}
		return retValue;
	}
	
	private static String trimToNull(String s) {
		if (s != null) {
			s = s.trim();
			if (s.length() == 0) {
				s = null;
			}
		}
		return s;
	}
	
}
