/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.message;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.filter.InputOutputFilter;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.util.ClassUtil;

import com.arjuna.common.util.propertyservice.PropertyManager;

import java.util.ArrayList;
import java.util.StringTokenizer;

/**
 * The InterceptorManager parses the PipelineInterceptors from the configuration
 * and provides methods which will execute the methods provided by the interceptors. 
 *  
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a> 
 */
public class InterceptorManager {
    public final static Logger logger = Logger.getLogger(InterceptorManager.class);

    private static InterceptorManager _instance = null;
	private ArrayList<PipelineInterceptor> failureInterceptors = new ArrayList<PipelineInterceptor>();
	private ArrayList<PipelineInterceptor> instantiateInterceptors = new ArrayList<PipelineInterceptor>();
	private ArrayList<PipelineInterceptor> startInterceptors = new ArrayList<PipelineInterceptor>();
	private ArrayList<PipelineInterceptor> endInterceptors = new ArrayList<PipelineInterceptor>();

    private InterceptorManager() {
        PropertyManager pm = ModulePropertyManager.getPropertyManager(ModulePropertyManager.INTERCEPTOR_MODULE);
        
        loadInterceptors(failureInterceptors, pm.getProperty(Environment.PROCESSING_PIPELINE_FAILURE_INTERCEPTORS));
        loadInterceptors(endInterceptors, pm.getProperty(Environment.PROCESSING_PIPELINE_END_INTERCEPTORS));
        loadInterceptors(startInterceptors, pm.getProperty(Environment.PROCESSING_PIPELINE_START_INTERCEPTORS));
        loadInterceptors(instantiateInterceptors, pm.getProperty(Environment.PROCESSING_PIPELINE_INSTANTIATION_INTERCEPTORS));
    }
	
    public static final synchronized InterceptorManager getInstance() {
        if (_instance == null)
            _instance = new InterceptorManager();

        return _instance;
    }

    private void loadInterceptors(ArrayList<PipelineInterceptor> al, String props) {
    	if ((props != null) && (! "".equals(props))) {
    		StringTokenizer tokenizer = new StringTokenizer(props, ",");

    		while (tokenizer.hasMoreTokens()) {	
    			String token = tokenizer.nextToken();
    			try {
	    			Class c = ClassUtil.forName(token, getClass());
	    			PipelineInterceptor interceptor = (PipelineInterceptor) c.newInstance();
	    			al.add(interceptor);
    			} catch (ClassNotFoundException ex) {
    				logger.warn("FilterManager problem loading class " + token, ex);
    			} catch (Throwable ex) {
    				logger.warn("FilterManager problem during load " + token, ex);
    			}
    		}
    	}
    }
    	
    public Message interceptFailure(Message msg, ConfigTree config) {
    	if (failureInterceptors == null)
    		return msg;
    	
    	for (PipelineInterceptor interceptor : failureInterceptors) {
    		interceptor.processMessage(msg, config);
    	}
    	
    	return msg;
    }
    
    public Message interceptEnding(Message msg, ConfigTree config) {
    	if (endInterceptors == null)
    		return msg;
    	
    	for (PipelineInterceptor interceptor : endInterceptors) {
    		interceptor.processMessage(msg, config);
    	}
    	
    	return msg;
    }

    
    public Message interceptStart(Message msg, ConfigTree config) {
    	if (startInterceptors == null)
    		return msg;
    	
    	for (PipelineInterceptor interceptor : startInterceptors) {
    		interceptor.processMessage(msg, config);
    	}
    	
    	return msg;
    }

    public Message interceptInstantiate(Message msg, ConfigTree config) {
    	if (instantiateInterceptors == null)
    		return msg;
    	
    	for (PipelineInterceptor interceptor : instantiateInterceptors) {
    		interceptor.processMessage(msg, config);
    	}
    	
    	return msg;
    }
}
