/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
package org.jboss.soa.esb.services.security;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;

import org.jboss.logging.Logger;
import org.jboss.security.plugins.FilePassword;


/**
 * Password plugin supporting FilePlugin.
 */
public class FilePasswordPlugin implements PasswordPlugin
{
	/**
	 * The logger for this plugin.
	 */
	private final Logger logger = Logger.getLogger(getClass()) ;
	
	/**
	 * Does the password plugin support the specified uri?
	 * @param uri The uri representing the password file.
	 * @return true if supported, false otherwise
	 */
	public boolean supportsPasswordFile(final URI uri)
	{
		try
		{
			uri.toURL() ;
			return true ;
		}
		catch (final IllegalArgumentException iae)
		{
			return false ;
		}
		catch (final MalformedURLException murle)
		{
			return false ;
		}
	}

	/**
	 * Return the plugin handler for the specified uri.
	 * @param uri The uri representing the password file.
	 * @return The PluginHandler or null if not supported
	 */
	public PasswordHandler pluginHandler(final URI uri)
	{
		final URL url ;
		try
		{
			url = uri.toURL() ;
		}
		catch (final IllegalArgumentException iae)
		{
			logger.warn("URI was not absolute", iae) ;
			return null ;
		}
		catch (final MalformedURLException murle)
		{
			logger.warn("Could not recognize URI", murle) ;
			return null ;
		}
		
		return new FilePasswordHandler(url) ;
	}
	
	/**
	 * The password handler for FilePassword
	 * @author kevin
	 */
	private static class FilePasswordHandler implements PasswordHandler
	{
		private final FilePassword filePassword ;
		
		FilePasswordHandler(final URL url)
		{
			if ("file".equals(url.getProtocol()))
			{
				// FilePassword in AS4 does not support URLs
				filePassword = new FilePassword(url.getFile()) ;
			}
			else
			{
				filePassword = new FilePassword(url.toExternalForm()) ;
			}
		}

		/**
		 * Retrieve the password
		 * @return the password
		 */
		public String getPassword()
			throws IOException
		{
			return new String(filePassword.toCharArray()) ;
		}
	}
}
