/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.internal.soa.esb.soap;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.xml.namespace.NamespaceContext;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import junit.framework.TestCase;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

/**
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class OGNLUtilsUnitTest extends TestCase {

    public void test() throws IOException, SAXException, ConfigurationException, XPathExpressionException {
        Node node;

        // Test regular node...
        node = getNode("/soapenv:Envelope/soapenv:Body/cus:customerOrder/cus:header/cus:customerName");
        assertEquals("customerOrder.header.customerName", OGNLUtils.getOGNLExpression((Element) node));

        // Test collection nodes...
        node = getNode("/soapenv:Envelope/soapenv:Body/cus:customerOrder/cus:items/cus:item[1]");
        assertEquals("customerOrder.items[0]", OGNLUtils.getOGNLExpression((Element) node));

        node = getNode("/soapenv:Envelope/soapenv:Body/cus:customerOrder/cus:items/cus:item[2]");
        assertEquals("customerOrder.items[1]", OGNLUtils.getOGNLExpression((Element) node));

        node = getNode("/soapenv:Envelope/soapenv:Body/cus:customerOrder/cus:items/cus:item[2]/cus:partNumber");
        assertEquals("customerOrder.items[1].partNumber", OGNLUtils.getOGNLExpression((Element) node));
        
        node = getNode("/soapenv:Envelope/soapenv:Body/cus:customerOrder/cus:items/cus:item[2]/cus:partNumber");
        final String nameSpace = "http://schemas.xmlsoap.org/soap/envelope/";
        assertEquals("customerOrder.items[1].partNumber", OGNLUtils.getOGNLExpression((Element) node, nameSpace));

        node = getNode("/soapenv:Envelope/soapenv:Body/cus:customerOrder/cus:queryRequest/cus:username");
        assertEquals("customerOrder.queryRequest.username", OGNLUtils.getOGNLExpression((Element) node));
    }
    
    public void test_checkParentNameSpace( )
    {
        final String parentNS = "http://schemas.xmlsoap.org/soap/envelope/";
    	boolean exists = OGNLUtils.checkParentNameSpace( parentNS, null );
    	assertTrue( exists );
    	
    	final String customNS = "http://www.w3.org/2003/05/soap-envelope";
    	exists = OGNLUtils.checkParentNameSpace( parentNS, customNS );
    	assertTrue( exists );
    	exists = OGNLUtils.checkParentNameSpace( null, null );
    	assertFalse( exists );
    }

    private Node getNode(String xpath) throws IOException, SAXException, XPathExpressionException {
        Document order = YADOMUtil.parseStream(getClass().getResourceAsStream("ognl-test-01.xml"), false, false, true);
        XPathFactory xPathFactory = XPathFactory.newInstance();
        XPath xpathEvaluater = xPathFactory.newXPath();

        xpathEvaluater.setNamespaceContext(new MockNamespaceContext());

        return (Node) xpathEvaluater.evaluate(xpath, order, XPathConstants.NODE);        
    }

    class MockNamespaceContext implements NamespaceContext {

        private Map<String, String> namespaces = new HashMap<String, String>();
        private Map<String, String> prefixes = new HashMap<String, String>();

        public MockNamespaceContext() {
            namespaces.put("soapenv", "http://schemas.xmlsoap.org/soap/envelope/");
            namespaces.put("cus", "http://schemas.active-endpoints.com/sample/customerorder/2006/04/CustomerOrder.xsd");
            namespaces.put("stan", "http://schemas.active-endpoints.com/sample/standardtypes/2006/04/StandardTypes.xsd");
            prefixes.put("http://schemas.xmlsoap.org/soap/envelope/", "soapenv");
            prefixes.put("http://schemas.active-endpoints.com/sample/customerorder/2006/04/CustomerOrder.xsd", "cus");
            prefixes.put("http://schemas.active-endpoints.com/sample/standardtypes/2006/04/StandardTypes.xsd", "stan");
        }

        public String getNamespaceURI(String prefix) {
            return namespaces.get(prefix);
        }

        public String getPrefix(String namespaceURI) {
            return prefixes.get(namespaceURI);
        }

        public Iterator getPrefixes(String namespaceURI) {
            ArrayList<String> list = new ArrayList<String>();

            if(prefixes.containsKey(namespaceURI)) {
                list.add(prefixes.get(namespaceURI));
            }

            return list.iterator();
        }
    }
}
