/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.addressing.eprs;

import junit.framework.TestCase;
import org.jboss.internal.soa.esb.couriers.MockCourier;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.message.format.xml.MessageImpl;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.PortReference;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.ActionProcessingPipeline;
import org.jboss.soa.esb.listeners.message.ActionProcessingPipelineUnitTest;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.message.tests.XMLMessageUnitTest;
import org.jboss.soa.esb.mock.MockAction;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionProcessingException;

import java.net.URI;

/**
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class LogicalEPRUnitTest extends TestCase {

    public void test_valid_uri() {
        LogicalEPR epr;

        epr = new LogicalEPR("a", "b");
        assertEquals("logical:a#b", epr.getAddr().getAddress());
        assertEquals("a", epr.toService().getCategory());
        assertEquals("b", epr.toService().getName());

        // Make sure the parts are properly encoded/decoded - cat and name with spaces, slashes etc...
        epr = new LogicalEPR("My / Category", "My / Name");
        assertEquals("logical:My%20/%20Category#My%20/%20Name", epr.getAddr().getAddress());
        assertEquals("My / Category", epr.toService().getCategory());
        assertEquals("My / Name", epr.toService().getName());

        epr = new LogicalEPR(URI.create("logical:My%20/%20Category#My%20/%20Name"));
        assertEquals("logical:My%20/%20Category#My%20/%20Name", epr.getAddr().getAddress());
        assertEquals("My / Category", epr.toService().getCategory());
        assertEquals("My / Name", epr.toService().getName());

        epr = new LogicalEPR(new PortReference("logical:My%20/%20Category#My%20/%20Name"));
        assertEquals("logical:My%20/%20Category#My%20/%20Name", epr.getAddr().getAddress());
        assertEquals("My / Category", epr.toService().getCategory());
        assertEquals("My / Name", epr.toService().getName());
        
        epr = new LogicalEPR(new EPR(URI.create("logical:My%20/%20Category#My%20/%20Name")));
        assertEquals("logical:My%20/%20Category#My%20/%20Name", epr.getAddr().getAddress());
        assertEquals("My / Category", epr.toService().getCategory());
        assertEquals("My / Name", epr.toService().getName());

        epr = new LogicalEPR(new EPR(new PortReference("logical:My%20/%20Category#My%20/%20Name")));
        assertEquals("logical:My%20/%20Category#My%20/%20Name", epr.getAddr().getAddress());
        assertEquals("My / Category", epr.toService().getCategory());
        assertEquals("My / Name", epr.toService().getName());

        epr = new LogicalEPR();
        epr.setAddr(new PortReference("logical:My%20/%20Category#My%20/%20Name"));
        assertEquals("logical:My%20/%20Category#My%20/%20Name", epr.getAddr().getAddress());
        assertEquals("My / Category", epr.toService().getCategory());
        assertEquals("My / Name", epr.toService().getName());

        epr = new LogicalEPR("1234567890 -=`!\"£$%^&*()_+\\|,./;'#[]<>?:@~{}", "1234567890 -=`!\"£$%^&*()_+\\|,./;'#[]<>?:@~{}");
        assertEquals("logical:1234567890%20-=%60!%22£$%25%5E&*()_+%5C%7C,./;'%23[]%3C%3E?:@~%7B%7D#1234567890%20-=%60!%22£$%25%5E&*()_+%5C%7C,./;'%23[]%3C%3E?:@~%7B%7D", epr.getAddr().getAddress());
        assertEquals("1234567890 -=`!\"£$%^&*()_+\\|,./;'#[]<>?:@~{}", epr.toService().getCategory());
        assertEquals("1234567890 -=`!\"£$%^&*()_+\\|,./;'#[]<>?:@~{}", epr.toService().getName());

    }

    public void test_invalid_uri() {
        try {
            new LogicalEPR(URI.create("xxxx:a#b"));
            fail("Expected IllegalArgumentException");
        } catch (IllegalArgumentException e) {
            assertEquals("'xxxx:a#b' is not a valid URI for a Logical EPR - URI scheme must be 'logical'.", e.getMessage());
        }

        try {
            new LogicalEPR(new PortReference("logical://a#b"));
            fail("Expected IllegalArgumentException");
        } catch (IllegalArgumentException e) {
            assertEquals("'logical://a#b' is not a valid URI for a Logical EPR - URI must be opaque.", e.getMessage());
        }

        try {
            new LogicalEPR(new EPR(URI.create("xxxx:a#b")));
            fail("Expected IllegalArgumentException");
        } catch (IllegalArgumentException e) {
            assertEquals("'xxxx:a#b' is not a valid URI for a Logical EPR - URI scheme must be 'logical'.", e.getMessage());
        }

        try {
            new LogicalEPR(new EPR(new PortReference("logical://a#b")));
            fail("Expected IllegalArgumentException");
        } catch (IllegalArgumentException e) {
            assertEquals("'logical://a#b' is not a valid URI for a Logical EPR - URI must be opaque.", e.getMessage());
        }

        try {
            LogicalEPR epr = new LogicalEPR();
            epr.setAddr(new PortReference("logical:a"));
            fail("Expected IllegalArgumentException");
        } catch (IllegalArgumentException e) {
            assertEquals("'logical:a' is not a valid URI for a Logical EPR - no URI fragment (service name) part.", e.getMessage());
        }

        try {
            LogicalEPR epr = new LogicalEPR();
            epr.setAddr(new PortReference("logical:#b"));
            fail("Expected IllegalArgumentException");
        } catch (IllegalArgumentException e) {
            assertEquals("Expected scheme-specific part at index 8: logical:#b", e.getCause().getMessage());
        }
    }

    public void test_Serialize_Deserialize()
        throws Exception
    {
        Message msg = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);

        EPR epr = new LogicalEPR(new PortReference("logical:a#b"));

        msg.getHeader().getCall().setTo(epr);

        final String xmlRepresentation = XMLMessageUnitTest.msgToXML((MessageImpl)msg) ;

        final MessageImpl nImpl = XMLMessageUnitTest.msgFromXML(xmlRepresentation) ;

        EPR theEpr = nImpl.getHeader().getCall().getTo();

        assertTrue("Expected a Logical EPR", theEpr instanceof LogicalEPR);

        assertEquals("logical:a#b", theEpr.getAddr().getAddress());
    }

    private MockCourier courier1;
    private MockCourier courier2;

    public void setUp() throws Exception {
        MockCourierFactory.install();
        MockRegistry.install();

        courier1 = new MockCourier(true);
        courier2 = new MockCourier(true);
        MockRegistry.register("cat", "servicex", courier1);
        MockRegistry.register("cat", "servicey", courier2);
     }

    public void tearDown() throws Exception {
        MockRegistry.uninstall();
        MockCourierFactory.uninstall();
    }

    public void test_replyTo() throws ConfigurationException {
        ActionProcessingPipeline pipeline = createTestPipeline();
        Message message;

        message = MessageFactory.getInstance().getMessage();
        message.getHeader().getCall().setReplyTo(new LogicalEPR("cat", "servicex"));
        message.getHeader().getCall().setFaultTo(new LogicalEPR("cat", "servicey"));
        assertTrue(!courier1.deliveryAttempted);
        assertTrue(!courier2.deliveryAttempted);
        pipeline.process(message);
        assertTrue(courier1.deliveryAttempted);
        assertTrue(!courier2.deliveryAttempted);
    }

    public void test_faultTo() throws ConfigurationException {
        ActionProcessingPipeline pipeline = createTestPipeline();
        Message message;

        message = MessageFactory.getInstance().getMessage();
        message.getHeader().getCall().setReplyTo(new LogicalEPR("cat", "servicex"));
        message.getHeader().getCall().setFaultTo(new LogicalEPR("cat", "servicey"));
        assertTrue(!courier1.deliveryAttempted);
        assertTrue(!courier2.deliveryAttempted);
        MockAction.exception = new ActionProcessingException("blah");
        pipeline.process(message);
        assertTrue(!courier1.deliveryAttempted);
        assertTrue(courier2.deliveryAttempted);
    }

    private ActionProcessingPipeline createTestPipeline() throws ConfigurationException {
        ConfigTree pipelineConfig = new ConfigTree("pipline");

        ActionProcessingPipelineUnitTest.addAction(pipelineConfig, MockAction.class.getName());

        ActionProcessingPipeline pipeline = new ActionProcessingPipeline(pipelineConfig);
        pipeline.initialise();

        return pipeline;
    }
}
