/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.parameters;

import java.io.File;

import junit.framework.TestCase;

import org.jboss.internal.soa.esb.parameters.ParamFileRepository;
import org.jboss.soa.esb.common.ModulePropertyManager;

/**
 * ParamFileRepository unit tests.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class ParamsFileRepositoryUnitTest extends TestCase
{

	public void test_ParamsFileRepository() throws Exception
	{
		File root;
		ParamFileRepository fileRepo;

		// Check it defaults to the working dir...
		root = new File("./");
		fileRepo = new ParamFileRepository();
		assertEquals(root, fileRepo.getRoot());

		// Check it can pick up from the System prop...
		root = new File("build/tests/");
		ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).setProperty(
				ParamFileRepository.FILE_PARAMS_REPOS_ROOT, root.getPath());
		fileRepo = new ParamFileRepository();
		assertEquals(root, fileRepo.getRoot());

		// Check for error on invalid System prop...
		ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).setProperty(
				ParamFileRepository.FILE_PARAMS_REPOS_ROOT, "blah/blah");
		try
		{
			new ParamFileRepository();
			fail("expected IllegalStateException");
		} catch (IllegalStateException e)
		{
			// expected
		}
	}

	public void test_add_get_remove_badargs() throws Exception
	{
		ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).setProperty(
				ParamFileRepository.FILE_PARAMS_REPOS_ROOT, "build/tests/");
		ParamFileRepository fileRepo = new ParamFileRepository();

		test_add_badargs(fileRepo, null, "val");
		test_add_badargs(fileRepo, "", "val");
		test_add_badargs(fileRepo, "", "val");
		test_add_badargs(fileRepo, "xxx/yyy", null);

		test_get_badargs(fileRepo, null);
		test_get_badargs(fileRepo, "");

		test_remove_badargs(fileRepo, null);
		test_remove_badargs(fileRepo, "");
	}

	private void test_add_badargs(ParamFileRepository fileRepo, String name,
			String value) throws ParamRepositoryException
	{
		try
		{
			fileRepo.add(name, value);
			fail("expected IllegalArgumentException");
		} catch (IllegalArgumentException e)
		{
			// expected
		}
	}

	private void test_get_badargs(ParamFileRepository fileRepo, String name)
			throws ParamRepositoryException
	{
		try
		{
			fileRepo.get(name);
			fail("expected IllegalArgumentException");
		} catch (IllegalArgumentException e)
		{
			// expected
		}
	}

	private void test_remove_badargs(ParamFileRepository fileRepo, String name)
	{
		try
		{
			fileRepo.remove(name);
			fail("expected IllegalArgumentException");
		} catch (IllegalArgumentException e)
		{
			// expected
		}
	}

	public void test_add_get_remove() throws Exception
	{
		ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).setProperty(
				ParamFileRepository.FILE_PARAMS_REPOS_ROOT, "build/tests/");
		ParamFileRepository fileRepo = new ParamFileRepository();
		String name = "repostests/jboss/testparam";

		// Make sure the parameter doesn't already exist...
		assertEquals(null, fileRepo.get(name));

		// Now the tests...
		assertEquals(null, fileRepo.get(name));
		fileRepo.add(name, "tomtestvalue");
		assertEquals("tomtestvalue", fileRepo.get(name));
		fileRepo.remove(name);
		assertEquals(null, fileRepo.get(name));
	}
}
