/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security.auth.ws;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.io.InputStream;
import java.security.cert.Certificate;
import java.util.List;

import junit.framework.JUnit4TestAdapter;

import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.soa.esb.util.ClassUtil;
import org.junit.Test;

/**
 * Unittest for {@link BinarySecurityTokenImpl}
 *
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class BinarySecurityTokenUnitTest
{
	private String expectedEncoding = "Base64Binary";
	private BinarySecurityTokenImpl token = new BinarySecurityTokenImpl();

	@Test
	public void setEncodingTypeWithNSPrefix()
	{
		token.setEncodingType("wsse:" + expectedEncoding);
		assertEquals( expectedEncoding, token.getEncodingType() );
	}

	@Test
	public void setEncodingTypeWithNSPrefix2()
	{
		token.setEncodingType("d:" + expectedEncoding);
		assertEquals( expectedEncoding, token.getEncodingType() );
	}

	@Test
	public void setEncodingTypeWithoutNSPrefix()
	{
		token.setEncodingType(expectedEncoding);
		assertEquals( expectedEncoding, token.getEncodingType() );
	}

	@Test
	public void setEncodingTypeFullPath()
	{
		String url = "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-soap-message-security-1.0#Base64Binary";
		token.setEncodingType(url);
		assertEquals( expectedEncoding, token.getEncodingType() );
	}

	@Test
	public void setValueType()
	{
		String url = "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-x509-token-profile-1.0#X509v3";
		token.setValueType(url);
		assertEquals( BinarySecurityTokenImpl.X509V3, token.getValueType() );
	}

	@Test
	public void setEncodingType()
	{
		token.setEncodingType(expectedEncoding);
		assertEquals( expectedEncoding, token.getEncodingType() );
	}

	@Test
	public void setEncodingTypeNull()
	{
		token.setEncodingType(null);
		assertNull(token.getEncodingType());
	}

	@Test
	public void setKey() throws Exception
	{
		token.setEncodingType(expectedEncoding);
		token.setValueType(BinarySecurityTokenImpl.X509V3);
		token.setKey(getExampleCert());
		List keys = token.getKeys();
		assertTrue(keys.size() > 0);
	}

	private String getExampleCert() throws Exception
	{
		InputStream inputStream = ClassUtil.getResourceAsStream("cert-example.xml", getClass() );
		return new String(StreamUtils.readStream(inputStream));
	
	}
	
	/*
	 * Need to generate a test cert for this
	 * 
	@Test
	public void setPKIPATHV1Key() throws Exception
	{
		token.setEncodingType("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-soap-message-security-1.0#Base64Binary");
		token.setValueType("http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-x509-token-profile-1.0#X509PKIPathv1");
		token.setKey(getPKIPATHV1Cert());
		List keys = token.getKeys();
		assertTrue(keys.size() > 0);		
	}
	
	private String getPKIPATHV1Cert() throws Exception
	{
		InputStream inputStream = ClassUtil.getResourceAsStream("pkipathv1-cert-example.xml", getClass() );
		return new String(StreamUtils.readStream(inputStream));		
	}
	*/
	
	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter(BinarySecurityTokenUnitTest.class);
	}

}
