/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions.soap;

import java.io.IOException;
import java.net.URI;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.publish.ContractInfo;
import org.jboss.internal.soa.esb.publish.ContractProvider;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.listeners.config.Action;
import org.jboss.wsf.spi.deployment.Endpoint;

import java.util.Properties;

/**
 * Contract publisher for a JBossWS Webservice endpoint.
 * 
 * @author <a href="mageshbk@jboss.com">Magesh Kumar B</a>
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class JBossWSWebserviceContractPublisher extends AbstractWsdlContractPublisher implements ContractProvider {

    private static Logger logger = Logger.getLogger(JBossWSWebserviceContractPublisher.class);
    private String endpointName;
    private String contextName;

    /**
     * Set the {@link SOAPProcessor} action configuration.
     * @param actionConfig action config.
     * @throws ConfigurationException Bad config.
     */
    public void setActionConfig(Action actionConfig) throws ConfigurationException {
        // Initialise the abstract base class first...
        super.setActionConfig(actionConfig);

        endpointName = actionConfig.getProperties().getProperty(WebServiceUtils.JBOSSWS_ENDPOINT);
        contextName = actionConfig.getProperties().getProperty(WebServiceUtils.JBOSSWS_CONTEXT);
        if(endpointName == null) {
            throw new ConfigurationException("Property '" + WebServiceUtils.JBOSSWS_ENDPOINT + "' not specified.");
        }
    }

    public String getWsdlAddress() {
        Endpoint endpoint = WebServiceUtils.getServiceEndpoint(endpointName, contextName);

        if(endpoint != null) {
            return endpoint.getAddress() + "?wsdl";
        } else {
            logger.warn("Requested contract info for unknown webservice endpoint '" + endpointName + "'.");
            return null;
        }
    }

    public Properties getHttpClientProperties() {
        return new Properties();
    }

    @Override
    public void setContractProperties(Properties contractProperties) {
        setActionProperties(contractProperties);

        // Making this class as a ContractProvider creates two instances of this class. One for CP and the other for Publisher
        // This is needed for the ContractProvider instance.
        endpointName = getActionProperties().getProperty(WebServiceUtils.JBOSSWS_ENDPOINT);
        contextName = getActionProperties().getProperty(WebServiceUtils.JBOSSWS_CONTEXT);
    }

    public ContractInfo provideContract(Service service) throws IOException {
        return provideContract(service, null);
    }

    public ContractInfo provideContract(Service service, String endpointAddressOverride) throws IOException {
        ContractInfo contract;
        if (endpointAddressOverride != null) {
            contract = getContractInfo(new EPR(URI.create(endpointAddressOverride)));
        } else {
            contract = getContractInfo(service, getWsdlAddress());
        }
        return contract;
    }
}
