/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * (C) 2005-2008, JBoss Inc.
 */
package org.jboss.soa.esb.actions.soap;

import com.ibm.wsdl.Constants;
import com.ibm.wsdl.util.xml.QNameUtils;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Result;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;

import org.w3c.dom.Document;
import org.w3c.dom.NodeList;
import org.w3c.dom.Node;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;

public class SchemaLocationParser {

	private static final String IMPORT_ELEM = "import";
	private static final String SCHEMA_ELEM = "schema";
	private static final String PORT_ELEM = "port";
	private static final String SCHEMA_LOCATION_ATTR = "schemaLocation";
	private static final String SOAP_ADDRESS_LOCATION = "location";

	private String soapAddress;
	
	DocumentBuilderFactory dbFactory;
	DocumentBuilder dBuilder;
	SchemaDeployer deployer;
		
	
	/**
	 * The constructor for this class takes a deployer which is used deploy the XSD file given a filename.
	 * @param deployer
	 * @throws ParserConfigurationException
	 */
	public SchemaLocationParser(SchemaDeployer deployer) throws ParserConfigurationException {
		this.deployer = deployer;
		
		dbFactory = DocumentBuilderFactory.newInstance();
		dBuilder = dbFactory.newDocumentBuilder();
	}

	public String getSoapAddress() {
		return soapAddress;
	}
	
	/**
	 * Get the SOAP Address attribute from the WSDL.
	 * @param wsdl
	 * @throws SAXException
	 * @throws IOException
	 * @throws TransformerException
	 */
	public void getSOAPAddress(String wsdl) throws SAXException, IOException, TransformerException {
		InputStream is = new ByteArrayInputStream(wsdl.getBytes());
		Document doc = dBuilder.parse(is);

		doc.getDocumentElement().normalize();

		NodeList nList = doc.getDocumentElement().getChildNodes();
		for (int i = 0; i < nList.getLength(); i++) {
			   Node nNode = nList.item(i);
			   	
			   if (nNode.getNodeType() == Node.ELEMENT_NODE && Constants.ELEM_SERVICE.equals(getLocalPart(nNode))) {
				   Element eElement = (Element) nNode;
				   getPort(eElement);
			   }
		}
		is.close();
	}
	
	/**
	 * Parse the wsdl for any specified schemaLocations.
	 * @param wsdl
	 * @return
	 * @throws SAXException
	 * @throws IOException
	 * @throws TransformerException
	 */
	public String parse(String wsdl) throws SAXException, IOException, TransformerException {
		getSOAPAddress(wsdl);
		InputStream is = new ByteArrayInputStream(wsdl.getBytes());
		Document doc = dBuilder.parse(is);

		doc.getDocumentElement().normalize();

		NodeList nList = doc.getDocumentElement().getChildNodes();
		for (int i = 0; i < nList.getLength(); i++) {
		   Node nNode = nList.item(i);
		   	
		   if (nNode.getNodeType() == Node.ELEMENT_NODE && Constants.ELEM_TYPES.equals(getLocalPart(nNode))) {
			   Element eElement = (Element) nNode;
			   getSchema(eElement);
		   }
		}		
		
		// Transform to string
		javax.xml.transform.TransformerFactory tfactory = TransformerFactory.newInstance();
		javax.xml.transform.Transformer xform = tfactory.newTransformer();

		javax.xml.transform.Source src = new DOMSource(doc);
		java.io.StringWriter writer = new StringWriter();
		Result result = new javax.xml.transform.stream.StreamResult(writer);
		xform.transform(src, result);
		
		return writer.toString();		
	}
	
	
	/**
	 * Extract the schema from an element.
	 * @param element
	 */
	public void getSchema(Element element) {
		NodeList nodes = element.getChildNodes();

		for(int i=0; i<nodes.getLength(); i++){
			Node node = nodes.item(i);

			if(node instanceof Element) {
				Element child = (Element) node;
				if (SCHEMA_ELEM.equals(getLocalPart(node))) {
					getImport(child);
				}
			}
		}
	}

	/**
	 * Extract the port.
	 * @param element
	 */
	public void getPort(Element element) {
		NodeList nodes = element.getChildNodes();

		for(int i=0; i<nodes.getLength(); i++){
			Node node = nodes.item(i);

			if(node instanceof Element) {
				Element child = (Element) node;
				if (PORT_ELEM.equals(getLocalPart(node))) {
					getSOAPAddress(child);
				}
			}
		}
	}	

	/**
	 * Extract an import element.
	 * @param element
	 */
	public void getImport(Element element) { 
		NodeList nodes = element.getChildNodes();

		for(int i=0; i<nodes.getLength(); i++){
			Node node = nodes.item(i);

			if(node instanceof Element){
				Element child = (Element) node;
				
				if (IMPORT_ELEM.equals(getLocalPart(node))) {

					if (child.hasAttribute(SCHEMA_LOCATION_ATTR)) {
						String schemaLocation = child.getAttribute(SCHEMA_LOCATION_ATTR);
						if (schemaLocation != null
								&& (schemaLocation.startsWith("http://") || schemaLocation.startsWith("https://"))) {
							String rewrittenLocation = deployXSD(schemaLocation);
							child.setAttribute(SCHEMA_LOCATION_ATTR, rewrittenLocation);
						}
					}
				}
			}
		}		
	}

	/**
	 * Get the SOAP Adddress.
	 * @param element
	 */
	public void getSOAPAddress(Element element) { 
		NodeList nodes = element.getChildNodes();

		for(int i=0; i<nodes.getLength(); i++){
			Node node = nodes.item(i);

			if(node instanceof Element){
				Element child = (Element) node;
				
				if ("address".equals(getLocalPart(node))) {
		    
					if ( (child.hasAttribute(SOAP_ADDRESS_LOCATION)) && (child.getAttribute(SOAP_ADDRESS_LOCATION) != null)) {
						soapAddress = child.getAttribute(SOAP_ADDRESS_LOCATION);
					}
				}
			}
		}		
	}
	
	/**
	 * Call the deployer to deploy the XSD.
	 * @param location
	 * @return
	 */
	public String deployXSD(String location) {
		String xsdLocation = deployer.deploy(location, soapAddress);
		return xsdLocation;
	}
	
	/**
	 * Utility method for parsing local part of a QName.
	 * @param child
	 * @return
	 */
	public static String getLocalPart(Node child) {
		try {
			QName q = QNameUtils.newQName(child);
			return q.getLocalPart();
		} catch (IllegalArgumentException iae){
			String nodeName = child.getNodeName();
			if (nodeName.contains(":")) {
				return nodeName.substring(nodeName.indexOf(":") + 1);
			} else {
				return nodeName;
			}
		}
	}	
}
