/*
 * Copyright 2021 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kie.kogito.quarkus.drools;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.inject.Inject;

import org.junit.jupiter.api.Test;
import org.kie.kogito.incubation.application.AppRoot;
import org.kie.kogito.incubation.common.*;
import org.kie.kogito.incubation.rules.InstanceQueryId;
import org.kie.kogito.incubation.rules.RuleUnitIds;
import org.kie.kogito.incubation.rules.RuleUnitInstanceId;
import org.kie.kogito.incubation.rules.data.DataSourceId;
import org.kie.kogito.incubation.rules.services.DataSourceService;
import org.kie.kogito.incubation.rules.services.StatefulRuleUnitService;
import org.kie.kogito.incubation.rules.services.contexts.RuleUnitMetaDataContext;
import org.kie.kogito.rules.DataSource;

import io.quarkus.test.junit.QuarkusTest;

import static org.junit.jupiter.api.Assertions.*;

@QuarkusTest
public class StatefulRuleUnitServiceTest {
    @Inject
    AppRoot appRoot;
    @Inject
    StatefulRuleUnitService ruleUnitService;
    @Inject
    DataSourceService dataSourceService;

    @Test
    void testCreate() {
        AnotherService ruleUnitData =
                new AnotherService(
                        DataSource.createStore(),
                        DataSource.createStore());

        var id = appRoot.get(RuleUnitIds.class).get(AnotherService.class);
        var result = ruleUnitService.create(id, ExtendedReferenceContext.ofData(ruleUnitData));
        var meta = result.as(RuleUnitMetaDataContext.class);
        var instanceId = meta.id(RuleUnitInstanceId.class);
        assertEquals("/rule-units/org.kie.kogito.quarkus.drools.AnotherService",
                instanceId.ruleUnitId().asLocalUri().path());
    }

    @Test
    void testQueryDsService() {
        AnotherService ruleUnitData =
                new AnotherService(
                        DataSource.createStore(),
                        DataSource.createStore());

        var id = appRoot.get(RuleUnitIds.class).get(AnotherService.class);
        MetaDataContext created = ruleUnitService.create(id, ExtendedReferenceContext.ofData(ruleUnitData));
        var meta = created.as(RuleUnitMetaDataContext.class);
        var ruid = meta.id(RuleUnitInstanceId.class);
        InstanceQueryId queryId = ruid.queries().get("Strings");

        DataSourceId dataSourceId = ruid.dataSources().get("strings");

        dataSourceService.add(dataSourceId, new StringHolder("hello folks"));
        dataSourceService.add(dataSourceId, new StringHolder("hello people"));
        dataSourceService.add(dataSourceId, new StringHolder("hello Mario"));
        dataSourceService.add(dataSourceId, new StringHolder("helicopter"));

        Stream<ExtendedDataContext> result = ruleUnitService.query(queryId, ExtendedReferenceContext.ofData(EmptyDataContext.Instance));
        List<String> strings = result
                .map(e -> e.data().as(MapDataContext.class).get("results", StringHolder.class).getValue())
                .collect(Collectors.toList());

        assertFalse(strings.isEmpty());

        strings.removeAll(List.of("hello folks", "hello people", "hello Mario"));
        assertTrue(strings.isEmpty());

    }

    @Test
    void testQueryDirectDs() {
        AnotherService ruleUnitData =
                new AnotherService(
                        DataSource.createStore(),
                        DataSource.createStore());

        var id = appRoot.get(RuleUnitIds.class).get(AnotherService.class);
        MetaDataContext created = ruleUnitService.create(id, ExtendedReferenceContext.ofData(ruleUnitData));
        var meta = created.as(RuleUnitMetaDataContext.class);
        var ruid = meta.id(RuleUnitInstanceId.class);
        InstanceQueryId queryId = ruid.queries().get("Strings");

        ruleUnitData.getStrings().add(new StringHolder("hello folks"));
        ruleUnitData.getStrings().add(new StringHolder("hello people"));
        ruleUnitData.getStrings().add(new StringHolder("hello Mario"));
        ruleUnitData.getStrings().add(new StringHolder("helicopter"));

        Stream<ExtendedDataContext> result = ruleUnitService.query(queryId, ExtendedReferenceContext.ofData(EmptyDataContext.Instance));
        List<String> strings = result
                .map(e -> e.data().as(MapDataContext.class).get("results", StringHolder.class).getValue())
                .collect(Collectors.toList());

        assertFalse(strings.isEmpty());

        strings.removeAll(List.of("hello folks", "hello people", "hello Mario"));
        assertTrue(strings.isEmpty());

    }

}
