#include <glib.h>
#include <glib-object.h>


#ifndef BD_KBD_API
#define BD_KBD_API

#define BD_KBD_ERROR bd_kbd_error_quark ()
typedef enum {
    BD_KBD_ERROR_KMOD_INIT_FAIL,
    BD_KBD_ERROR_MODULE_FAIL,
    BD_KBD_ERROR_MODULE_NOEXIST,
    BD_KBD_ERROR_ZRAM_NOEXIST,
    BD_KBD_ERROR_ZRAM_INVAL,
    BD_KBD_ERROR_BCACHE_PARSE,
    BD_KBD_ERROR_BCACHE_SETUP_FAIL,
    BD_KBD_ERROR_BCACHE_DETACH_FAIL,
    BD_KBD_ERROR_BCACHE_NOT_ATTACHED,
    BD_KBD_ERROR_BCACHE_UUID,
    BD_KBD_ERROR_BCACHE_MODE_FAIL,
    BD_KBD_ERROR_BCACHE_MODE_INVAL,
} BDKBDError;

typedef enum {
    BD_KBD_MODE_WRITETHROUGH,
    BD_KBD_MODE_WRITEBACK,
    BD_KBD_MODE_WRITEAROUND,
    BD_KBD_MODE_NONE,
    BD_KBD_MODE_UNKNOWN,
} BDKBDBcacheMode;

#define BD_KBD_TYPE_ZRAM_STATS (bd_kbd_zram_stats_get_type ())
GType  bd_kbd_zram_stats_get_type ();


/* see zRAM kernel documentation for details */
typedef struct BDKBDZramStats {
    guint64 disksize;
    guint64 num_reads;
    guint64 num_writes;
    guint64 invalid_io;
    guint64 zero_pages;
    guint64 max_comp_streams;
    gchar* comp_algorithm;
    guint64 orig_data_size;
    guint64 compr_data_size;
    guint64 mem_used_total;
} BDKBDZramStats;

/**
 * bd_kbd_zram_stats_copy: (skip)
 *
 * Creates a new copy of @data.
 */
BDKBDZramStats* bd_kbd_zram_stats_copy (BDKBDZramStats *data);


/**
 * bd_kbd_zram_stats_free: (skip)
 *
 * Frees @data.
 */
void  bd_kbd_zram_stats_free (BDKBDZramStats *data);



#define BD_KBD_TYPE_BCACHE_STATS (bd_kbd_bcache_stats_get_type ())
GType  bd_kbd_bcache_stats_get_type ();


typedef struct BDKBDBcacheStats {
    gchar *state;
    guint64 block_size;
    guint64 cache_size;
    guint64 cache_used;
    guint64 hits;
    guint64 misses;
    guint64 bypass_hits;
    guint64 bypass_misses;
} BDKBDBcacheStats;

/**
 * bd_kbd_bcache_stats_copy: (skip)
 *
 * Creates a new copy of @data.
 */
BDKBDBcacheStats* bd_kbd_bcache_stats_copy (BDKBDBcacheStats *data);


/**
 * bd_kbd_bcache_stats_free: (skip)
 *
 * Frees @data.
 */
void  bd_kbd_bcache_stats_free (BDKBDBcacheStats *data);




/**
 * bd_kbd_zram_create_devices:
 * @num_devices: number of devices to create
 * @sizes: (array zero-terminated=1): requested sizes (in bytes) for created zRAM
 *                                    devices
 * @nstreams: (allow-none) (array zero-terminated=1): numbers of streams for created
 *                                                    zRAM devices
 * @error: (out): place to store error (if any)
 *
 * Returns: whether @num_devices zRAM devices were successfully created or not
 *
 * **Lengths of @size and @nstreams (if given) have to be >= @num_devices!**
 */
gboolean  bd_kbd_zram_create_devices (guint64 num_devices, guint64 *sizes, guint64 *nstreams, GError **error);


/**
 * bd_kbd_zram_destroy_devices:
 * @error: (out): place to store error (if any)
 *
 * Returns: whether zRAM devices were successfully destroyed or not
 *
 * The only way how to destroy zRAM device right now is to unload the 'zram'
 * module and thus destroy all of them. That's why this function doesn't allow
 * specification of which devices should be destroyed.
 */
gboolean  bd_kbd_zram_destroy_devices (GError **error);


/**
 * bd_kbd_zram_get_stats:
 * @device: zRAM device to get stats for
 * @error: (out): place to store error (if any)
 *
 * Returns: (transfer full): statistics for the zRAM device
 */
BDKBDZramStats* bd_kbd_zram_get_stats (gchar *device, GError **error);


/**
 * bd_kbd_bcache_create:
 * @backing_device: backing (slow) device of the cache
 * @cache_device: cache (fast) device of the cache
 * @bcache_device: (out) (allow-none): place to store the name of the new bcache device (if any)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the bcache device was successfully created or not
 */
gboolean  bd_kbd_bcache_create (gchar *backing_device, gchar *cache_device, gchar **bcache_device, GError **error);


/**
 * bd_kbd_bcache_attach:
 * @c_set_uuid: cache set UUID of the cache to attach
 * @bcache_device: bcache device to attach @c_set_uuid cache to
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the @c_set_uuid cache was successfully attached to @bcache_device or not
 */
gboolean  bd_kbd_bcache_attach (gchar *c_set_uuid, gchar *bcache_device, GError **error);


/**
 * bd_kbd_bcache_detach:
 * @bcache_device: bcache device to detach the cache from
 * @c_set_uuid: (out) (allow-none) (transfer full): cache set UUID of the detached cache
 * @error: (out): place to store error (if any)
 * Returns: whether the bcache device @bcache_device was successfully destroyed or not
 *
 * Note: Flushes the cache first.
 */
gboolean  bd_kbd_bcache_detach (gchar *bcache_device, gchar **c_set_uuid, GError **error);


/**
 * bd_kbd_bcache_destroy:
 * @bcache_device: bcache device to destroy
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the bcache device @bcache_device was successfully destroyed or not
 */
gboolean  bd_kbd_bcache_destroy (gchar *bcache_device, GError **error);


/**
 * bd_kbd_bcache_get_mode:
 * @bcache_device: device to get mode of
 * @error: (out): place to store error (if any)
 *
 * Returns: current mode of the @bcache_device
 */
BDKBDBcacheMode  bd_kbd_bcache_get_mode (gchar *bcache_device, GError **error);


/**
 * bd_kbd_bcache_get_mode_str:
 * @mode: mode to get string representation of
 * @error: (out): place to store error (if any)
 *
 * Returns: (transfer none): string representation of @mode or %NULL in case of error
 */
const gchar* bd_kbd_bcache_get_mode_str (BDKBDBcacheMode mode, GError **error);


/**
 * bd_kbd_bcache_get_mode_from_str:
 * @mode_str: string representation of mode
 * @error: (out): place to store error (if any)
 *
 * Returns: mode matching the @mode_str given or %BD_KBD_MODE_UNKNOWN in case of no match
 */
BDKBDBcacheMode  bd_kbd_bcache_get_mode_from_str (gchar *mode_str, GError **error);


/**
 * bd_kbd_bcache_set_mode:
 * @bcache_device: bcache device to set mode of
 * @mode: mode to set
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the mode was successfully set or not
 */
gboolean  bd_kbd_bcache_set_mode (gchar *bcache_device, BDKBDBcacheMode mode, GError **error);


/**
 * bd_kbd_bcache_status:
 * @bcache_device: bcache device to get status for
 * @error: (out): place to store error (if any)
 *
 * Returns: (transfer full): status of the @bcache_device or %NULL in case of
 *                           error (@error is set)
 */
BDKBDBcacheStats* bd_kbd_bcache_status (gchar *bcache_device, GError **error);


/**
 * bd_kbd_bcache_get_backing_device:
 * @bcache_device: Bcache device to get the backing device for
 * @error: (out): place to store error (if any)
 *
 * Returns: (transfer full): name of the backing device of the @bcache_device
 *                           or %NULL if failed to determine (@error is populated)
 */
gchar* bd_kbd_bcache_get_backing_device (gchar *bcache_device, GError **error);


/**
 * bd_kbd_bcache_get_cache_device:
 * @bcache_device: Bcache device to get the cache device for
 * @error: (out): place to store error (if any)
 *
 * Returns: (transfer full): name of the cache device of the @bcache_device
 *                           or %NULL if failed to determine (@error is populated)
 *
 * Note: returns the name of the first cache device of @bcache_device (in case
 *       there are more)
 */
gchar* bd_kbd_bcache_get_cache_device (gchar *bcache_device, GError **error);


#endif  /* BD_KBD_API */
