using System;
using System.Collections.Generic;
using System.Text;

namespace org.ovirt.engine.ui.uicompat
{
	public class Event : IEquatable<EventDefinition>, IEquatable<Event>
	{
		private readonly IList<IEventListener> listeners;
		private readonly IDictionary<IEventListener, object> contexts;
		public Type OwnerType { get; private set; }
		public string Name { get; private set; }

		/// <summary>
		/// Gets an object representing current event context.
		/// Specified when add listener.
		/// </summary>
		public object Context { get; private set; }


		public Event(string name, Type ownerType)
		{
			Name = name;
			OwnerType = ownerType;

			listeners = new List<IEventListener>();
			contexts = new Dictionary<IEventListener, object>();
		}

		public Event()
		{
		}

		public Event(EventDefinition definition)
			: this(definition.Name, definition.OwnerType)
		{
		}

		/// <summary>
		/// Add listener with no context specified.
		/// </summary>
		public void addListener(IEventListener listener)
		{
			listeners.Add(listener);
		}

		public void addListener(IEventListener listener, object context)
		{
			listeners.Add(listener);
			contexts.Add(listener, context);
		}

		public void removeListener(IEventListener listener)
		{
			listeners.Remove(listener);

			if (contexts.ContainsKey(listener))
			{
				contexts.Remove(listener);
			}
		}

		public void removeAll()
		{
			listeners.Clear();
		}

		public void raise(object sender, EventArgs e)
		{
			//Iterate on a new instance of listeners list,
			//to enable listener unsubscribe from event
			//as a result on event fairing.
			List<IEventListener> list = new List<IEventListener>();
			foreach (IEventListener listener in listeners)
			{
				list.Add(listener);
			}

			foreach (IEventListener listener in list)
			{
				//Update current context.
				Context = contexts.ContainsKey(listener)
					? contexts[listener]
					: null;

				listener.eventRaised(this, sender, e);
			}
		}

		public bool Equals(Event other)
		{
			return other != null && Name == other.Name && OwnerType == other.OwnerType;
		}

		public bool Equals(EventDefinition other)
		{
			return other != null && Name == other.Name && OwnerType == other.OwnerType;
		}
	}
}
