﻿using System;
using System.Text;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using System.Collections.Generic;
using System.Collections;
using VdcCommon;
using VdcCommon.Interfaces;


namespace org.ovirt.engine.ui.uicommon.dataprovider
{
	public static class AsyncDataProvider
	{
		public static void GetDomainListViaPublic(AsyncQuery aQuery, bool filterInternalDomain)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			                                	{
													return source != null
													? new List<string>((List<string>)(valueObjectEnumerableList)source)
													: new List<string>();
			                                	};//END_CONVERTER
			Frontend.RunPublicQuery(VdcQueryType.GetDomainList, new GetDomainListParameters() { FilterInternalDomain = filterInternalDomain }, aQuery);
		}

		public static void IsBackendAvailable(AsyncQuery aQuery)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null;
			};//END_CONVERTER
			Frontend.RunPublicQuery(VdcQueryType.GetDomainList, new GetDomainListParameters() { FilterInternalDomain = true }, aQuery);
		}

		public static void IsCustomPropertiesAvailable(AsyncQuery aQuery, string version)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null ? (bool)source : true;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetConfigurationValue,
				new GetConfigurationValueParameters(ConfigurationValues.SupportCustomProperties)
				{
					Version = version
				},
				aQuery);
		}

		public static void GetIsoDomainByDataCenterId(AsyncQuery aQuery, Guid dataCenterId)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source != null)
				{
					List<storage_domains> storageDomains = (List<storage_domains>)source;
					foreach (storage_domains domain in storageDomains)
					{
						if (domain.storage_domain_type == StorageDomainType.ISO)
						{
							return domain;
						}
					}
				}

				return null;
			};//END_CONVERTER

			StoragePoolQueryParametersBase getIsoParams = new StoragePoolQueryParametersBase(dataCenterId);
			Frontend.RunQuery(VdcQueryType.GetStorageDomainsByStoragePoolId, getIsoParams, aQuery);
		}

		public static void GetIrsImageList(AsyncQuery aQuery, Guid isoDomainId, bool forceRefresh)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source != null)
				{
					List<RepoFileMetaData> repoList = (List<RepoFileMetaData>)source;
					List<string> fileNameList = new List<string>();
					foreach (RepoFileMetaData RepoFileMetaData in repoList)
					{
						fileNameList.Add(RepoFileMetaData.RepoFileName);
					}

					fileNameList.Sort(new Linq.CaseInsensitiveComparer());
					return fileNameList;
				}
				return new List<string>();
			};//END_CONVERTER
			
			GetAllIsoImagesListParameters parameters = new GetAllIsoImagesListParameters();
			parameters.StorageDomainId = isoDomainId;
			parameters.ForceRefresh = forceRefresh;
			Frontend.RunQuery(VdcQueryType.GetAllIsoImagesList, parameters, aQuery);
		}

		public static void GetFloppyImageList(AsyncQuery aQuery, Guid isoDomainId, bool forceRefresh)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source != null)
				{
					List<RepoFileMetaData> repoList = (List<RepoFileMetaData>)source;
					List<string> fileNameList = new List<string>();
					foreach (RepoFileMetaData RepoFileMetaData in repoList)
					{
						fileNameList.Add(RepoFileMetaData.RepoFileName);
					}

					fileNameList.Sort(new Linq.CaseInsensitiveComparer());
					return fileNameList;
				}
				return new List<string>();
			};//END_CONVERTER

			GetAllIsoImagesListParameters parameters = new GetAllIsoImagesListParameters();
			parameters.StorageDomainId = isoDomainId;
			parameters.ForceRefresh = forceRefresh;
			Frontend.RunQuery(VdcQueryType.GetAllFloppyImagesList, parameters, aQuery);
		}

		public static void GetClusterById(AsyncQuery aQuery, Guid id)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetVdsGroupById, new GetVdsGroupByIdParameters(id), aQuery);
		}

		public static void GetPoolById(AsyncQuery aQuery, Guid poolId)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			                                	{
			                                		return source;
			                                	};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetVmPoolById, new GetVmPoolByIdParameters(poolId), aQuery);
		}

		public static void GetVmById(AsyncQuery aQuery, Guid vmId)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetVmByVmId, new GetVmByVmIdParameters(vmId), aQuery);
		}

		public static void GetAnyVm(AsyncQuery aQuery, string poolName)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				List<VM> vms = Linq.Cast<VM>((List<IVdcQueryable>)(valueObjectEnumerableList)source);
				return vms.Count > 0 ? vms[0] : null;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.Search,
				new SearchParameters("Vms: pool=" + poolName, SearchType.VM),
				aQuery);
		}

		public static void GetTimeZoneList(AsyncQuery aQuery)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source == null)
					return new Dictionary<string, string>();
				return source;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetTimeZones, new VdcQueryParametersBase(), aQuery);
		}

		public static void GetDataCenterList(AsyncQuery aQuery)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source == null)
					return new List<storage_pool>();
				return source;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.Search, new SearchParameters("DataCenter: sortby name", SearchType.StoragePool), aQuery);
		}

		public static void GetMinimalVmMemSize(AsyncQuery aQuery)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null ? (int)source : 1;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetConfigurationValue, new GetConfigurationValueParameters(ConfigurationValues.VMMinMemorySizeInMB), aQuery);
		}

		public static void GetMaximalVmMemSize64OS(AsyncQuery aQuery, string version)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null ? (int)source : 262144;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetConfigurationValue, 
				new GetConfigurationValueParameters(ConfigurationValues.VM64BitMaxMemorySizeInMB)
				{
					Version = version
				},
				aQuery);
		}

		public static void GetMaximalVmMemSize32OS(AsyncQuery aQuery)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null ? (int)source : 20480;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetConfigurationValue, new GetConfigurationValueParameters(ConfigurationValues.VM32BitMaxMemorySizeInMB), aQuery);
		}

		public static void GetMaxNumOfVmSockets(AsyncQuery aQuery, string version)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null ? (int)source : 1;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetConfigurationValue,
				new GetConfigurationValueParameters(ConfigurationValues.MaxNumOfVmSockets)
				{
					Version = version
				},
				aQuery);
		}

		public static void GetMaxNumOfVmCpus(AsyncQuery aQuery, string version)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null ? (int)source : 1;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetConfigurationValue,
				new GetConfigurationValueParameters(ConfigurationValues.MaxNumOfVmCpus)
				{
					Version = version
				},
				aQuery);
		}

		public static void GetMaxNumOfCPUsPerSocket(AsyncQuery aQuery, string version)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null ? (int)source : 1;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetConfigurationValue,
				new GetConfigurationValueParameters(ConfigurationValues.MaxNumOfCpuPerSocket)
				{
					Version = version
				},
				aQuery);
		}

		public static void GetClusterList(AsyncQuery aQuery, Guid dataCenterId)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source != null)
				{
					List<VDSGroup> list = (List<VDSGroup>)source;
					list.Sort(new Linq.VdsGroupByNameComparer());
					return list;
				}
				return new List<VDSGroup>();
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetVdsGroupsByStoragePoolId, new StoragePoolQueryParametersBase(dataCenterId), aQuery);
		}

		public static void GetTemplateDiskList(AsyncQuery aQuery, guid templateId)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source == null)
					return new List<DiskImage>();
				return source;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetVmTemplatesDisks, new GetVmTemplatesDisksParameters(templateId), aQuery);
		}

		public static void GetRoundedPriority(AsyncQuery aQuery, int priority)
		{
			aQuery.Data = new object[1] {priority};
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				int max = (int)source;
				int medium = max / 2;

				int[] levels = new[] { 1, medium, max };

				for (int i = 0; i < levels.Length; i++)
				{
					int lengthToLess = levels[i] - (int)_asyncQuery.Data[0];
					int lengthToMore = levels[i + 1] - (int)_asyncQuery.Data[0];

					if (lengthToMore < 0)
						continue;

					return Math.Abs(lengthToLess) < lengthToMore
						? levels[i]
						: levels[i + 1];
				}


				return 0;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetConfigurationValue, new GetConfigurationValueParameters(ConfigurationValues.VmPriorityMaxValue), aQuery);
		}

		public static void GetTemplateListByDataCenter(AsyncQuery aQuery, Guid dataCenterId)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				List<VmTemplate> list = new List<VmTemplate>();
				if (source != null)
				{
					VmTemplate blankTemplate = new VmTemplate();
					foreach (VmTemplate template in (List<VmTemplate>)source)
					{
						if (template.Id.Equals(Guid.Empty))
						{
							blankTemplate = template;
						}
						else if (template.status == VmTemplateStatus.OK)
						{
							list.Add(template);
						}
					}

					list.Sort(new Linq.VmTemplateByNameComparer());
					list.Insert(0, blankTemplate);
				}

				return list;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetVmTemplatesByStoragePoolId, new GetVmTemplatesByStoragePoolIdParameters(dataCenterId), aQuery);
		}

		public static void GetTemplateListByStorage(AsyncQuery aQuery, Guid storageId)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				List<VmTemplate> list = new List<VmTemplate>();
				if (source != null)
				{
					foreach (VmTemplate template in (List<VmTemplate>)source)
					{
						if (template.status == VmTemplateStatus.OK)
						{
							list.Add(template);
						}
					}

					list.Sort(new Linq.VmTemplateByNameComparer());
				}

				return list;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetVmTemplatesFromStorageDomain, new StorageDomainQueryParametersBase(storageId), aQuery);
		}

		public static void GetNumOfMonitorList(AsyncQuery aQuery)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				List<int> nums = new List<int>();
				if (source != null)
				{
					IEnumerable numEnumerable = (IEnumerable)source;
					IEnumerator numIterator = numEnumerable.GetEnumerator();
					while (numIterator.MoveNext())
					{
						nums.Add(Convert.ToInt32(numIterator.Current.ToString()));
					}
				}
				return nums;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetConfigurationValue, new GetConfigurationValueParameters(ConfigurationValues.ValidNumOfMonitors), aQuery);
		}

		public static void GetStorageDomainListByTemplate(AsyncQuery aQuery, Guid templateId)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if(source== null) 
					return new List<storage_domains>();
				return source;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetStorageDomainsByVmTemplateId, new GetStorageDomainsByVmTemplateIdQueryParameters(templateId), aQuery);
		}

		public static void GetStorageDomainList(AsyncQuery aQuery, Guid dataCenterId)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source == null)
					return new List<storage_domains>();
				return source;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetStorageDomainsByStoragePoolId, new StoragePoolQueryParametersBase(dataCenterId), aQuery);
		}

		public static void GetMaxVmPriority(AsyncQuery aQuery)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source == null)
					return 100;
				return (int)source;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetConfigurationValue, new GetConfigurationValueParameters(ConfigurationValues.VmPriorityMaxValue), aQuery);
		}

		public static void GetDefaultTimeZone(AsyncQuery aQuery)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source != null) 
					return ((KeyValuePair<string, string>) source).Key;
				return String.Empty;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetDefualtTimeZone, new VdcQueryParametersBase(), aQuery);
		}

		public static void GetHostListByCluster(AsyncQuery aQuery, string clusterName)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source != null)
				{
					List<VDS> list = Linq.Cast<VDS>((List<IVdcQueryable>)source);
					return list;
				}

				return new List<VDS>();
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.Search, new SearchParameters("Host: cluster = " + clusterName + " sortby name", SearchType.VDS), aQuery);
		}

		public static void GetVmDiskList(AsyncQuery aQuery, guid vmId)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source != null)
					return source;
				return new List<DiskImage>();
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetAllDisksByVmId, new GetAllDisksByVmIdParameters(vmId), aQuery);
		}

		public sealed class GetSnapshotListQueryResult
		{
			public Guid PreviewingImage { get; private set; }
			public List<DiskImage> Snapshots { get; private set; }
			public DiskImage Disk { get; private set; }
			public Guid VmId { get; set; }

			public GetSnapshotListQueryResult(Guid previewingImage, List<DiskImage> snapshots, DiskImage disk)
			{
				PreviewingImage = previewingImage;
				Snapshots = snapshots;
				Disk = disk;
			}
		}
		public static void GetSnapshotList(AsyncQuery aQuery, Guid vmId, DiskImage disk)
		{
			aQuery.Data = new object[1] {disk};
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				GetAllVmSnapshotsByDriveQueryReturnValue returnValue = (GetAllVmSnapshotsByDriveQueryReturnValue)_asyncQuery.OriginalReturnValue;
				return new GetSnapshotListQueryResult(returnValue.TryingImage, (List<DiskImage>)(valueObjectEnumerableList)source, (DiskImage)_asyncQuery.Data[0]);
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetAllVmSnapshotsByDrive, new GetAllVmSnapshotsByDriveParameters(vmId, disk.internal_drive_mapping), aQuery);
		}

		public static void GetMaxVmMemSize(AsyncQuery aQuery, bool is64)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source != null)
					return source;
				return 262144;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetConfigurationValue,
				new GetConfigurationValueParameters(is64 ? ConfigurationValues.VM64BitMaxMemorySizeInMB : ConfigurationValues.VM32BitMaxMemorySizeInMB),
				aQuery);
		}

		public static void GetDomainList(AsyncQuery aQuery, bool filterInternalDomain)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null
					? new List<string>((List<string>)(valueObjectEnumerableList)source)
					: new List<string>();
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetDomainList, new GetDomainListParameters() { FilterInternalDomain = filterInternalDomain }, aQuery);
		}

		public static void GetRoleList(AsyncQuery aQuery)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null
					? (List<roles>)(valueObjectEnumerableList)source
					: new List<roles>();
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetAllRoles,
				new MultilevelAdministrationsQueriesParameters(), aQuery);
		}

		public static void GetStorageDomainById(AsyncQuery aQuery, Guid storageDomainId)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null ? (storage_domains)(valueObjectSingle)source : null;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetStorageDomainById, new StorageDomainQueryParametersBase(storageDomainId), aQuery);
		}

		public static void GetDiskPresetList(AsyncQuery aQuery, VmType vmType, StorageType storageType)
		{
			aQuery.Data = new object[2] {vmType, storageType};
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source == null)
				{
					return null;
				}

				List<DiskImageBase> list = new List<DiskImageBase>();
				DiskImageBase presetData = null;
				DiskImageBase presetSystem = null;
				foreach (DiskImageBase disk in (List<DiskImageBase>)(valueObjectEnumerableList)source)
				{
					if (disk.disk_type == DiskType.System || disk.disk_type == DiskType.Data)
					{
						list.Add(disk);
					}
					if (disk.disk_type == DiskType.System && presetSystem == null)
					{
						presetSystem = disk;
					}
					else if (disk.disk_type == DiskType.Data && presetData == null)
					{
						presetData = disk;
					}
				}
				List<DiskImageBase> presetList = list;

				if (presetData != null)
				{
					presetData.volume_type = VolumeType.Preallocated;
					presetData.volume_format = DataProvider.GetDiskVolumeFormat(presetData.volume_type, (StorageType)_asyncQuery.Data[1]);
				}
				if (presetSystem != null)
				{
					presetSystem.volume_type = (VmType)_asyncQuery.Data[0] == VmType.Server ? VolumeType.Preallocated : VolumeType.Sparse;
					presetSystem.volume_format = DataProvider.GetDiskVolumeFormat(presetSystem.volume_type, (StorageType)_asyncQuery.Data[1]);
				}

				return presetList;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetDiskConfigurationList, new VdcQueryParametersBase(), aQuery);
		}

		public static void GetClusterNetworkList(AsyncQuery aQuery, Guid clusterId)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source == null)
					return new List<network>();
				return (List<network>)(valueObjectEnumerableList)source;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetAllNetworksByClusterId, new VdsGroupQueryParamenters(clusterId), aQuery);
		}

		public static void GetDataCenterById(AsyncQuery aQuery, Guid dataCenterId)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			                                	{
			                                		return source;
			                                	};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetStoragePoolById, new StoragePoolQueryParametersBase(dataCenterId), aQuery);
		}

		public static void GetTemplateById(AsyncQuery aQuery, Guid templateId)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetVmTemplate, new GetVmTemplateParameters(templateId), aQuery);
		}

		public static void GetHostList(AsyncQuery aQuery)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source != null)
				{
					List<VDS> list = Linq.Cast<VDS>((IEnumerable)source);
					return list;
				}

				return new List<VDS>();
			};//END_CONVERTER
			SearchParameters searchParameters = new SearchParameters("Host:", SearchType.VDS);
			searchParameters.MaxCount = 9999;
			Frontend.RunQuery(VdcQueryType.Search, searchParameters, aQuery);
		}

		public static void GetRpmVersionViaPublic(AsyncQuery aQuery)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null
					? (string)(valueObjectSingle)source
					: string.Empty;
			};//END_CONVERTER
			Frontend.RunPublicQuery(VdcQueryType.GetConfigurationValue,
				new GetConfigurationValueParameters(ConfigurationValues.ProductRPMVersion),
				aQuery);
		}

		public static void GetSearchResultsLimit(AsyncQuery aQuery)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null ? (int)(valueObject)source : 100;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetConfigurationValue,
				new GetConfigurationValueParameters(ConfigurationValues.SearchResultsLimit),
				aQuery);
		}

		public static void GetSANWipeAfterDelete(AsyncQuery aQuery)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null ? (bool)(valueObject)source : false;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetConfigurationValue,
				new GetConfigurationValueParameters(ConfigurationValues.SANWipeAfterDelete),
				aQuery);
		}

		public static void GetCustomPropertiesList(AsyncQuery aQuery)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null ? (string)(valueObject)source : null;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetVmCustomProperties, new VdcQueryParametersBase(), aQuery);
		}

		public static void GetPermissionsByAdElementId(AsyncQuery aQuery, Guid userId)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null
					? (List<permissions>)(valueObjectEnumerableList)source
					: new List<permissions>();
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetPermissionsByAdElementId, new MultilevelAdministrationByAdElementIdParameters(userId), aQuery);
		}

		public static void GetRoleActionGroupsByRoleId(AsyncQuery aQuery, Guid roleId)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
				{
					return source != null
						? (List<ActionGroup>)(valueObjectEnumerableList)source
						: new List<ActionGroup>();
				};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetRoleActionGroupsByRoleId, new MultilevelAdministrationByRoleIdParameters(roleId), aQuery);
		}

		public static void IsTemplateNameUnique(AsyncQuery aQuery, string name)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null ? !(bool)(valueObject)source : false;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.IsVmTemlateWithSameNameExist,
				new IsVmTemlateWithSameNameExistParameters(name),
				aQuery);
		}

		public static void IsVmNameUnique(AsyncQuery aQuery, string name)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null ? !(bool)(valueObject)source : false;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.IsVmWithSameNameExist,
				new IsVmWithSameNameExistParameters(name),
				aQuery);
		}

		public static void GetDataCentersWithPermittedActionOnClusters(AsyncQuery aQuery, ActionGroup actionGroup)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source == null)
					return new List<storage_pool>();
				return source;
			};//END_CONVERTER
			
			GetEntitiesWithPermittedActionParameters getEntitiesWithPermittedActionParameters = new GetEntitiesWithPermittedActionParameters();
			getEntitiesWithPermittedActionParameters.ActionGroup = actionGroup;
			Frontend.RunQuery(VdcQueryType.GetDataCentersWithPermittedActionOnClusters, getEntitiesWithPermittedActionParameters, aQuery);
 		}

		public static void GetClustersWithPermittedAction(AsyncQuery aQuery, ActionGroup actionGroup)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				if (source != null)
				{
					List<VDSGroup> list = (List<VDSGroup>)source;
					list.Sort(new Linq.VdsGroupByNameComparer());
					return list;
				}
				return new List<VDSGroup>();
			};//END_CONVERTER

			GetEntitiesWithPermittedActionParameters getEntitiesWithPermittedActionParameters = new GetEntitiesWithPermittedActionParameters();
			getEntitiesWithPermittedActionParameters.ActionGroup = actionGroup;
			Frontend.RunQuery(VdcQueryType.GetClustersWithPermittedAction, getEntitiesWithPermittedActionParameters, aQuery);
 		}

		public static void GetVmTemplatesWithPermittedAction(AsyncQuery aQuery, ActionGroup actionGroup)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				List<VmTemplate> list = new List<VmTemplate>();
				if (source != null)
				{
					VmTemplate blankTemplate = new VmTemplate();
					foreach (VmTemplate template in (List<VmTemplate>)source)
					{
						if (template.Id.Equals(Guid.Empty))
						{
							blankTemplate = template;
						}
						else if (template.status == VmTemplateStatus.OK)
						{
							list.Add(template);
						}
					}

					list.Sort(new Linq.VmTemplateByNameComparer());
					list.Insert(0, blankTemplate);
				}

				return list;
			};//END_CONVERTER

			GetEntitiesWithPermittedActionParameters getEntitiesWithPermittedActionParameters = new GetEntitiesWithPermittedActionParameters();
			getEntitiesWithPermittedActionParameters.ActionGroup = actionGroup;
			Frontend.RunQuery(VdcQueryType.GetVmTemplatesWithPermittedAction, getEntitiesWithPermittedActionParameters, aQuery);
		}

		public static void IsUSBEnabledByDefault(AsyncQuery aQuery)
		{
			/*START_CONVERTER*/aQuery.converterCallback = delegate(valueObject source, AsyncQuery _asyncQuery)
			{
				return source != null ? (bool)source : false;
			};//END_CONVERTER
			Frontend.RunQuery(VdcQueryType.GetConfigurationValue, new GetConfigurationValueParameters(ConfigurationValues.EnableUSBAsDefault), aQuery);
		}
	}
}
