﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Resources;
using System.Collections;
using System.Text.RegularExpressions;
using System.IO;

namespace VdcUtils
{
    public class ErrorTranslator
    {
		public ErrorTranslator()
		{
		}

        private Dictionary<string, string> mMessages = new Dictionary<string, string>();
        public ErrorTranslator(string errorFileName)
        {
			if (!File.Exists(errorFileName))
            {
				QLogger.Instance.Error("File: " + errorFileName + " not exists");
                return;
            }

			ResXResourceReader reader = new ResXResourceReader(errorFileName);
            foreach (DictionaryEntry entry in reader)
            {
                try
                {
                    if (!mMessages.ContainsKey((string)entry.Key))
                    {
                        mMessages.Add((string)entry.Key, (string)entry.Value);
                    }
                    else
                    {
                        QLogger.Instance.ErrorFormat("Code {0} appears more then once in string table.",
                                                    entry.Key);
                    }
                }
                catch
                {
                    QLogger.Instance.ErrorFormat("Cannot convert code {0} to error message", entry.Key);
                }
            }
        }

        /// <summary>
        /// Translate errors from error types. 
        /// error messages contains errors and variables.
        /// Variable used in messages.
        /// Variable definition must be in format:
        /// $variableName variableValue.
        /// Variable usage must be in format
        /// #variableName
        /// </summary>
        /// <param name="errorMsg">messages to be translated</param>
		/// <param name="changeIfNotFound">If true: if message key is not found in the resource, 
		/// return a beautified key. If false, returned unfound key as is.</param>
        /// <returns></returns>
        public List<string> TranslateErrorText(List<string> errorMsg, bool changeIfNotFound)
        {
			List<string> translatedMessages = TranslateMessages(errorMsg, changeIfNotFound);
			return ResolveMessages(translatedMessages);
        }

		/// <summary>
		/// Translates and resolves errors from error types. 
		/// error messages contains errors and variables.
		/// Variable used in messages.
		/// Variable definition must be in format:
		/// $variableName variableValue.
		/// Variable usage must be in format
		/// #variableName
		/// Note: Unfound message keys will be beautified!
		/// </summary>
		/// <param name="errorMsg">messages to be translated</param>
		/// <returns></returns>
		public virtual List<string> TranslateErrorText(List<string> errorMsg)
		{
			return TranslateErrorText(errorMsg, true);
		}

		public List<string> TranslateMessages(List<string> errorMsg, bool changeIfNotFound)
		{
			List<string> translatedMessages = new List<string>();
			if (errorMsg != null && errorMsg.Count > 0)
			{
				foreach (string curError in errorMsg)
				{
					translatedMessages.Add(TranslateErrorTextSingle(curError, changeIfNotFound));
				}
			}
			return translatedMessages;
		}

		/// <summary>
		/// returns true if the specified strMessage is in the format:
		/// "$variable-name variable-value", false otherwise.
		/// </summary>
		/// <param name="strMessage">the string that may be a dynamic variable.</param>
		/// <returns>true if input is dynamic variable, false otherwise.</returns>
		public bool IsDynamicVariable(string strMessage)
		{
			return strMessage.StartsWith("$");
		}

		/// <summary>
		/// Translates a single error message.
		/// </summary>
		/// <param name="errorMsg">the message to be translated</param>
		/// <param name="changeIfNotFound">If true: if message key is not found in the resource, 
		/// return a beautified key. If false, returned unfound key as is.</param>
		/// <returns></returns>
        public string TranslateErrorTextSingle(string errorMsg, bool changeIfNotFound)
        {
            string ret = string.Empty;
            
            if (!string.IsNullOrEmpty(errorMsg))
            {
                if (mMessages.ContainsKey(errorMsg))
                {
                    ret = mMessages[errorMsg];
                }

                else
                {
                    if ((IsDynamicVariable(errorMsg)) || (!changeIfNotFound))
                    {
                        ret = errorMsg;
                    }
                    else
                    // just a message that doesn't have a value in the resource:
                    {
                        string[] splitted = errorMsg.ToLower().Split('_');
                        ret = string.Join(" ", splitted);
                    }
                }
            }

            return ret;
        }

		/// <summary>
		/// Translates a single error message.
		/// Note: if message key not found, a beautified message will return!
		/// </summary>
		/// <param name="errorMsg">the message to translate</param>
		/// <returns>the translated message or a beautifed message key</returns>
		public string TranslateErrorTextSingle(string errorMsg)
		{
			return TranslateErrorTextSingle(errorMsg, true);
		}

		/// <summary>
		/// Replacing variables ('#...') within translatedMessages with
		/// their values ('$...') that are also within translatedMessages.
		/// </summary>
		/// <param name="translatedMessages"></param>
		/// <returns></returns>
		public List<string> ResolveMessages(List<string> translatedMessages)
		{
			List<string> translatedErrors = new List<string>();
			Dictionary<string, string> variables = new Dictionary<string, string>();

			foreach (string currentMessage in translatedMessages)
			{
				if (currentMessage.StartsWith("$"))
				{
					AddVariable(currentMessage, variables);
				}
				else
				{
					translatedErrors.Add(currentMessage);
				}
			}
			///Place to global variable adding
			List<string> returnValue = new List<string>();
			foreach (string error in translatedErrors)
			{
				returnValue.Add(resolveMessage(error, variables));
			}

			return returnValue;
		}

		private void AddVariable(string variable, Dictionary<string, string> variables)
		{
			int firstSpace = variable.IndexOf(' ');
			if (firstSpace != -1 && firstSpace < variable.Length)
			{
				string key = variable.Substring(1, firstSpace - 1);
				string value = variable.Substring(firstSpace + 1);
				if (!variables.ContainsKey(key))
				{
					variables.Add(key, value);
				}
			}
		}

        private string resolveMessage(string message, Dictionary<string, string> variables)
        {
            string returnValue = message;

            Regex regex = new Regex(@"#\w*");
            MatchCollection collection = regex.Matches(message);
            foreach (Match match in collection)
            {
                string value = match.Value.Substring(1);
				if (variables.ContainsKey(value))
				{
					returnValue = returnValue.Replace(match.Value, variables[value]);
				}
            }
            return returnValue;
        }
    }
}
