#!/usr/bin/python

# RHEV Manager post upgrade scripts
# Imported dynamically from rhevm-upgrade

#TODO: Implement all

import logging
import shutil
import os
import pwd
import sys
import common_utils as utils
import jasper30_utils as jsutils30
import basedefs
import basedefs30
import output_messages

# MSGS
MSG_PKI_CERTS_RESTORED = "PKI certificates were successfully restored from previous setup"
MSG_PKI_CERTS_OWNER_RESTORED = "The ownership of the pki files was restored"
MSG_KEYS_RENAMED = "Keys from previous setup were successfully renamed"

# CONST
WEB_CONF_JS = "/etc/rhevm-old/web-conf.js"

def updateVdcPkiOptions():

    options = (
        {
            "CABaseDirectory":["/etc/pki/ovirt-engine", 'text'],
            "keystoreUrl":["/etc/pki/ovirt-engine/.keystore", 'text'],
            "CertificateFileName":["/etc/pki/ovirt-engine/certs/engine.cer", 'text'],
            "CAEngineKey":["/etc/pki/ovirt-engine/private/ca.pem", 'text'],
            "TruststoreUrl":["/etc/pki/ovirt-engine/.keystore", 'text'],
            "ENGINEEARLib":["%s/engine.ear" %(basedefs.DIR_ENGINE), 'text'],
            "CACertificatePath":["/etc/pki/ovirt-engine/ca.pem", 'text'],
            "CertAlias":["engine", 'text'],
        },
        {
            "TruststorePass":[basedefs.CONST_CA_PASS, 'text'],
            "keystorePass":[basedefs.CONST_CA_PASS, 'text'],
            "SSLEnabled":[ "true", 'text'],
            "ScriptsPath":["/usr/share/ovirt-engine", 'text'],
            "SysPrepXPPath":["/etc/ovirt-engine/sysprep/sysprep.xp", 'text'],
            "SysPrep2K3Path":["/etc/ovirt-engine/sysprep/sysprep.2k3", 'text'],
            "SysPrep2K8Path":["/etc/ovirt-engine/sysprep/sysprep.2k8x86", 'text'],
            "SysPrep2K8x64Path":["/etc/ovirt-engine/sysprep/sysprep.2k8", 'text'],
            "SysPrep2K8R2Path":["/etc/ovirt-engine/sysprep/sysprep.2k8", 'text'],
            "SysPrepWindows7Path":["/etc/ovirt-engine/sysprep/sysprep.w7", 'text'],
            "SysPrepWindows7x64Path":["/etc/ovirt-engine/sysprep/sysprep.w7x64", 'text'],
            "ConfigDir":["/etc/ovirt-engine", 'text'],
            "DataDir":["/usr/share/ovirt-engine", 'text'],
        }
    )

    try:
        if os.path.exists(basedefs.FILE_ENGINE_CONFIG_BIN):
            if os.path.exists(basedefs.FILE_ENGINE_EXTENDED_CONF):
                #1st iterate on the CA related options
                for subDict in options:
                    for key in subDict:
                        value, keyType = subDict[key]
                        utils.updateVDCOption(key, value, [basedefs.CONST_CA_PASS], keyType)

                logging.debug("finished updating vdc options")
            else:
                raise Exception(output_messages.ERR_CANT_FIND_VDC_OPTION_FILE%(basedefs.FILE_ENGINE_EXTENDED_CONF))
        else:
            raise Exception(output_messages.ERR_CANT_FIND_RHEVM_CONFIG_FILE%(basedefs.FILE_ENGINE_CONFIG_BIN))

    except:
        raise Exception(output_messages.ERR_FAILED_UPD_VDC_OPTIONS%(sys.exc_info()[1]))

def updateKeyAlias():

    # Rename key alias from rhevm to engine

    cmd = [
        "keytool",
        "-changealias",
        "-storepass", "mypass",
        "-alias", "rhevm",
        "-destalias", "engine",
        "-keystore", os.path.join(basedefs.DIR_OVIRT_PKI, ".keystore"),
    ]

    utils.execCmd(cmdList=cmd)

def restoreKrb():
    TEMP_KRB = "/var/krb5.conf"
    if os.path.exists(TEMP_KRB):
        logging.debug("Temporary krb5.conf file detected. Restoring.")
        shutil.copy(TEMP_KRB, "/etc/ovirt-engine")

def restoreKeysCerts():
    # Restore pki certificates and correct their ownership.
    OLD_PKI_DIR = basedefs30.DIR_RHEVM_PKI + "-old/"
    pkiFolders = ["certs", "keys", "private", "requests"]
    pkiFiles = [".keystore", ".truststore", ".rnd", "serial.txt", "ca.pem"]
    for folder in pkiFolders:
        item = os.path.join(basedefs.DIR_OVIRT_PKI, folder)
        if os.path.exists(item):
            shutil.rmtree(item, item + ".new")
            logging.debug("Copy %s to %s", OLD_PKI_DIR + folder, item)
            shutil.copytree(OLD_PKI_DIR + folder, item)
            utils.chownToEngine(item)

    for item in pkiFiles:
        pkifile = os.path.join(basedefs.DIR_OVIRT_PKI, item)
        logging.debug("Copy %s to %s", OLD_PKI_DIR + item, pkifile)
        utils.copyFile(OLD_PKI_DIR + item, pkifile)
        utils.chownToEngine(pkifile)

    logging.debug(MSG_PKI_CERTS_RESTORED)

    # Update ownership of all files from jboss to ovirt
    # Find all files that owned by jboss in the pki folder and change the
    # owner of the file from jboss to engine/ovirt.
    for f in os.walk(basedefs.DIR_OVIRT_PKI):
        if len(f[2]) > 0:
            for nf in f[2]:
                pkifile = os.path.join(f[0], nf)
                oldpkifile = pkifile.replace("ovirt-engine", "rhevm-old")
                if os.path.exists(oldpkifile):
                    owner = pwd.getpwuid(os.stat(oldpkifile).st_uid).pw_name
                    if "jboss" in owner:
                        utils.chownToEngine(pkifile)

    logging.debug(MSG_PKI_CERTS_OWNER_RESTORED)

    # rename all keys to 'engine'
    keyFiles = [
        "/requests/rhevm.req",
        "/keys/rhevm_id_rsa",
        "/certs/rhevm.cer",
        "/certs/rhevm.der",
        ]
    for keyfile in keyFiles:
        keyfile = basedefs.DIR_OVIRT_PKI + keyfile
        newkey = keyfile.replace('rhevm', 'engine')
        logging.debug("Rename %s to %s", keyfile, newkey)
        os.rename(keyfile, newkey)

    logging.debug(MSG_KEYS_RENAMED)

def configureEngine():
    '''
    Configure ovirt-engine post upgrade
    '''
    # Get DB Pass
    dbPassword = utils.getDbPassword(basedefs.DB_USER)

    # Set DB Url
    dbUrl = "jdbc:postgresql://" + utils.getDbHostName() + ":" + utils.getDbPort() + "/" + basedefs.DB_NAME

    # Get current configuration
    fqdn, httpPort, httpsPort = utils.getHostParams(WEB_CONF_JS)

    # Get encrypted password
    masked_value_set = [dbPassword]
    encryptedPassword = utils.encryptEngineDBPass(password=dbPassword,
                                                  maskList=masked_value_set)

    # Edit sysconfig file
    utils.editEngineSysconfig(proxyEnabled=False,
                              dbUrl=dbUrl,
                              dbUser=utils.getDbUser(),
                              fqdn=fqdn,
                              http=httpPort,
                              https=httpsPort,
                              javaHome=utils.findJavaHome(),
                             )

    # Set encrypted password
    utils.configEncryptedPass(encryptedPassword)

    logging.debug("JBoss configuration was successfully restored")

def startEngine():
    logging.debug("using chkconfig to enable engine service to load on system startup.")
    srv = utils.Service(basedefs.ENGINE_SERVICE_NAME)
    srv.autoStart(True)

def run():
    # Configure ovirt-engine service
    configureEngine()

    # Restore pki certificates and correct their ownership.
    restoreKeysCerts()

    # Update pki locations in VDC options
    updateVdcPkiOptions()

    # Update key alias
    updateKeyAlias()

    # Restore krb5 conf
    restoreKrb()

    # Start ovirt-engine on reboot
    startEngine()

    # Raise an upgrade flag for reports
    jsutils30.setUpgraded()
