#!/bin/bash

set -o xtrace
set -o errexit
#set -o pipefail

#replica of tree comand using find and sed
function tree() {
  find ${1:-.} | sed -e "s/[^-][^\/]*\//  |/g" -e "s/|\([^ ]\)/|-\1/"
}

# Set magic variables for current file & dir
__dir="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

EXPECTED_ENV_VARS=(CHANNEL OPERATOR_CONTAINER_IMAGE METADATA_DESCRIPTION NAME DISPLAY_NAME
    APICAST_IMAGE DOC_URL CSV_VERSION VALID_SUBSCRIPTION)

source $__dir/apicast.properties

for var in "${EXPECTED_ENV_VARS[@]}"
do
    if [[ -z "${!var}" ]]
    then
      echo "$var not defined or empty"
      exit 1
    fi
    # extend scope to subshells
    # shellcheck disable=SC2163
    export "${var}"

done

# Upstream bundle dir
SOURCE_BUNDLE_DIR="${UPSTREAM_BUNDLE_DIR:-.}"
# Output bundle dir with autogenerated bundles
OUTPUT_DIR="${DOWNSTREAM_BUNDLE_DIR:-$__dir}"

# Clean up
# Useful to delete existing files not longer needed.
if [ -d "$OUTPUT_DIR" ]; then rm -Rf $OUTPUT_DIR/manifests; rm -Rf $OUTPUT_DIR/bundle; rm -Rf $OUTPUT_DIR/metadata; fi
mkdir -p $OUTPUT_DIR

tree
cp -vr "$SOURCE_BUNDLE_DIR/manifests" "$OUTPUT_DIR/"
mv "$OUTPUT_DIR/manifests" "$OUTPUT_DIR/bundle"
cp -vr "$SOURCE_BUNDLE_DIR/metadata" "$OUTPUT_DIR"

# Using ${YQ} 4.8 in development

# metadata/annotations.yaml
ANNOTATIONS_RESOURCE=$OUTPUT_DIR/metadata/annotations.yaml
${YQ} eval '.annotations."operators.operatorframework.io.bundle.channels.v1" = strenv(CHANNEL)' -i $ANNOTATIONS_RESOURCE
${YQ} eval '.annotations."operators.operatorframework.io.bundle.channel.default.v1" = strenv(CHANNEL)' -i $ANNOTATIONS_RESOURCE
${YQ} eval '.annotations."com.redhat.openshift.versions" = "v4.6"' -i $ANNOTATIONS_RESOURCE
${YQ} eval 'del(.annotations."operators.operatorframework.io.test.config.v1")' -i $ANNOTATIONS_RESOURCE
${YQ} eval 'del(.annotations."operators.operatorframework.io.test.mediatype.v1")' -i $ANNOTATIONS_RESOURCE

# bundle/apicast-operator-metrics-reader_rbac.authorization.k8s.io_v1_clusterrole.yaml
# NOOP

# bundle/apicast-operator.clusterserviceversion.yaml
CSV_RESOURCE=$OUTPUT_DIR/bundle/apicast-operator.clusterserviceversion.yaml
${YQ} eval '.metadata.annotations.containerImage = strenv(OPERATOR_CONTAINER_IMAGE)' -i $CSV_RESOURCE
${YQ} eval '.metadata.annotations.description = strenv(METADATA_DESCRIPTION)' -i $CSV_RESOURCE
${YQ} eval '.metadata.annotations.tectonic-visibility = "ocs"' -i $CSV_RESOURCE
${YQ} eval '.metadata.annotations."operators.openshift.io/valid-subscription" = strenv(VALID_SUBSCRIPTION)' -i $CSV_RESOURCE
${YQ} eval '.metadata.name = strenv(NAME)' -i $CSV_RESOURCE

if [ -n "${SKIP_RANGE}" ]
then
  export SKIP_RANGE
  ${YQ} eval '.metadata.annotations."olm.skipRange" = strenv(SKIP_RANGE)' -i $CSV_RESOURCE
else
  ${YQ} eval 'del(.metadata.annotations."olm.skipRange")' -i $CSV_RESOURCE
fi

if [ -n "${REPLACES}" ]; then
  export REPLACES
  ${YQ} eval '.spec.replaces = strenv(REPLACES)' -i $CSV_RESOURCE
else
  ${YQ} eval 'del(.spec.replaces)' -i $CSV_RESOURCE
fi

if [ -n "${SKIP_RANGE}" ]
then
  export SKIP_RANGE
  ${YQ} eval '.metadata.annotations."olm.skipRange" = strenv(SKIP_RANGE)' -i $CSV_RESOURCE
else
  ${YQ} eval 'del(.metadata.annotations."olm.skipRange")' -i $CSV_RESOURCE
fi

if [ -n "${SKIPS}" ]; then
  # ${SKIPS} holds multiple items separated with semicolon as delimiter
  arrSKIPS=$(echo ${SKIPS} | tr ";" "\n")
  for SKIP in ${arrSKIPS}
  do
    export SKIP
    ${YQ} eval '.spec.skips += [ strenv(SKIP) ]' -i $CSV_RESOURCE
  done
else
  ${YQ} eval 'del(.spec.skips)' -i $CSV_RESOURCE
fi

DESCRIPTION=$(cat "$__dir/DESCRIPTION") ${YQ} eval '.spec.description = strenv(DESCRIPTION)' -i $CSV_RESOURCE
${YQ} eval '.spec.displayName = strenv(DISPLAY_NAME)' -i $CSV_RESOURCE
ICON=$(cat "$__dir/ICON") ${YQ} eval '.spec.icon[0].base64data = strenv(ICON)' -i $CSV_RESOURCE
${YQ} eval '.spec.install.spec.deployments[0].spec.template.spec.containers[0].env[1].value = strenv(APICAST_IMAGE)' -i $CSV_RESOURCE
${YQ} eval '.spec.install.spec.deployments[0].spec.template.spec.containers[0].image = strenv(OPERATOR_CONTAINER_IMAGE)' -i $CSV_RESOURCE
${YQ} eval '.spec.links[0].name = "GitHub"' -i $CSV_RESOURCE
${YQ} eval '.spec.links[0].url = "https://github.com/3scale/apicast-operator"' -i $CSV_RESOURCE
${YQ} eval '.spec.links[1].name = "Documentation"' -i $CSV_RESOURCE
${YQ} eval '.spec.links[1].url = strenv(DOC_URL)' -i $CSV_RESOURCE
# Intentionally unset the relatedImages array to ensure no matter what is
# in upstream, we do not set the section as so digest pinning OSBS mechanism is run
${YQ} eval 'del(.spec.relatedImages)' -i $CSV_RESOURCE
${YQ} eval '.spec.version = strenv(CSV_VERSION)' -i $CSV_RESOURCE
# Pretty print the array to have consistent formatting everywhere
${YQ} eval --prettyPrint -i $CSV_RESOURCE

