#!/usr/bin/env ruby

# This script:
#   1) Converts the events of a Redis shard to the JSON format that we use.
#   2) Uploads the resulting JSON to S3.
#   3) Imports the events from S3 to Redshift.
#
# This script receives the events of the Redis shard by STDIN. The format
# expected is a JSON generated by rdb-tools using an rdb file.
#
# This script accepts two parameters: 'time_gen' and 'out_file_name'. The
# latter is just the name of the JSON file that will be created. 'time_gen' is
# very important. All the events that we generate have a 'time_gen' field. This
# field indicates when the event was generated (approximately). When we import
# events to Redshift, we only update it if the one we are importing has a more
# recent 'time_gen' than the one that already exists in Redshift.
# The 'time_gen' input parameter of this scripts define the 'time_gen' of all
# the events generated using this script.
# 'time_gen' needs to have this format: 'YYYYMMDD'.
#
# There are some restrictions regarding where this can be executed:
#   1) The host must have the .3scale_backend.conf file. This is required
#      because the AWS and Redshift credentials are there.
#   2) The host must be authorized in the security groups of Redshift.
#
# Before running the script, check the constants below and make sure that the
# S3 bucket, region, and root path are correct.

require 'json'
require 'time'
require 'pathname'
require 'aws-sdk'
require_relative '../../lib/3scale/backend'

FILTERED_PERIODS = %w(week eternity).freeze
S3_REGION = 'S3_REGION'.freeze
S3_BUCKET = 'S3_BUCKET'.freeze
S3_ROOT_PATH = "s3://#{S3_BUCKET}/historical".freeze

def event_parser
  ThreeScale::Backend::Stats::StatsParser
end

def redshift_adapter
  ThreeScale::Backend::Analytics::Redshift::Adapter
end

def invalid_key
  ThreeScale::Backend::Stats::StatsParser::StatsKeyValueInvalid
end

def config
  ThreeScale::Backend.configuration
end

def parse_line?(line)
  !line.start_with?('[{'.freeze)
end

def parsed_event(line, time_gen)
  event = parsed_line(line)
  parsed_event = event_parser.parse(event[:key], event[:value])
  parsed_event.merge(time_gen: "#{time_gen} 00:00:00")
end

def parsed_line(line)
  split_line = line.split('"')
  key = split_line[1]
  value = if split_line[2].chomp.end_with?('}]'.freeze)
            split_line[2].chomp[1..-3] # Remove starting ':' and ending '}]'
          else
            split_line[2].chomp[1..-2] # Remove starting ':' and ending ','
          end
  { key: key, value: value }
end

def filter_event?(event)
  # Right now we only want to export events that have a 'metric_field'.
  # Not all of them have that field, for example events that refer to
  # response codes have a 'response_code' field instead.
  FILTERED_PERIODS.include?(event[:period]) || event[:metric].nil?
end

def exit_invalid_time_gen
  STDOUT.puts 'Error: time_gen does not have a valid format'
  exit 1
end

def validate_time_gen(time_gen)
  exit_invalid_time_gen if time_gen.length != 8 # Expected format is YYYYMMDD

  begin
    DateTime.parse(time_gen)
  rescue
    exit_invalid_time_gen
  end
end

def validate_output_file(out_file_name)
  if Pathname.new("./#{out_file_name}").exist?
    STDOUT.puts 'Error: output file already exists'
    exit 1
  end
end

def validate_input_args(time_gen, out_file_name)
  validate_time_gen(time_gen)
  validate_output_file(out_file_name)
end

def aws_credentials
  Aws::Credentials.new(config.aws_access_key_id, config.aws_secret_access_key)
end

def aws_s3
  Aws::S3::Resource.new(region: S3_REGION, credentials: aws_credentials)
end

def s3_path(time_gen, out_file_name)
  "historical/#{time_gen}/#{out_file_name}"
end

def write_parsed_events_to_file(out_file_name, time_gen)
  File.open(out_file_name, 'w') do |out|
    STDIN.each_line do |line|
      if parse_line?(line)
        begin
          event = parsed_event(line, time_gen)
        rescue invalid_key
          next
        end
        out.puts event.to_json unless filter_event?(event)
      end
    end
  end
end

def upload_to_s3(out_file_name, time_gen)
  obj = aws_s3.bucket(S3_BUCKET).object(s3_path(time_gen, out_file_name))

  begin
    obj.upload_file(out_file_name)
  rescue Aws::Errors::MissingCredentialsError => e
    STDOUT.puts "Error while uploading to S3: #{e}"
    exit 1
  end
end

def import_into_redshift(path)
  redshift_adapter.insert_path(path)
end

if ARGV.include?('-h') || ARGV.include?('--help')
  STDOUT.puts "usage: #{$0} [time_gen] [out_file_name]\n"
  exit 0
end

time_gen, out_file_name = ARGV
validate_input_args(time_gen, out_file_name)
write_parsed_events_to_file(out_file_name, time_gen)
upload_to_s3(out_file_name, time_gen)
import_into_redshift(s3_path(time_gen, out_file_name))
