require 'test_helper'

class Abilities::ProviderAdminTest < ActiveSupport::TestCase

  def setup
    @account = FactoryGirl.build_stubbed(:provider_account)
    @admin   = FactoryGirl.build_stubbed(:admin, account: @account)
  end

  def test_forum
    Account.any_instance.expects(:provider_can_use?).returns(true).at_least_once
    assert_can ability, :manage, :forum

    Account.any_instance.expects(:provider_can_use?).returns(false).at_least_once
    assert_cannot ability, :manage, :forum
  end

  def test_web_hooks
    Settings::Switch.any_instance.stubs(:allowed?).returns(false)
    assert_can ability, :admin, :web_hooks
    assert_cannot ability, :manage, :web_hooks

    # ability :manage depends on :admin ability and the switch
    Settings::Switch.any_instance.stubs(:allowed?).returns(true)
    assert_can ability, :admin, :web_hooks
    assert_can ability, :manage, :web_hooks
  end

  def test_services
    service_1 = FactoryGirl.build_stubbed(:service, id: 1)
    service_2 = FactoryGirl.build_stubbed(:service, id: 2, account: @account)
    service_3 = FactoryGirl.build_stubbed(:service, id: 3, account: @account)

    assert_cannot ability, :show, service_1
    assert_can ability, :show, service_2
    assert_can ability, :show, service_3
  end

  def test_destroy_services
    service_1 = FactoryGirl.create(:simple_service)
    account = service_1.account
    Settings::Switch.any_instance.stubs(:allowed?).returns(true)

    @admin = FactoryGirl.create(:admin, account: account)
    service_2 = FactoryGirl.create(:simple_service, account: account)

    assert_can ability, :destroy, service_1
    assert_can ability, :destroy, service_2

    service_2.destroy!
    assert_cannot ability, :destroy, service_1
  end

  def test_csv_data_export_event
    admin_1 = @admin
    admin_2 = FactoryGirl.build_stubbed(:admin, account: @account)
    admin_1_ability = Ability.new(admin_1)
    admin_2_ability = Ability.new(admin_2)

    event_for_admin_1 = Reports::CsvDataExportEvent.create(@account, admin_1, 'users', 'week')
    event_for_admin_2 = Reports::CsvDataExportEvent.create(@account, admin_2, 'users', 'week')

    assert_can admin_1_ability, :show, event_for_admin_1
    assert_cannot admin_2_ability, :show, event_for_admin_1

    assert_cannot admin_1_ability, :show, event_for_admin_2
    assert_can admin_2_ability, :show, event_for_admin_2

    assert_can ability, :export, :data
  end

  def test_partner_can_manage_user_and_multiple_users?
    partner = FactoryGirl.build_stubbed(:partner)
    provider = FactoryGirl.build_stubbed(:simple_provider, partner: partner)
    @admin = FactoryGirl.build_stubbed(:admin, account: provider)

    partner.system_name = 'appdirect'
    assert_cannot ability, :manage, User
    assert_cannot ability, :manage, Invitation
    assert_cannot ability, :manage, :multiple_users

    partner.system_name = 'heroku'
    assert_cannot ability, :manage, User
    assert_cannot ability, :manage, Invitation
    assert_cannot ability, :manage, :multiple_users

    partner.system_name = 'redhat'
    assert_can ability, :manage, User
    assert_can ability, :manage, Invitation
    assert_cannot ability, :manage, :multiple_users
  end

  def test_manage_account
    assert_can ability, :manage, @account
  end

  def test_manage_user
    assert_can ability, :manage, @admin
  end

  def test_cinstances
    assert_can ability, :show, Cinstance
  end

  def test_billing
    invoice = FactoryGirl.build_stubbed(:invoice, provider_account_id: @account.id)
    Rails.application.config.three_scale.stubs(finance_enabled: false)

    assert_cannot ability, :manage, :credit_card
    assert_cannot ability, :manage, invoice

    Rails.application.config.three_scale.stubs(finance_enabled: true)
    finance = mock
    finance.stubs(allowed?: true)
    @account.settings.stubs(finance: finance)

    assert_can ability, :manage, :credit_card
    assert_can ability, :manage, invoice
  end

  def ability
    Ability.new(@admin)
  end
end
