/*
Copyright (C) 2016 Red Hat, Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package cmd

import (
	"encoding/json"
	"fmt"
	"io"
	"os"
	"strings"
	"text/template"

	"io/ioutil"
	"path"
	"runtime"

	"github.com/docker/machine/libmachine"
	"github.com/minishift/minishift/cmd/minishift/cmd/addon"
	addonbindata "github.com/minishift/minishift/out/bindata"
	"github.com/minishift/minishift/pkg/minikube/cluster"
	"github.com/minishift/minishift/pkg/minikube/constants"
	"github.com/minishift/minishift/pkg/minishift/addon/manager"
	"github.com/minishift/minishift/pkg/version"
	"github.com/minishift/minishift/setup-cdk/bindata"
	"github.com/spf13/cobra"
)

const (
	homeDirFlag       = "minishift-home"
	defaultDriverFlag = "default-vm-driver"
	forceInstallFlag  = "force"

	isoName        = "minishift-rhel7.iso"
	configFileName = "config.json"
	markerTmpl     = `openshift.auth.scheme=basic
openshift.auth.username=developer
openshift.auth.password=developer
cdk.version={{ . }}
`
	CDKMarker        = "cdk"
	CDKDefaultMemory = "4096"
)

var (
	homeDir        string
	defaultDriver  string
	forceInstall   bool
	defaultsAddons = []string{"anyuid", "admin-user", "xpaas"}
)

// cdkSetupCmd represents the command to setup CDK 3 on the host
var cdkSetupCmd = &cobra.Command{
	Use:   "setup-cdk",
	Short: "Configures CDK 3 on the host.",
	Long:  `Configures CDK 3 on the host.`,
	Run:   runCdkSetup,
	// Make sure we skip the default persistent pre-run, since this creates already directories in $MINISHIFT_HOME
	PersistentPreRun: func(cmd *cobra.Command, args []string) {
		// noop
	},
}

func init() {
	cdkSetupCmd.Flags().StringVar(&homeDir, homeDirFlag, constants.Minipath, "Sets the Minishift home directory.")
	cdkSetupCmd.Flags().StringVar(&defaultDriver, defaultDriverFlag, constants.DefaultVMDriver, "Sets the default VM driver.")
	cdkSetupCmd.Flags().BoolVar(&forceInstall, forceInstallFlag, false, "Forces the deletion of the exsiting Minishift install, if it exists.")
	RootCmd.AddCommand(cdkSetupCmd)
}

func runCdkSetup(cmd *cobra.Command, args []string) {
	constants.Minipath = homeDir
	fmt.Println(fmt.Sprintf("Setting up CDK 3 on host using '%s' as Minishift's home directory", constants.Minipath))

	ensureNoExistingConfigurationExists(os.Stdout, os.Stdin, homeDir)

	isoCachePath, ocCachePath, configDir := createDirectories(homeDir)
	unpackIso(isoCachePath)
	unpackOc(ocCachePath)
	createConfig(configDir, isoCachePath)

	createCDKMarker(homeDir)

	addOnManager := addon.GetAddOnManager()
	unpackAddons(addOnManager.BaseDir())
	enableDefaultAddon()

	fmt.Println("CDK 3 setup complete.")
}

func createDirectories(baseDir string) (string, string, string) {
	// need to rebuild the directories manually here, since OcCachePath is already initialised. Improving this requires some
	// refactoring upstream (HF)
	configDir := path.Join(baseDir, "config")
	err := os.MkdirAll(configDir, 0777)
	check(err)

	isoCacheDir := path.Join(baseDir, "cache", "iso")
	err = os.MkdirAll(isoCacheDir, 0777)
	check(err)
	isoCachePath := path.Join(isoCacheDir, isoName)

	ocCacheDir := path.Join(baseDir, "cache", "oc", version.GetOpenShiftVersion())
	err = os.MkdirAll(ocCacheDir, 0777)
	check(err)
	ocCachePath := path.Join(ocCacheDir, constants.OC_BINARY_NAME)

	addonsDir := path.Join(baseDir, "addons")
	err = os.MkdirAll(addonsDir, 0777)
	check(err)

	return isoCachePath, ocCachePath, configDir
}

func unpackOc(ocCachePath string) {
	data, err := bindata.Asset(runtime.GOOS + "/" + constants.OC_BINARY_NAME)
	check(err)

	fmt.Println(fmt.Sprintf("Copying %s to '%s'", constants.OC_BINARY_NAME, ocCachePath))
	err = ioutil.WriteFile(ocCachePath, []byte(data), 0774)
	check(err)
}

func unpackIso(iso string) {
	data, err := bindata.Asset("iso/" + isoName)
	check(err)

	fmt.Println(fmt.Sprintf("Copying %s to '%s'", isoName, iso))
	err = ioutil.WriteFile(iso, []byte(data), 0664)
	check(err)
}

func createConfig(configDir string, iso string) {
	config := make(map[string]string)
	// on Windows we might have to convert backward slashes to forward slashes
	iso = strings.Replace(iso, "\\", "/", -1)
	config["iso-url"] = "file://" + iso
	config["vm-driver"] = defaultDriver
	config["memory"] = CDKDefaultMemory

	configPath := path.Join(configDir, configFileName)
	fmt.Println(fmt.Sprintf("Creating configuration file '%s'", configPath))
	file, err := os.Create(configPath)
	check(err)

	toJson(file, config)
}

func toJson(w io.Writer, config map[string]string) {
	b, err := json.MarshalIndent(config, "", "    ")
	check(err)

	_, err = w.Write(b)
	check(err)
}

func check(e error) {
	if e != nil {
		fmt.Printf("Error setting up CDK 3 environment: %s\n", e)
		os.Exit(1)
	}
}

func ensureNoExistingConfigurationExists(w io.Writer, r io.Reader, baseDir string) {
	if _, err := os.Stat(baseDir); os.IsNotExist(err) {
		return
	}

	if !forceInstall {
		fmt.Fprintln(w, fmt.Sprintf("The MINISHIFT_HOME directory '%s' exists. Continuing will delete any existing VM and all other data in this directory. Do you want to continue? [y/N]", baseDir))

		var confirm string
		fmt.Fscanln(r, &confirm)
		if strings.ToLower(confirm) != "y" {
			fmt.Fprintln(w, "Aborting CDK setup")
			os.Exit(0)
		}
	}

	deleteExistingMachine()

	err := os.RemoveAll(baseDir)
	check(err)
}

func machineExists(api *libmachine.Client) bool {
	status, _ := cluster.GetHostStatus(api)

	if status == "Does Not Exist" {
		return false
	} else {
		return true
	}
}

func deleteExistingMachine() {
	api := libmachine.NewClient(constants.Minipath, constants.MakeMiniPath("certs"))
	defer api.Close()

	if machineExists(api) {
		if err := cluster.DeleteHost(api); err != nil {
			fmt.Println("Warning: ", err)
		}
		fmt.Println("Existing VM deleted")
	}
}

func createCDKMarker(minishiftHome string) {
	markerFileName := path.Join(minishiftHome, CDKMarker)
	markerFile, err := os.Create(markerFileName)
	defer markerFile.Close()
	check(err)

	fmt.Println(fmt.Sprintf("Creating marker file '%s'", markerFileName))
	tmpl := template.Must(template.New("cdkMarkerTmpl").Parse(markerTmpl))
	err = tmpl.Execute(markerFile, version.GetCDKVersion())
	check(err)
}

func unpackAddons(dir string) {
	for _, asset := range defaultsAddons {
		err := addonbindata.RestoreAssets(dir, asset)
		check(err)
	}
	fmt.Println("Default add-ons", strings.Join(defaultsAddons, ", "), "installed")
}

func enableDefaultAddon() {
	for _, addonName := range defaultsAddons {
		addOnManager := addon.GetAddOnManager()
		enableAddon(addOnManager, addonName, 0)
	}
	fmt.Println("Default add-ons", strings.Join(defaultsAddons, ", "), "enabled")
}

func enableAddon(addOnManager *manager.AddOnManager, addonName string, priority int) {
	addOnConfig, err := addOnManager.Enable(addonName, priority)
	check(err)

	addOnConfigMap := addon.GetAddOnConfiguration()
	addOnConfigMap[addOnConfig.Name] = addOnConfig
	addon.WriteAddOnConfig(addOnConfigMap)
}
