;;;; 	copyright (C) 1995 Free Software Foundation, Inc.
;;;; 
;;;; This program is free software; you can redistribute it and/or modify
;;;; it under the terms of the GNU General Public License as published by
;;;; the Free Software Foundation; either version 2, or (at your option)
;;;; any later version.
;;;; 
;;;; This program is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;;; GNU General Public License for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License
;;;; along with this software; see the file COPYING.  If not, write to
;;;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;;;; 


(set! lc-ascii #x00)
(set! lc-ltn1 #x81)
(set! lc-ltn2 #x82)
(set! lc-ltn3 #x83)
(set! lc-ltn4 #x84)
(set! lc-thai #x85)
(set! lc-grk #x86)
(set! lc-arb #x87)
(set! lc-hbw #x88)
(set! lc-kana #x89)
(set! lc-roman #x8A)
(set! lc-crl #x8C)
(set! lc-ltn5 #x8D)
(set! lc-jpold #x90)
(set! lc-cn #x91)
(set! lc-jp #x92)
(set! lc-kr #x93)
(set! lc-jp2 #x94)
(set! lc-cns1 #x95)
(set! lc-cns2 #x96)
(set! lc-cns14 #x97)
(set! lc-big5-1 #x98)
(set! lc-big5-2 #x99)
(set! lc-prv11 #x9A)
(set! lc-prv12 #x9B)
(set! lc-prv21 #x9C)
(set! lc-prv22 #x9D)
(set! lc-prv3 #x9E)
(set! lc-prv11-ext #xA0)
(set! lc-prv12-ext #xE0)
(set! lc-prv21-ext #xF0)
(set! lc-prv22-ext #xF5)
(set! lc-prv3-ext #xFF)
(set! lc-invalid #x9F)
(set! lc-composite #x80)

(define *predefined-character-set*
  (list
   ;; (cons lc '(bytes width type graphic final direction doc))
   ;; (cons lc-ascii '(0 1 0 0 ?B 0 "ASCII" "ISO8859-1")) ;; predefined in C
   (cons lc-ltn1 '(1 1 1 1 65 0 "Latin-1" "ISO8859-1"))
   (cons lc-ltn2 '(1 1 1 1 66 0 "Latin-2" "ISO8859-2"))
   (cons lc-ltn3 '(1 1 1 1 67 0 "Latin-3" "ISO8859-3"))
   (cons lc-ltn4 '(1 1 1 1 68 0 "Latin-4" "ISO8859-4"))
   (cons lc-thai '(1 1 1 1 84 0 "Thai" "TIS620"))
   (cons lc-grk '(1 1 1 1 70 0 "Greek" "ISO8859-7"))
   (cons lc-arb '(1 1 1 1 71 1 "Arabic" "ISO8859-6"))
   (cons lc-hbw '(1 1 1 1 72 1 "Hebrew" "ISO8859-8"))
   (cons lc-kana '(1 1 0 1 73 0 "Japanese Katakana" "JISX0201.1976"))
   (cons lc-roman '(1 1 0 0 74 0 "Japanese Roman" "JISX0201.1976"))
   (cons lc-crl '(1 1 1 1 76 0 "Cyrillic" "ISO8859-5"))
   (cons lc-ltn5 '(1 1 1 1 77 0 "Latin-5" "ISO8859-9"))
   (cons lc-jpold '(2 2 2 0 64 0 "Japanese Old" "JISX0208.1978"))
   (cons lc-cn '(2 2 2 0 65 0 "Chinese" "GB2312"))
   (cons lc-jp '(2 2 2 0 66 0 "Japanese" "JISX0208.\\(1983\\|1990\\)"))
   (cons lc-kr '(2 2 2 0 67 0 "Korean" "KSC5601"))
   (cons lc-jp2 '(2 2 2 0 68 0 "Japanese Supplement" "JISX0212"))
   (cons lc-cns1 '(2 2 2 0 71 0 "CNS Plane1" "CNS11643.1"))
   (cons lc-cns2 '(2 2 2 0 72 0 "CNS Plane2" "CNS11643.2"))
   (cons lc-big5-1 '(2 2 2 0 48 0 "Big5 Level 1" "Big5"))
   (cons lc-big5-2 '(2 2 2 0 49 0 "Big5 Level 2" "Big5"))))

(let ((c *predefined-character-set*)
      (lc #f)
      (data #f))
  (do () ((null? c))
    (set! lc (car (car c)))
    (set! data (cdr (car c)))
    (apply new-character-set (cons lc data))
    (set! c (cdr c))))

(define (make-coding-system name type mnemonic doc . eol-type-flags)
  "Register symbol NAME as a coding-system of:
 TYPE, MNEMONIC, DOC, EOL-TYPE, FLAGS.
 TYPE is information for encoding or decoding.  If it is one of below,
	nil: no conversion, t: automatic conversion,
	0:Internal, 1:Shift-JIS, 2:ISO2022, 3:Big5.
  the system provides appropriate code conversion facility.  If TYPE is 4, 
  appropriate code conversion programs (CCL) should be supplied in FLAGS.
 MNEMONIC: a character to be displayed on mode-line for this coding-system,
 DOC: a describing documents for the coding-system,
 EOL-TYPE (option): specify type of end-of-line,
   nil: no-conversion, 1: LF, 2: CRLF, 3: CR,
   t: generate coding-system for each end-of-line type
      by names NAMEunix, NAMEdos, and NAMEmac
 FLAGS (option): more precise information about the coding-system,
If TYPE is 2 (ISO2022), FLAGS should be a list of:
 LC-G0, LC-G1, LC-G2, LC-G3:
	Leading character of charset initially designated to G? graphic set,
	nil means G? is not designated initially,
	lc-invalid means G? can never be designated to,
	if (- leading-char) is specified, it is designated on output,
 SHORT: non-nil - allow such as \"ESC $ B\", nil - always \"ESC $ \( B\",
 ASCII-EOL: non-nil - designate ASCII to g0 at each end of line on output,
 ASCII-CNTL: non-nil - designate ASCII to g0 before TAB and SPACE on output,
 SEVEN: non-nil - use 7-bit environment on output,
 LOCK-SHIFT: non-nil - use locking-shift (SO/SI) instead of single-shift
	or designation by escape sequence,
 USE-ROMAN: non-nil - designate JIS0201-1976-Roman instead of ASCII,
 USE-OLDJIS: non-nil - designate JIS0208-1976 instead of JIS0208-1983,
 NO-ISO6429: non-nil - don't use ISO6429's direction specification,
If TYPE is 3 (Big5), FLAGS means nothing.
If TYPE is 4 (private), FLAGS should be a cons of CCL programs,
 for encoding and decoding.  See documentation of CCL for more detail."
  (intern-symbol #f name)
  (symbol-set! #f name name)
  (let ((code (make-vector 5 nil))
	(eol-type (if (null? eol-type-flags) #f (car eol-type-flags)))
	(flags (if (< 1 (length eol-type-flags)) (cadr eol-type-flags) #f)))
    (vector-set! code 0 type)
    (vector-set! code 1 (if (and (char>? mnemonic #\ )
				 (< (char->integer mnemonic) 127))
			    mnemonic #\ ))
    (vector-set! code 2 (if (string? doc) doc ""))
    (vector-set! code 3 nil)
    (cond ((eq? type 2)
	   (let ((i 0)
		 (vec (make-vector 32 nil)))
	     (do () ((not (and (< i 32) flags)))
	       (vector-set! vec i (car flags))
	       (set! flags (cdr flags))
	       (set! i (1+ i)))
	     (vector-set! code 4 vec)))
	  ((eq? type 4)
	   (if (and (pair? flags)
		    (vector? (car flags))
		    (vector? (cdr flags)))
	       (vector-set! code 4 flags)
	     (error "Invalid FLAGS argument for TYPE 4 (CCL)")))
	  (t (vector-set! code 4 flags)))
    (set-symbol-property! name 'coding-system code)
    (if (or (null? eol-type) (eq? eol-type 1)
	    (eq? eol-type 2) (eq? eol-type 3))
	(set-symbol-property! name 'eol-type eol-type)
      (if (eq? eol-type t)
	  (let* ((s (symbol->string name))
		 (codings (vector (string->symbol (string-append s "unix"))
				  (string->symbol (string-append s "dos"))
				  (string->symbol (string-append s "mac"))))
		 (i 0))
	    (set-symbol-property! name 'eol-type codings)
	    (do () ((= i 3))
	      (intern-symbol #f (vector-ref codings i))
	      (symbol-set! #f (vector-ref codings i) (vector-ref codings i))
	      (set-symbol-property! (vector-ref codings i) 'coding-system name)
	      (set-symbol-property! (vector-ref codings i) 'eol-type (+ i 1))
	      (set! i (1+ i))))
	(error "Invalid eol-type %s" eol-type)))
    ))

(define (copy-coding-system from to)
  "Make the same coding-system as FROM and name it TO.
If 'eol-type property of FROM is a vector, coding-systems
TOunix, TOdos, or TOmac are generated, and 'eol-type property
of TO becomes a vector of them."
  (intern-symbol #f to)
  (symbol-set! #f to to)
  (set-symbol-property! to 'coding-system from)
  (set-symbol-property! to 'post-read-conversion
			(symbol-property from 'post-read-conversion))
  (set-symbol-property! to 'pre-write-conversion
			(symbol-property from 'pre-write-conversion))
  (let ((eol-type (symbol-property from 'eol-type)))
    (if (number? eol-type)
	(set-symbol-property! to 'eol-type eol-type)
	(if (and (vector? eol-type) (= (length eol-type) 3))
	  (let* ((s (symbol->string to))
		 (codings (vector (string->symbol (string-append s "unix"))
				  (string->symbol (string-append s "dos"))
				  (string->symbol (string-append s "mac"))))
		 (i 0))
	    (set-symbol-property! to 'eol-type codings)
	    (do () ((= i 3))
	      (intern-symbol #f (vector-ref codings i))
	      (symbol-set! #f (vector-ref codings i) (vector-ref codings i))
	      (set-symbol-property! (vector-ref codings i) 'coding-system to)
	      (set-symbol-property! (vector-ref codings i) 'eol-type (+ i 1))
	      (set! i (1+ i))))
	  (error "Invalid eol-type %s in %s" eol-type from)))))

;; Definitions of predefined coding-systems

(make-coding-system
 '*noconv* nil
 #\= "No conversion.")

(make-coding-system
 '*autoconv* t
 #\+ "Automatic conversion." t)

(make-coding-system
 '*internal* 0
 #\= "Internal coding-system used in a buffer.")

(make-coding-system
 '*sjis* 1
 #\S "Coding-system of Shift-JIS used in Japan." t)

(make-coding-system
 '*iso-2022-jp* 2
 #\J "Coding-system used for communication with mail and news in Japan."
 t
 (list lc-ascii lc-invalid lc-invalid lc-invalid
       'short 'ascii-eol 'ascii-cntl 'seven))

(copy-coding-system '*iso-2022-jp* '*junet*)

(make-coding-system
 '*iso-2022-int-1* 2
 #\I "ISO-2022-INT-1"
 t
 (list lc-ascii lc-kr lc-invalid lc-invalid
       'short 'ascii-eol 'ascii-cntl 'seven 'locking-shift))

(make-coding-system
 '*oldjis* 2
 #\J "Coding-system used for old jis terminal."
 t
 (list lc-ascii lc-invalid lc-invalid lc-invalid
       'short 'ascii-eol 'ascii-cntl 'seven nil 'use-roman 'use-oldjis))

(make-coding-system
 '*ctext* 2
 #\X "Coding-system used in X as Compound Text Encoding."
 1
 (list lc-ascii lc-ltn1 lc-invalid lc-invalid
       nil 'ascii-eol 'ascii-cntl))

(copy-coding-system '*ctext* '*iso-8859-1*)

(make-coding-system
 '*euc-japan* 2
 #\E "Coding-system of Japanese EUC (Extended Unix Code)."
 t
 (list lc-ascii lc-jp lc-kana lc-jp2
       'short 'ascii-eol 'ascii-cntl))

(make-coding-system
 '*euc-korea* 2
 #\K "Coding-system of Korean EUC (Extended Unix Code)."
 1
 (list lc-ascii lc-kr lc-invalid lc-invalid
       nil 'ascii-eol 'ascii-cntl))
;; 93.12.16 by K.Handa
(copy-coding-system '*euc-korea* '*euc-kr*)

(make-coding-system
 '*iso-2022-kr* 2
 #\k "Coding-System used for communication with mail in Korea."
 1
 (list lc-ascii (- lc-kr) lc-invalid lc-invalid
       nil 'ascii-eol 'ascii-cntl 'seven 'lock-shift))
(copy-coding-system '*iso-2022-kr* '*korean-mail*)

(make-coding-system
 '*iso-2022-ss2-8* 2
 #\I "ISO-2022 coding system using SS2 for 96-charset in 8-bit code."
 t
 (list lc-ascii lc-invalid nil lc-invalid
       nil 'ascii-eol 'ascii-cntl))

(make-coding-system
 '*iso-2022-ss2-7* 2
 #\I "ISO-2022 coding system using SS2 for 96-charset in 7-bit code."
 t
 (list lc-ascii lc-invalid nil lc-invalid
       'short 'ascii-eol 'ascii-cntl 'seven))

(make-coding-system
 '*iso-2022-lock* 2
 #\i "ISO-2022 coding system using Locking-Shift for 96-charset."
 t
 (list lc-ascii nil lc-invalid lc-invalid
       nil 'ascii-eol 'ascii-cntl 'seven
       'lock-shift))			;93.12.1 by H.Minamino

(make-coding-system
 '*big5* 3
 #\B "Coding-system of BIG5."
 t nil)

(copy-coding-system '*big5* '*big5-eten*)

(set! input-coding-system *junet*)
(set! output-coding-system *junet*)
