/* bitops.h: MN10300 bit operations
 *
 * Copyright (C) 2005 Red Hat, Inc. All Rights Reserved.
 * Written by David Howells (dhowells@redhat.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 *
 * These have to be done with inline assembly: that way the bit-setting
 * is guaranteed to be atomic. All bit operations return 0 if the bit
 * was cleared before the operation and != 0 if it was not.
 *
 * bit 0 is the LSB of addr; bit 32 is the LSB of (addr+1).
 */
#ifndef __ASM_BITOPS_H
#define __ASM_BITOPS_H

#include <asm/cpu/cpu-regs.h>

/*
 * Some hacks to defeat gcc over-optimizations..
 */
struct __dummy { unsigned long a[100]; };
#define ADDR (*(volatile struct __dummy *) addr)
#define CONST_ADDR (*(volatile const struct __dummy *) addr)

#define smp_mb__before_clear_bit()	barrier()
#define smp_mb__after_clear_bit()	barrier()

/*****************************************************************************/
/*
 * set bit
 */
#define __set_bit(nr,addr)					\
({								\
        volatile unsigned char *_a = (unsigned char *)(addr);	\
	const unsigned shift = (nr) & 7;			\
	_a += (nr) >> 3;					\
								\
	asm volatile("bset %2,(%1) # set_bit reg"		\
		     : "=m"(*_a)				\
		     : "a"(_a), "d"(1 << shift),  "m"(*_a)	\
		     : "memory", "cc");				\
})

#define set_bit(nr,addr) __set_bit((nr),(addr))

/*****************************************************************************/
/*
 * clear bit
 */
#define ___clear_bit(nr,addr)					\
({								\
        volatile unsigned char *_a = (unsigned char *)(addr);	\
	const unsigned shift = (nr) & 7;			\
	_a += (nr) >> 3;					\
								\
	asm volatile("bclr %2,(%1) # clear_bit reg"		\
		     : "=m"(*_a)				\
		     : "a"(_a), "d"(1 << shift), "m"(*_a)	\
		     : "memory", "cc");				\
})

#define clear_bit(nr,addr) ___clear_bit((nr),(addr))


static inline void __clear_bit(int nr, volatile void * addr)
{
        unsigned int *a = (unsigned int *)addr;
        int mask;

        a += nr >> 5;
        mask = 1 << (nr & 0x1f);
        *a &= ~mask;
}

/*****************************************************************************/
/*
 * test bit
 */
static inline int test_bit(int nr, const volatile void * addr)
{
        return 1UL & (((const unsigned int *) addr)[nr >> 5] >> (nr & 31));
}

/*****************************************************************************/
/*
 * change bit
 */
static inline void __change_bit(int nr, volatile void * addr)
{
	int	mask;
	unsigned int *a = (unsigned int *) addr;

	a += nr >> 5;
	mask = 1 << (nr & 0x1f);
	*a ^= mask;
}

extern void change_bit(int nr, volatile void * addr);

/*****************************************************************************/
/*
 * test and set bit
 */
#define __test_and_set_bit(nr,addr)				\
({								\
        volatile unsigned char *_a = (unsigned char *)(addr);	\
	const unsigned shift = (nr) & 7;			\
	unsigned epsw;						\
	_a += (nr) >> 3;					\
								\
	asm volatile("bset %3,(%2) # test_set_bit reg\n"	\
		     "mov epsw,%1"				\
		     : "=m"(*_a), "=d"(epsw)			\
		     : "a"(_a), "d"(1 << shift), "m"(*_a)	\
		     : "memory", "cc");				\
								\
	!(epsw & EPSW_FLAG_Z);					\
})

#define test_and_set_bit(nr,addr) __test_and_set_bit((nr),(addr))

/*****************************************************************************/
/*
 * test and clear bit
 */
#define __test_and_clear_bit(nr,addr)				\
({								\
        volatile unsigned char *_a = (unsigned char *)(addr);	\
	const unsigned shift = (nr) & 7;			\
	unsigned epsw;						\
	_a += (nr) >> 3;					\
								\
	asm volatile("bclr %3,(%2) # test_clear_bit reg\n"	\
		     "mov epsw,%1"				\
		     : "=m"(*_a), "=d"(epsw)			\
		     : "a"(_a), "d"(1 << shift), "m"(*_a)	\
		     : "memory", "cc");				\
								\
	!(epsw & EPSW_FLAG_Z);					\
})

#define test_and_clear_bit(nr,addr) __test_and_clear_bit((nr),(addr))

/*****************************************************************************/
/*
 * test and change bit
 */
static inline int __test_and_change_bit(int nr, volatile void * addr)
{
	int	mask, retval;
	unsigned int *a = (unsigned int *)addr;

	a += nr >> 5;
	mask = 1 << (nr & 0x1f);
	retval = (mask & *a) != 0;
	*a ^= mask;

	return retval;
}

extern int test_and_change_bit(int nr, volatile void * addr);


#ifdef __KERNEL__

/*****************************************************************************/
/**
 * __ffs - find first bit in word.
 * @word: The word to search
 *
 * Undefined if no bit exists, so code should check against 0 first.
 */
static inline unsigned long __ffs(unsigned long word)
{
	unsigned result = -1; /* result (0-31 or 32 if no bit set)" */

	asm("or	%2,psw		\n"	/* preset the C flag */
	    "setlb		\n"
	    "ror	%0	\n"	/* rotate C->Dn->C */
	    "inc	%1	\n"	/* count in An (no flag clobber) */
	    "nop		\n"
	    "lcc		\n"	/* exit loop when C set */
	    : "=d"(word), "=a"(result)
	    : "i"(EPSW_FLAG_C), "0"(word), "1"(result)
	    : "cc"
	    );
	return result;
}

/*****************************************************************************/
/*
 * ffz = Find First Zero in word. Undefined if no zero exists,
 * so code should check against ~0UL first..
 */
static inline unsigned long ffz(unsigned long word)
{
        return __ffs(~word);
}

#endif /* __KERNEL__ */


/*****************************************************************************/
/*
 *
 */
static inline int find_next_zero_bit(const void *addr, int size, int offset)
{
	unsigned long *p = ((unsigned long *) addr) + (offset >> 5);
	unsigned long result = offset & ~31UL;
	unsigned long tmp;

	if (offset >= size)
		return size;
	size -= result;
	offset &= 31UL;
	if (offset) {
		tmp = *(p++);
		tmp |= ~0UL >> (32-offset);
		if (size < 32)
			goto found_first;
		if (~tmp)
			goto found_middle;
		size -= 32;
		result += 32;
	}
	while (size & ~31UL) {
		if (~(tmp = *(p++)))
			goto found_middle;
		result += 32;
		size -= 32;
	}
	if (!size)
		return result;
	tmp = *p;

found_first:
	tmp |= ~0UL << size;
found_middle:
	return result + ffz(tmp);
}

#define find_first_zero_bit(addr, size) \
        find_next_zero_bit((addr), (size), 0)

/*****************************************************************************/
/*
 *
 */
static inline int find_next_bit(const void *addr, int size, int offset)
{
	unsigned long *p = ((unsigned long *) addr) + (offset >> 5);
	unsigned long result = offset & ~31UL;
	unsigned long tmp;

	if (offset >= size)
		return size;
	size -= result;
	offset &= 31UL;
	if (offset) {
		tmp = *(p++);
		tmp &= ~(~0UL >> (32 - offset));
		if (size < 32)
			goto found_first;
		if (tmp)
			goto found_middle;
		size -= 32;
		result += 32;
	}
	while (size & ~31UL) {
		if ((tmp = *(p++)))
			goto found_middle;
		result += 32;
		size -= 32;
	}
	if (!size)
		return result;
	tmp = *p;

found_first:
	tmp &= ~(~0UL << size);
found_middle:
	return result + __ffs(tmp);
}

#define find_first_bit(addr, size) \
        find_next_bit((addr), (size), 0)

#ifdef __KERNEL__

/*
 * ffs: find first bit set. This is defined the same way as
 * the libc and compiler builtin ffs routines, therefore
 * differs in spirit from the above ffz (man ffs).
 */

#define ffs(x) generic_ffs(x)

/*
 * fls: find last bit set.
 */

#define fls(x) generic_fls(x)

/*
 * hweightN: returns the hamming weight (i.e. the number
 * of bits set) of a N-bit word
 */

#define hweight32(x) generic_hweight32(x)
#define hweight16(x) generic_hweight16(x)
#define hweight8(x) generic_hweight8(x)

/*
 * Every architecture must define this function. It's the fastest
 * way of searching a 140-bit bitmap where the first 100 bits are
 * unlikely to be set. It's guaranteed that at least one of the 140
 * bits is cleared.
 */
static inline int sched_find_first_bit(unsigned long *b)
{
	if (unlikely(b[0]))
		return __ffs(b[0]);
	if (unlikely(b[1]))
		return __ffs(b[1]) + 32;
	if (unlikely(b[2]))
		return __ffs(b[2]) + 64;
	if (b[3])
		return __ffs(b[3]) + 96;
	return __ffs(b[4]) + 128;
}

#define ext2_set_bit(nr, addr) \
	__test_and_set_bit((nr), (unsigned long *) addr)
#define ext2_set_bit_atomic(lock, nr, addr) \
	test_and_set_bit((nr), (unsigned long *) addr)
#define ext2_clear_bit(nr, addr) \
	__test_and_clear_bit((nr), (unsigned long *) addr)
#define ext2_clear_bit_atomic(lock, nr,  addr) \
	test_and_clear_bit((nr), (unsigned long *) addr)
#define ext2_test_bit(nr, addr) \
	test_bit((nr), (const unsigned long *) addr)
#define ext2_find_first_zero_bit(addr, size) \
	find_first_zero_bit((const unsigned long *) addr, size)
#define ext2_find_next_zero_bit(addr, size, off) \
	find_next_zero_bit((const unsigned long *) addr, size, off)

/* Bitmap functions for the minix filesystem */
#define minix_test_and_set_bit(nr, addr) \
	__test_and_set_bit(nr, (void *) addr)
#define minix_set_bit(nr, addr) \
	__set_bit(nr, (void *) addr)
#define minix_test_and_clear_bit(nr, addr) \
	__test_and_clear_bit(nr, (void *) addr)
#define minix_test_bit(nr, addr) \
	test_bit(nr, (const void *) addr)
#define minix_find_first_zero_bit(addr, size) \
	find_first_zero_bit((const void *) addr, size)

#endif /* __KERNEL__ */
#endif /* __ASM_BITOPS_H */
