/*
   (c) Copyright 2002-2003  Denis Oliver Kropp <dok@directfb.org>
   All rights reserved.

   XDirectFB is mainly based on XDarwin and
   also contains some KDrive, XFree and XWin code.
*/
/*
 * Generic rootless to Aqua specific glue code
 *
 * This code acts as a glue between the generic rootless X server code
 * and the Aqua specific implementation, which includes definitions that
 * conflict with stardard X types.
 *
 * Greg Parker     gparker@cs.stanford.edu
 */
/* $XFree86: xc/programs/Xserver/hw/directfb/bundle/rootlessAquaGlue.c,v 1.7 2001/12/22 05:28:35 torrey Exp $ */

#include "directfbScreen.h"
#include "directfbClipboard.h"
#include "directfbPasteboard.h"
#include "directfbX.h"
#include "rootlessCommon.h"
#include "rootlessDirectFB.h"
#include "rootless.h"

#include "regionstr.h"
#include "scrnintstr.h"
#include "servermd.h"
#include "globals.h" // dixScreenOrigins[]


/*
 * Private data of the rootless glue.
 */
struct _DirectFBWindowRec {
     DirectFBWindowRec *next;      /* doubly linked list */
     DirectFBWindowRec *prev;

     DFBWindowID        window_id; /* id to lookup from incoming events */

     Bool               focused;   /* if the window has the focus */

     struct {
          __u8             current;   /* current opacity */
          __u8             focused;   /* opacity while focused */
          __u8             unfocused; /* opacity while unfocused */
     } opacity;

     IDirectFBWindow   *window;    /* interface to the window */
     IDirectFBSurface  *surface;   /* interface to its surface */

     IDirectFBSurface  *layer_surface;   /* temporary interface */
     
     RootlessFramePtr   frame;     /* backpointer to frame */
};

/*
 * Extract the private data pointer from a frame.
 */
#define DFBWINREC(rw) ((DirectFBWindowRec *)((rw)->devPrivate))

/*
 * List of all windows created by XDirectFB.
 */
static DirectFBWindowRec *windows = NULL;

/*
 * Set the colorkeying option.
 */
static void
EnableColorkeying (IDirectFBWindow *window)
{
     DFBWindowOptions options;

     /* Retrieve current options. */
     window->GetOptions (window, &options);

     /* Add colorkeying bit. */
     window->SetOptions (window, options | DWOP_COLORKEYING);
}

/*
 * Clear the colorkeying option.
 */
static void
DisableColorkeying (IDirectFBWindow *window)
{
     DFBWindowOptions options;

     /* Retrieve current options. */
     window->GetOptions (window, &options);

     /* Remove colorkeying bit. */
     window->SetOptions (window, options & ~DWOP_COLORKEYING);
}

/*
 * Add a window to the list of XDirectFB's windows.
 */
static void
XDirectFBAddWindow (DirectFBWindowRec *win)
{
     win->next = windows;
     win->prev = NULL;

     if (windows)
          windows->prev = win;

     windows = win;
}

/*
 * Remove a window from the list of XDirectFB's windows.
 */
static void
XDirectFBRemoveWindow (DirectFBWindowRec *win)
{
     if (win->prev)
          win->prev->next = win->next;
     else
          windows = win->next;

     if (win->next)
          win->next->prev = win->prev;
}

/*
 * Look up a window by its id.
 */
static DirectFBWindowRec *
XDirectFBLookupWindow (DFBWindowID window_id)
{
     DirectFBWindowRec *win = windows;

     /* Loop through the list... */
     while (win) {
          /* ...until the id matches. */
          if (win->window_id == window_id)
               break;

          win = win->next;
     }

     /* Return the window. */
     return win;
}

/*
 * Fade the opacity from its current value to the specified.
 */
static void
XDirectFBWindowFade (DirectFBWindowRec *win, __u8 opacity, long ms_duration)
{
     long             t1, t2;
     int              diff;
     IDirectFBWindow *window = win->window;

     /* Retrieve its current opacity. */
     window->GetOpacity (window, &win->opacity.current);

     /* Calculate the opacity difference. */
     diff = opacity - win->opacity.current;

     /* Initialize both time stamps. */
     t1 = t2 = GetTimeInMillis();

     do {
          /* Calculate an intermediate opacity. */
          __u8 op = win->opacity.current + ((t2 - t1 + 1) * diff / ms_duration);

          /* Set the intermediate opacity. */
          window->SetOpacity (window, op);

          /* Update the time stamp. */
          t2 = GetTimeInMillis();
     }
     while (t2 - t1 < ms_duration);
}

/*
 * Show the window with the specified opacity, possibly fade in.
 */
static void
XDirectFBWindowFadeIn (DirectFBWindowRec *win, __u8 opacity)
{
     IDirectFBWindow *window = win->window;

     /* Fade in? */
     if (directfbPrefs.enableFadeIn)
          XDirectFBWindowFade (win, opacity, 40);

     /* Set the exact opacity. */
     window->SetOpacity (window, opacity);
     win->opacity.current = opacity;
}

/*
 * Hide the window, possibly fade out.
 */
static void
XDirectFBWindowFadeOut (DirectFBWindowRec *win)
{
     IDirectFBWindow *window = win->window;

     /* Fade out? */
     if (directfbPrefs.enableFadeOut)
          XDirectFBWindowFade (win, 0, 40);

     /* Set the exact opacity (zero). */
     window->SetOpacity (window, 0);
     win->opacity.current = 0;
}

/*
 * Create a new window with the specified position and size.
 * Retrieve its surface and window id.
 * Add the window to the list of XDirectFB's
 * windows and set default the options.
 * Attach the event buffer to receive its events.
 */
static DirectFBWindowRec *
XDirectFBNewWindow (IDirectFBDisplayLayer *layer,
                    int x, int y, int width, int height)
{
     DFBResult             ret;
     DFBWindowDescription  dsc;
     DirectFBWindowRec    *win;
     IDirectFBWindow      *window;

     /* Allocate the private data. */
     win = calloc (1, sizeof(DirectFBWindowRec));
     if (!win)
          return NULL;

     /* Fill the window description. */
     dsc.flags       = DWDESC_POSX | DWDESC_POSY |
                       DWDESC_WIDTH | DWDESC_HEIGHT;
     dsc.posx        = x;
     dsc.posy        = y;
     dsc.width       = width;
     dsc.height      = height;

     /* Create a new window. */
     ret = layer->CreateWindow (layer, &dsc, &window);
     if (ret) {
          DirectFBError ("XDirectFBNewWindow: Layer::CreateWindow", ret);
          free (win);
          return NULL;
     }

     /* Get the window's surface. */
     ret = window->GetSurface (window, &win->surface);
     if (ret) {
          DirectFBError ("XDirectFBNewWindow: Window::GetSurface", ret);
          window->Release (window);
          free (win);
          return NULL;
     }

     /* Clear the window's surface. */
     win->surface->Clear (win->surface, 0, 0, 0, 0);

     /* Store the window interface. */
     win->window = window;

     /* Initialize the opacity values. */
     win->opacity.focused   = directfbPrefs.defaultOpacity;
     win->opacity.unfocused = directfbPrefs.unfocusedOpacity;

     /* Store the window id. */
     window->GetID (window, &win->window_id);

     /* Add the window to our list. */
     XDirectFBAddWindow (win);

     /* Disable DirectFB's builtin wm. */
     window->SetOptions (window, DWOP_KEEP_POSITION | DWOP_KEEP_SIZE |
                         DWOP_KEEP_STACKING | DWOP_INDESTRUCTIBLE);

     /* Attach the window to our event buffer. */
     window->AttachEventBuffer (window, dfbEvents);

     /* Return the pointer to our private data. */
     return win;
}

/*
 * Ensure the window is visible.
 */
static void
XDirectFBShowWindow (DirectFBWindowRec *win)
{
     /* Show the window if it's not visible yet. */
     if (!win->opacity.current)
          XDirectFBWindowFadeIn (win,
                                 (directfbPrefs.enableUnfocused && !win->focused) ?
                                 win->opacity.unfocused : win->opacity.focused);
}

/*
 * Lock a surface for direct access to its pixel data.
 */
static Bool
XDirectFBLockSurface (RootlessFramePtr     pFrame,
                      IDirectFBSurface    *surface,
                      DFBSurfaceLockFlags  flags)
{
     DFBResult              ret;
     int                    pitch;
     void                  *data;
     DFBSurfacePixelFormat  format;

     /* Lock the surface. */
     ret = surface->Lock (surface, flags, &data, &pitch);
     if (ret) {
          DirectFBError ("XDirectFBLockSurface: IDirectFBSurface::Lock", ret);
          return FALSE;
     }

     /* Retrieve its pixel format. */
     surface->GetPixelFormat (surface, &format);

     /* Set data pointer, pitch etc. */
     pFrame->pixelData    = data;
     pFrame->bytesPerRow  = pitch;
     pFrame->depth        = DFB_BITS_PER_PIXEL(format);
     pFrame->bitsPerPixel = DFB_BYTES_PER_PIXEL(format) * 8;

     /* Update the data pointer in pixmap. */
     UpdatePixmap (pFrame->win);

     return TRUE;
}

/*
 * Lock a window's surface for direct access to its pixel data.
 */
static Bool
XDirectFBLockWindowSurface (RootlessFramePtr pFrame)
{
     DirectFBWindowRec *win = DFBWINREC(pFrame);
     
     return XDirectFBLockSurface (pFrame, win->surface, DSLF_READ | DSLF_WRITE);
}

/*
 * Unlock a surface.
 */
static Bool
XDirectFBUnlockSurface (RootlessFramePtr  pFrame,
                        IDirectFBSurface *surface)
{
     DFBResult ret;

     /* Unlock the window's surface. */
     ret = surface->Unlock (surface);
     if (ret) {
          DirectFBError ("XDirectFBUnlockSurface: IDirectFBSurface::Unlock", ret);
          return FALSE;
     }

     /* Reset data pointer, pitch etc. for safetiness. */
     pFrame->pixelData    = NULL;
     pFrame->bytesPerRow  = 0;
     pFrame->depth        = 0;
     pFrame->bitsPerPixel = 0;

     /* Update the data pointer in pixmap. */
     UpdatePixmap (pFrame->win);

     return TRUE;
}

/*
 * Lock a window's surface for direct access to its pixel data.
 */
static Bool
XDirectFBUnlockWindowSurface (RootlessFramePtr pFrame)
{
     DirectFBWindowRec *win = DFBWINREC(pFrame);
     
     return XDirectFBUnlockSurface (pFrame, win->surface);
}

/*
 * Set the drawing color to the color of 'pixel'.
 */
static void
XDirectFBSetColorByPixel (IDirectFBSurface *surface, unsigned int pixel)
{
     /* Lookup tables for conversion without shifting errors */
     static const __u8 lookup3to8[] = { 0x00, 0x24, 0x49, 0x6d, 0x92, 0xb6, 0xdb, 0xff};
     static const __u8 lookup2to8[] = { 0x00, 0x55, 0xaa, 0xff};

     DFBColor              color = { a: 0xff};
     DFBSurfacePixelFormat format;

     /* Get the pixel format being also the format of 'pixel' */
     surface->GetPixelFormat (surface, &format);

     /* Extract the color components */
     switch (format) {
          case DSPF_RGB332:
               color.r = lookup3to8[pixel >> 5];
               color.g = lookup3to8[(pixel & 0x1C) >> 2];
               color.b = lookup2to8[pixel & 0x03];
               break;

          case DSPF_ARGB1555:
               color.r = (pixel >> 7) & 0xF8;
               color.g = (pixel >> 2) & 0xF8;
               color.b = (pixel << 3) & 0xF8;
               break;

          case DSPF_RGB16:
               color.r = (pixel >> 8) & 0xF8;
               color.g = (pixel >> 3) & 0xFC;
               color.b = (pixel << 3) & 0xF8;
               break;

          case DSPF_ARGB:
               color.a = (pixel >> 24);
               /* fall through */

          case DSPF_RGB24:
          case DSPF_RGB32:
               color.r = (pixel >> 16) & 0xFF;
               color.g = (pixel >>  8) & 0xFF;
               color.b =  pixel        & 0xFF;
               break;

          default:
               return;
     }

     /* Set the color components */
     surface->SetColor (surface, color.r, color.g, color.b, color.a);
}

/*****************************************************************
 * Rootless mode callback glue                                   *
 *****************************************************************/

/*
 * Create a new toplevel or root window.
 */
static void
XDirectFBCreateFrame(ScreenPtr pScreen, RootlessFramePtr pFrame,
                     RootlessFramePtr pUpper)
{
     int sx = dixScreenOrigins[pScreen->myNum].x + directfbMainScreenX;
     int sy = dixScreenOrigins[pScreen->myNum].y + directfbMainScreenY;

     DirectFBScreenPtr priv = DIRECTFB_PRIV(pScreen);

     DirectFBWindowRec *win;

     /* Create a DirectFB window. */
     win = XDirectFBNewWindow (priv->layer,
                               pFrame->x + sx, pFrame->y + sy,
                               pFrame->w, pFrame->h);

     /* Assign private data pointer. */
     pFrame->devPrivate = win;

     /* Assign backpointer. */
     win->frame = pFrame;

     /* If it's the root window... */
     if (pFrame->isRoot) {
          IDirectFBWindow *window = win->window;

          /* ...put it into the lower stacking class */
          window->SetStackingClass (window, DWSC_LOWER);

          /* and lower it to the bottom within its class. */
          window->LowerToBottom (window);

          /* TODO: make runtime configurable */
          if (directfbPrefs.enableRoot) {
               /* Root window is always fully opaque. */
               win->opacity.focused = win->opacity.unfocused = 0xff;
          }
          else {
               /* Never show the root window. */
               win->opacity.focused = win->opacity.unfocused = 0x00;
          }
     }
     else if (pUpper) {
          IDirectFBWindow *window = win->window;

          /* Put the new window below another existing one. */
          window->PutBelow (window, DFBWINREC(pUpper)->window);
     }

     /* Lock its surface for direct access. */
     XDirectFBLockWindowSurface (pFrame);
}

/*
 * Destroy a toplevel or root window.
 */
static void
XDirectFBDestroyFrame(ScreenPtr pScreen, RootlessFramePtr pFrame)
{
     DirectFBWindowRec *win     = DFBWINREC(pFrame);
     IDirectFBWindow   *window  = win->window;
     IDirectFBSurface  *surface = win->surface;

     /* Unlock its surface. */
     surface->Unlock (surface);
     
     /* Release the surface. */
     surface->Release (surface);

     /* Hide/fade out. */
     XDirectFBWindowFadeOut (win);

     /* Remove it from the list of XDirectFB's windows. */
     XDirectFBRemoveWindow (win);
     
     /* Release the window (destroys it). */
     window->Release (window);

     /* Free our private data. */
     free (win);

     /* Reset the private data pointer. */
     pFrame->devPrivate = NULL;
}

/*
 * Move a toplevel window.
 */
static void
XDirectFBMoveFrame(ScreenPtr pScreen, RootlessFramePtr pFrame,
                   int oldX, int oldY)
{
     DirectFBWindowRec *win    = DFBWINREC(pFrame);
     IDirectFBWindow   *window = win->window;

     /* Calculate screen offset. */
     int sx = dixScreenOrigins[pScreen->myNum].x + directfbMainScreenX;
     int sy = dixScreenOrigins[pScreen->myNum].y + directfbMainScreenY;

     /* Unlock its surface. */
     XDirectFBUnlockWindowSurface (pFrame);

     /* Move it to the new position. */
     window->MoveTo (window, pFrame->x + sx, pFrame->y + sy);

     /* Lock its surface. */
     XDirectFBLockWindowSurface (pFrame);
}

/*
 * Start resizing of a toplevel window.
 */
static void
XDirectFBStartResizeFrame(ScreenPtr pScreen, RootlessFramePtr pFrame,
                          int oldX, int oldY,
                          unsigned int oldW, unsigned int oldH)
{
     DirectFBWindowRec *win     = DFBWINREC(pFrame);
     IDirectFBWindow   *window  = win->window;

     /* Calculate screen offset. */
     int sx = dixScreenOrigins[pScreen->myNum].x + directfbMainScreenX;
     int sy = dixScreenOrigins[pScreen->myNum].y + directfbMainScreenY;

     /* Unlock its surface. */
     XDirectFBUnlockWindowSurface (pFrame);

     /* Move and resize the window. */
     window->MoveTo (window, pFrame->x + sx, pFrame->y + sy);
     window->Resize (window, pFrame->w, pFrame->h);

     /* Lock its surface. */
     XDirectFBLockWindowSurface (pFrame);
}

/*
 * Finish resizing of a toplevel window.
 *
 * FIXME: only update the portion of the window that won't be repainted
 */
static void
XDirectFBFinishResizeFrame(ScreenPtr pScreen, RootlessFramePtr pFrame,
                           int oldX, int oldY,
                           unsigned int oldW, unsigned int oldH)
{
     DirectFBWindowRec *win     = DFBWINREC(pFrame);
     IDirectFBSurface  *surface = win->surface;

     /* Unlock its surface. */
     XDirectFBUnlockWindowSurface (pFrame);

     /* Update the whole (FIXME: see above) window. */
     surface->Flip (surface, NULL, 0);

     /* Lock its surface. */
     XDirectFBLockWindowSurface (pFrame);
}

/*
 * GetImage needs to fetch the screen content (composed window stack).
 * Therefore we lock the layer's surface for reading.
 */
static void
XDirectFBRootBeforeGetImage(ScreenPtr pScreen,
                            RootlessFramePtr pFrame,
                            int x, int y, int w, int h)
{
     DFBResult              ret;
     DirectFBScreenPtr      priv  = DIRECTFB_PRIV(pScreen);
     IDirectFBDisplayLayer *layer = priv->layer;
     DirectFBWindowRec     *win   = DFBWINREC(pFrame);

     /*
      * Get the surface of the layer that displays the window stack.
      */
     ret = layer->GetSurface (layer, &win->layer_surface);
     if (ret) {
          DirectFBError ("XDirectFBRootBeforeGetImage: IDirectFBDisplayLayer->GetSurface", ret);
          return;
     }

     /* Unlock the root window's surface. */
     XDirectFBUnlockWindowSurface (pFrame);

     /* Lock the layer's surface for reading. */
     XDirectFBLockSurface (pFrame, win->layer_surface, DSLF_READ);
}

/*
 * After GetImage is done we have to unlock the layer's surface.
 */
static void
XDirectFBRootAfterGetImage(ScreenPtr pScreen,
                           RootlessFramePtr pFrame)
{
     DirectFBWindowRec *win = DFBWINREC(pFrame);

     /* Unlock the layer's surface. */
     XDirectFBUnlockSurface (pFrame, win->layer_surface);

     /* Release the temporary interface. */
     win->layer_surface->Release (win->layer_surface);
     win->layer_surface = NULL;
     
     /* Lock the root window's surface. */
     XDirectFBLockWindowSurface (pFrame);

}

/*
 * Change the stacking order of a toplevel window.
 */
static void
XDirectFBRestackFrame(ScreenPtr        pScreen,
                      RootlessFramePtr pFrame,
                      RootlessFramePtr pOldUpper,
                      RootlessFramePtr pNewUpper)
{
     DirectFBWindowRec *win    = DFBWINREC(pFrame);
     IDirectFBWindow   *window = win->window;

     /* Unlock it surface. */
     XDirectFBUnlockWindowSurface (pFrame);

     /* If there's a window above... */
     if (pNewUpper) {
          DirectFBWindowRec *upwin    = DFBWINREC(pNewUpper);
          IDirectFBWindow   *upwindow = upwin->window;

          /* ...put the window below it. */
          window->PutBelow (window, upwindow);
     }
     else
          /* Make it the topmost window. */
          window->RaiseToTop (window);

     /* Lock it surface. */
     XDirectFBLockWindowSurface (pFrame);
}

/*
 * Change the shape of a toplevel window.
 */
static void
XDirectFBReshapeFrame(ScreenPtr pScreen, RootlessFramePtr pFrame,
                      RegionPtr pNewShape)
{
     int                i, num;
     DirectFBWindowRec *win      = DFBWINREC(pFrame);
     IDirectFBWindow   *window   = win->window;
     IDirectFBSurface  *surface  = win->surface;
     BoxRec             shapeBox = {0, 0, pFrame->w, pFrame->h};

     /* Get the region not belonging to the window. */
     REGION_INVERSE(pScreen, pNewShape, pNewShape, &shapeBox);

     /* Get the number of rectangles not belonging to the window. */
     num = REGION_NUM_RECTS(pNewShape);

     /* Unlock the window's surface. */
     XDirectFBUnlockWindowSurface (pFrame);

     /* If there are rectangles not belonging to the window... */
     if (num) {
          BoxRec *rects = REGION_RECTS(pNewShape);

          /* ...use a colorkey. */
          surface->SetColor (surface, 0xf0, 0x00, 0xc0, 0xff);

          /* Fill each rectangle with the colorkey. */
          for (i=0; i<num; i++)
               surface->FillRectangle (surface,
                                       rects[i].x1, rects[i].y1,
                                       rects[i].x2 - rects[i].x1 + 1,
                                       rects[i].y2 - rects[i].y1 + 1);

          /* Set the window colorkey. */
          window->SetColorKey (window, 0xf0, 0x00, 0xc0);

          /* Enable colorkeying for this window. */
          EnableColorkeying (window);
     }
     else
          /* Disable colorkeying for this window. */
          DisableColorkeying (window);

     /* Update the whole window. */
     surface->Flip (surface, NULL, 0);

     /* Ensure it's visible. */
     XDirectFBShowWindow (win);

     /* Lock its surface. */
     XDirectFBLockWindowSurface (pFrame);
}

/*
 * Copy boxes.
 */
static void
XDirectFBCopyBoxes(ScreenPtr pScreen, RootlessFramePtr pFrame,
                   BoxPtr pBox, int nBox, int dx, int dy)
{
     int                i;
     DirectFBWindowRec *win     = DFBWINREC(pFrame);
     IDirectFBSurface  *surface = win->surface;
     
     if (nBox <= 0)
          return;
     
     /* Unlock its surface. */
     XDirectFBUnlockWindowSurface (pFrame);

     /* For each box in the list... */
     for (i=0; i<nBox; i++) {
          DFBRectangle src = { pBox[i].x1 + dx,
               pBox[i].y1 + dy,
               pBox[i].x2 - pBox[i].x1,
               pBox[i].y2 - pBox[i].y1};

          /* ...perform a blit to it from the specified offset. */
          surface->Blit (surface, surface, &src, pBox[i].x1, pBox[i].y1);
     }

     /* Lock its surface. */
     XDirectFBLockWindowSurface (pFrame);
}

/*
 * Fill all boxes with a solid color.
 */
static void
XDirectFBPolyFillRect(ScreenPtr pScreen, RootlessFramePtr pFrame,
                      unsigned int pixel, int nBox, BoxPtr pBox)
{
     DirectFBWindowRec *win     = DFBWINREC(pFrame);
     IDirectFBSurface  *surface = win->surface;

     /* Unlock its surface. */
     XDirectFBUnlockWindowSurface (pFrame);

     /* Set the color encoded in 'pixel' */
     XDirectFBSetColorByPixel (surface, pixel);

     /* For each box... */
     while (nBox--) {
          /* ...perform a rectangle fill. */
          surface->FillRectangle (surface, pBox->x1, pBox->y1,
                                  pBox->x2 - pBox->x1, pBox->y2 - pBox->y1);

          pBox++;
     }

     /* Lock its surface. */
     XDirectFBLockWindowSurface (pFrame);
}

/*
 * Fill all boxes with a solid color.
 */
static void
XDirectFBFillRegion(ScreenPtr pScreen, RootlessFramePtr pFrame,
                    unsigned int pixel, RegionPtr pRegion)
{
     int                i, num  = REGION_NUM_RECTS(pRegion);
     BoxRec            *rects   = REGION_RECTS(pRegion);
     DirectFBWindowRec *win     = DFBWINREC(pFrame);
     IDirectFBSurface  *surface = win->surface;

     if (num <= 0)
          return;

     /* Unlock its surface. */
     XDirectFBUnlockWindowSurface (pFrame);

     /* Set the color encoded in 'pixel' */
     XDirectFBSetColorByPixel (surface, pixel);

     /* For each rectangle in the region... */
     for (i=0; i<num; i++) {
          /* ...perform a rectangle fill. */
          surface->FillRectangle (surface, rects[i].x1, rects[i].y1,
                                  rects[i].x2 - rects[i].x1,
                                  rects[i].y2 - rects[i].y1);
     }

     /* Lock its surface. */
     XDirectFBLockWindowSurface (pFrame);
}

/*
 * Update portions of a window, i.e. make changes to its data visible.
 */
static void
XDirectFBUpdateRegion(ScreenPtr pScreen, RootlessFramePtr pFrame,
                      RegionPtr pDamage)
{
     int                i, num  = REGION_NUM_RECTS(pDamage);
     BoxRec            *rects   = REGION_RECTS(pDamage);
     DirectFBWindowRec *win     = DFBWINREC(pFrame);
     IDirectFBSurface  *surface = win->surface;

     /* Unlock its surface. */
     XDirectFBUnlockWindowSurface (pFrame);

     /* For each rectangle in the region... */
     for (i=0; i<num; i++) {
          DFBRegion reg = { rects[i].x1, rects[i].y1,
               rects[i].x2, rects[i].y2};

          /* ...perform an update. */
          surface->Flip (surface, &reg, 0);
     }

     /* Ensure the window is visible. */
     XDirectFBShowWindow (win);

     /* Lock its surface. */
     XDirectFBLockWindowSurface (pFrame);
}

#if 0
static void
XDirectFBStartDrawing(ScreenPtr pScreen, RootlessFramePtr pFrame)
{
}

static void
XDirectFBStopDrawing(ScreenPtr pScreen, RootlessFramePtr pFrame)
{
}
#endif

/*
 * Function pointer table for rootless support.
 */
static RootlessFrameProcs directfbRootlessProcs = {
     CreateFrame:                XDirectFBCreateFrame,
     DestroyFrame:               XDirectFBDestroyFrame,
     MoveFrame:                  XDirectFBMoveFrame,
     StartResizeFrame:           XDirectFBStartResizeFrame,
     FinishResizeFrame:          XDirectFBFinishResizeFrame,
     RestackFrame:               XDirectFBRestackFrame,
     ReshapeFrame:               XDirectFBReshapeFrame,
     CopyBoxes:                  XDirectFBCopyBoxes,
     PolyFillRect:               XDirectFBPolyFillRect,
     FillRegion:                 XDirectFBFillRegion,
     UpdateRegion:               XDirectFBUpdateRegion,
     RootBeforeGetImage:         XDirectFBRootBeforeGetImage,
     RootAfterGetImage:          XDirectFBRootAfterGetImage
};


/*****************************************************************
 * Rootless mode initialization.                                 *
 * Exported by rootlessDirectFB.h                                *
 *****************************************************************/

/*
 * Setup the screen for rootless access.
 */
Bool
XDirectFBRootlessSetupScreen (int index, ScreenPtr pScreen)
{
     /* Setup generic rooless support. */
     return RootlessInit (pScreen, &directfbRootlessProcs);
}

/*
 * Destroys all remaining windows for a clean shutdown.
 */
void XDirectFBDestroyAll()
{
     while (windows)
          XDirectFBDestroyFrame (NULL, windows->frame);
}

/*
 * Event processing function called from directfbX.c
 *
 * Returns TRUE if the event is not relevant for input.
 */
Bool
XDirectFBProcessWindowEvent (DFBWindowEvent *event)
{
     static Bool          grabbed = 0;
     static unsigned int  buttons = 0;

     DirectFBWindowRec   *win;
     IDirectFBWindow     *window = NULL;

     /* Lookup the window via the window id. */
     win = XDirectFBLookupWindow (event->window_id);

     /* The interface to the window. */
     if (win)
          window = win->window;

     /* Examine the event. */
     switch (event->type) {
          case DWET_GOTFOCUS:
               /* Remember that the window is focused. */
               if (win)
                    win->focused = TRUE;

               XDirectFBClipboardGotFocus();

               /* If the focus influences the opacity... */
               if (win && directfbPrefs.enableUnfocused) {
                    /* Remember the unfocused opacity. */
                    window->GetOpacity (window, &win->opacity.unfocused);

                    /* Set the focused opacity. */
                    window->SetOpacity (window, win->opacity.focused);
               }

               /* Not relevant for input. */
               return TRUE;

          case DWET_LOSTFOCUS:
               /* Remember that the window is unfocused. */
               if (win)
                    win->focused = FALSE;

               XDirectFBClipboardLostFocus();

               /* If the focus influences the opacity... */
               if (win && directfbPrefs.enableUnfocused) {
                    /* Remember the focused opacity. */
                    window->GetOpacity (window, &win->opacity.focused);

                    /* Set the unfocused opacity. */
                    window->SetOpacity (window, win->opacity.unfocused);
               }

               /* Not relevant for input. */
               return TRUE;

          case DWET_BUTTONDOWN:
               /* Set the appropriate bit in the button mask. */
               buttons |= 1 << (event->button - DIBI_FIRST + 1);

               /* Grab the pointer if it's not grabbed yet. */
               if (window && !grabbed && window->GrabPointer (window) == DFB_OK)
                    grabbed = TRUE;

               break;

          case DWET_BUTTONUP:
               /* Clear the appropriate bit in the button mask. */
               buttons &= ~(1 << (event->button - DIBI_FIRST + 1));

               /* If no more buttons are pressed and the pointer is grabbed... */
               if (window && grabbed && !buttons) {
                    /* ...ungrab it. */
                    window->UngrabPointer (window);
                    grabbed = FALSE;
               }

               break;

          default:
               ;
     }

     /* May be relevant for input. */
     return FALSE;
}

void
XDirectFBSetCursorShape (IDirectFBSurface *shape, int hot_x, int hot_y)
{
     DirectFBWindowRec *win = windows;

     /* Loop through the list... */
     while (win) {
          /* The interface to the window. */
          IDirectFBWindow *window = win->window;

          /* Set the new shape. */
          window->SetCursorShape (window, shape, hot_x, hot_y);

          win = win->next;
     }
}

