/*
 * JBoss, Home of Professional Open Source
 * Copyright XXXX, Red Hat Middleware LLC, and individual contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a full listing
 * of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU General Public License, v. 2.0.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License,
 * v. 2.0 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */
package org.mobicents.media.server.impl.resource.dtmf;

import java.util.ArrayList;
import org.mobicents.media.Buffer;
import org.mobicents.media.Format;
import org.mobicents.media.format.AudioFormat;
import org.mobicents.media.server.impl.AbstractSource;
import org.mobicents.media.server.spi.dsp.Codec;
import org.mobicents.media.server.spi.dsp.CodecFactory;
import org.mobicents.media.server.spi.resource.DtmfGenerator;

/**
 * InbandGenerator generates Inband DTMF Tone only for uncompressed LINEAR
 * codec. After creating instance of InbandGenerator, it needs to be initialized
 * so that all the Tones are generated and kept ready for transmission once
 * start is called.
 * 
 * By default the Tone duration is 80ms. This is suited for Tone Detector who
 * has Tone duration of greater than 40 and less than 80ms. For Tone Detector
 * who's Tone duration is set greater than 80ms may fail to detect Tone
 * generated by InbandGenerator(with duration 80ms). In that case increase the
 * duration here too.
 * 
 * @author Oleg Kulikov
 * @author amit bhayani
 */
public class GeneratorImpl extends AbstractSource implements DtmfGenerator {

    private final static AudioFormat LINEAR_AUDIO = new AudioFormat(
            AudioFormat.LINEAR, 8000, 16, 1,
            AudioFormat.LITTLE_ENDIAN,
            AudioFormat.SIGNED);
    
    private final static Format[] FORMATS = new Format[]{
        Codec.LINEAR_AUDIO,
        Codec.PCMA,
        Codec.PCMU,
        Codec.GSM,
        Codec.SPEEX,
        Codec.G729
    };
    
    public final static String[][] events = new String[][]{
        {"1", "2", "3", "A"},
        {"4", "5", "6", "B"},
        {"7", "8", "9", "C"},
        {"*", "0", "#", "D"}
    };
    private int[] lowFreq = new int[]{697, 770, 852, 941};
    private int[] highFreq = new int[]{1209, 1336, 1477, 1633};
    private String digit = null;    // Min duration = 40ms and max = 500ms
    private int duration = 50;
    private short A = Short.MAX_VALUE / 2;
    private int volume = 0;
    private int f1,  f2;
    private double dt;
    private int pSize;
    private double time = 0;
    private Codec codec;

    private final static ArrayList<CodecFactory> codecFactories = new ArrayList();
    static {
        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.g711.alaw.DecoderFactory());
        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.g711.alaw.EncoderFactory());

        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.g711.ulaw.DecoderFactory());
        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.g711.ulaw.EncoderFactory());

        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.gsm.DecoderFactory());
        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.gsm.EncoderFactory());

        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.speex.DecoderFactory());
        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.speex.EncoderFactory());

        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.g729.DecoderFactory());
        codecFactories.add(new org.mobicents.media.server.impl.dsp.audio.g729.EncoderFactory());
    }
    
    public GeneratorImpl(String name) {
        super(name);
        dt = 1 / LINEAR_AUDIO.getSampleRate();
    }

    @Override
    public void start() {
        if (digit == null) {
            return;
        }
        
        super.start();
    }
    
    @Override
    public void setPreffered(Format fmt) {
        codec = selectCodec(fmt);
        super.setPreffered(fmt);
    }
    
    private Codec selectCodec(Format f) {
        for (CodecFactory factory : codecFactories) {
            if (factory.getSupportedOutputFormat().matches(f) &&
                    factory.getSupportedInputFormat().matches(Codec.LINEAR_AUDIO)) {
                return factory.getCodec();
            }
        }
        return null;
    }
    
    public void setDigit(String digit) {
        this.digit = digit;
        for (int i = 0; i < 4; i++) {
            for (int j = 0; j < 4; j++) {
                if (events[i][j].equalsIgnoreCase(digit)) {
                    f1 = lowFreq[i];
                    f2 = highFreq[j];
                }
            }
        }
    }

    public String getDigit() {
        return this.digit;
    }

    public void setToneDuration(int duration) {
        if (duration < 40) {
            throw new IllegalArgumentException("Duration cannot be less than 40ms");
        }
        this.duration = duration;
    }

    public int getToneDuration() {
        return duration;
    }

    public int getVolume() {
        return this.volume;
    }

    public void setVolume(int volume) {
        if (volume > 0) {
            throw new IllegalArgumentException("Volume has to be negative value expressed in dBm0");
        }
        this.volume = volume;
        A = (short) (Math.pow(Math.pow(10, volume), 0.1) * (Short.MAX_VALUE / 2));
    }

    private short getValue(double t) {
        return (short) (A * (Math.sin(2 * Math.PI * f1 * t) + Math.sin(2 * Math.PI * f2 * t)));
    }

    public Format[] getFormats() {
        return FORMATS;
    }

    @Override
    public void evolve(Buffer buffer, long timestamp) {
        int k = 0;

        int frameSize = (int) ((double) 20 / 1000.0 / dt);

        byte[] data = new byte[2 * frameSize];
        for (int i = 0; i < frameSize; i++) {
            short v = getValue(time + dt * i);
            data[k++] = (byte) v;
            data[k++] = (byte) (v >> 8);
        }
        buffer.setHeader(new DtmfEvent(this, DtmfEvent.getId(digit), volume));
        buffer.setData(data);
        buffer.setOffset(0);
        buffer.setLength(2* frameSize);
        buffer.setFormat(LINEAR_AUDIO);
        buffer.setTimeStamp(getMediaTime());
        buffer.setDuration(20);

        if (time == 0) {
            buffer.setFlags(Buffer.FLAG_KEY_FRAME);
        }
        
        time += ((double) 20) / 1000.0;
        buffer.setEOM(time > (double) duration / 1000.0);
        
        if (codec != null) {
            codec.process(buffer);
        }
    }

    @Override
    public void stop() {
    }

	/* (non-Javadoc)
	 * @see org.mobicents.media.server.impl.AbstractSource#getInterface(java.lang.Class)
	 */
	@Override
	public <T> T getInterface(Class<T> interfaceType) {
		if(interfaceType.equals(DtmfGenerator.class))
		{
			return (T) this;
		}else
		{
			return null;
		}
	}
	
}

