package org.hibernate.plugins;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.hibernate.plugins.guards.*;
import org.hibernate.plugins.util.Util;

import java.io.File;
import java.util.*;

/**
 * Create safe guards around all sql statements
 *
 * @goal createSafeGuards
 * @phase process-classes
 * @requiresDependencyResolution compile
 */
public class CreateSafeGuards extends AbstractMojo {
    private static Map<String, Guard> DIALECT_GUARD_MAP = new HashMap<String, Guard>() {{
        put("PostgreSQLDialect", new PostgreSQLGuard());
        put("MySQL5InnoDBDialect", new MySQL5InnoDBGuard());
        put("Oracle10gDialect", new Oracle10gGuard());
        put("H2Dialect", new H2Guard());
        put("SQLServer2008Dialect", new SQLServer2008Guard());
        put("DB2Dialect", new DB2Guard());
    }};

    /**
     * Set input directory
     * The sql statements of the files in this directory are overwritten by the safe-guarded sql statements
     *
     * @parameter
     * @required
     * @readonly
     */
    private String inputDir;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        Set<File> sqlFiles = Util.getSqlFiles(inputDir);

        for (File sqlFile : sqlFiles) {
            crateSafeGuard(sqlFile);
        }
    }

    private void crateSafeGuard(File sqlFile) throws MojoExecutionException {
        String dialect = Util.getSqlFileDialect(sqlFile);
        Guard guard = DIALECT_GUARD_MAP.get(dialect);
        List<String> sqls = Util.loadSqlStatementsFromFile(sqlFile);
        List<String> guardedSqls = new ArrayList<String>();

        for (String sql : sqls) {
            String guardedSql = guard.safeGuard(sql);
            guardedSqls.add(guardedSql);
        }

        String joinedSqls = "\n    " + Util.join(guardedSqls, Util.SQL_SEPARATOR);
        Util.writeFile(sqlFile, joinedSqls);
    }
}
