package com.redhat.installer.validator;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.installer.DataValidator;
import com.redhat.installer.util.ValidatorConstants;

/**
 * Validator checks bin/product.conf and modules/layers.conf to try and determine what product we're currently 
 * installing onto. 
 * @author thauser
 *
 */

public class EapExistsValidator implements DataValidator {
    static final String PACK_ID = "eap";
    AutomatedInstallData idata;
	String error;
	Status status;

	@Override
        public Status validateData(AutomatedInstallData idata) 
	{
        this.idata = idata;
        //System.out.println("== In EapExistsValidator");
		String path = idata.getVariable("INSTALL_PATH");
		String subdir = idata.getVariable("INSTALL_SUBPATH");
		try {
			switch (eapExistsAt(path+File.separator+subdir, idata)){
			case 0:
				return Status.OK;
			case 1:
				return Status.WARNING;
			case 2:
				return Status.ERROR;
			default:
				return Status.ERROR;
			}
		}
		catch (IOException e)
		{
			setError("IOE in EapExistsValidator");
			return Status.ERROR;
		}
	}
	
	/** Returns a status based upon the results of a look into the provided INSTALL_PATH
	 * Status Meanings
	 * ===============
	 * 0: Found every file we needed. [OK]
	 * 1: Some files are missing, but we can just warn the user [WARNING]
	 * 2: critical files are missing. [ERROR]
	 * 
	 * Variables that will be set
	 * =============================
	 * 	ValidatorConstants.existingLayers   [existing.layers.conf]
	 *	ValidatorConstants.existingProduct  [existing.product.conf]
	 */
	//TODO: Localize all of the strings
	private int eapExistsAt(String path, AutomatedInstallData idata) throws IOException 
	{
		int status = 0;		
		File emptyFolderCheck = new File(path);
		
		// this is necessary because we could have the user go back and forth, which would mean old values will
		// destroy the logic at the end of the installation
		idata.setVariable(ValidatorConstants.existingLayers, "");
		idata.setVariable(ValidatorConstants.existingProduct, "");
		
		

		/** Installing to a new folder
		 * 1) Flag the EAP needs to be installed
		 * 2) Set the "existing.product.conf" variable to EAP
		 * 3) Add the EAP pack to the TreePacks
		 */
		if (!emptyFolderCheck.exists())
		{
			idata.setVariable("eap.needs.install", "true");
			idata.setVariable(ValidatorConstants.existingProduct, ValidatorConstants.eap);		
            idata.enablePackageInstall(PACK_ID);
			return status;
		}
		
		
		// if we get to here, we have to try and confirm that EAP does exist here. 
		// EAP exists if the following are all true:
		//   a) $JBOSS_HOME/bin/product.conf exists and contains "slot=eap"
		//   b) $JBOSS_HOME/modules/layers.conf does not exist.
		//   c) $JBOSS_HOME/standalone/configuration/standalone.xml exists
		//   d) $JBOSS_HOME/bin/standalone.sh exists
		//   e) $JBOSS_HOME/domain/configuration/domain.xml exists
		//   f) $JBOSS_HOME/domain/configuration/host.xml exists
		//   g) $JBOSS_HOME/bin/domain.sh exists
                //   h) $JBOSS_HOME/version.txt contains EAP 6.1.1.GA

		boolean productExists =
					readFileAndSetVariable(path+ValidatorConstants.productConfLoc, ValidatorConstants.existingProduct, idata, "slot");
		boolean layersExists =
				readFileAndSetVariable(path+ValidatorConstants.layersConfLoc, ValidatorConstants.existingLayers, idata, "layers");
		if (!productExists) {
			if (!idata.getVariable(ValidatorConstants.existingProduct).equals(ValidatorConstants.eap) && !layersExists) {
				setError(String.format("The required product.conf or layers.conf file at: %s is missing. Please supply either an empty directory or an EAP installation.", path
						+ ValidatorConstants.productConfLoc));
				status = 2; // no product conf is not possible for a valid EAP
							// installation. fail immediately.
				return status; // fail instantly
			}
		}
		
		/** Check that standalone.sh and standalone.xml exists, if they do not exist return a warning */
		File standaloneScript = new File(path+ValidatorConstants.standaloneLoc);
		File standaloneXml    = new File(path+ValidatorConstants.standaloneXmlLoc);
		if (!standaloneScript.exists() && !standaloneXml.exists())
		{
			setError("Standalone configuration and standalone.sh are missing. Is this really an EAP installation?");
			status = 1;
			return status;
		}
		else 
		{
			idata.setVariable("installer.has.standalone", "true");
		}
		
		/** Check that domain.sh, domain.xml and host.xml exists, if they do not exist return a warning*/
		File domainScript = new File(path+ValidatorConstants.domainLoc);
		File domainXml = new File(path+ValidatorConstants.domainXmlLoc);
		File domainHostXml = new File(path+ValidatorConstants.hostXmlLoc);
		if (!domainScript.exists() && !domainXml.exists() && !domainHostXml.exists())
		{
			setError("Domain configuration and domain.sh are missing. Is this really an EAP installation?");
			status = 1;
			return status;
		} 
		else 
		{
			idata.setVariable("installer.has.domain", "true");
		}
		
		
		if (!standaloneScript.exists() && !standaloneXml.exists() && !domainScript.exists() && !domainHostXml.exists()){
			setError("No server start scripts could be found. Neither domain.sh nor standalone.sh exist in this path.");
			status = 2;
			return status;
		}
		
                File versionText = new File(path+File.separator+"version.txt");
                BufferedReader br = new BufferedReader(new InputStreamReader(new FileInputStream(versionText)));
                String versionLine = br.readLine ();

                if ( ! versionLine.contains ("6.1.1.GA"))
                {
                    setError ("Invalid EAP version found in version.txt (" + versionLine + ")");
                    status = 2;
                    return status;
                }

		/** EAP installation verified */
		idata.setVariable("eap.needs.install","false");
        idata.disablePackageInstall(PACK_ID);
		return status;
	}
	
	/**
	 * Reads the given file, and fills the given variable with the result.<br/>
	 * 
	 * @param path path of the file
	 * @param variable variable to fill with contents of said file
	 * @param idata idata reference
	 * @param filePrefix the prefix the valid line should start with (either slot= or layers=)
	 * @return
	 * @throws IOException
	 */
	private boolean readFileAndSetVariable(String path, String variable, AutomatedInstallData idata, String filePrefix) throws IOException{
		File file = new File(path);
		String value = getValidLineFromConf(file, filePrefix);
		if (value == null)
		{
			return false;
		} 
		else 
		{
			idata.setVariable(variable, value);
		}
		return true;
	}

	/**
	 * Simple method that looks for a valid line in a file of the following format:<br/>
	 * Lines starting with # denote a comment and are ignored.<br/>
	 * Other than commented lines, files consist of a single line containing a <word>=<value> key value pair. <br/>
	 * If this is not the case, the method returns null.
	 * @param file
	 * @return
	 * @throws IOException
	 */
	private String getValidLineFromConf(File file, String prefix) throws IOException {
		if (file.exists())
		{
			BufferedReader br = new BufferedReader(new InputStreamReader(new FileInputStream(file)));
			String line = "";
			
			while ((line = br.readLine()) != null)
			{
				if (line.startsWith("#")) //Ignore Comments
				{
					continue;
				} 
				else 					  // First line we find is what we need, unless it's malformed
				{
					break;
				}
			}	
			
			if (line == null || br.readLine() != null || !line.startsWith(prefix))
			{ // there are no lines, or there are > 1 lines, or the first non-comment line is not valid for this file
				br.close();
				return null;
			}
			
			br.close();
			String[] split = line.split("=");
			if (split.length < 2)
			{
				return null; // Malformed file
			}
			return split[1];		
		} 
		return null;
	}

	public void setError(String e)
	{
		this.error = e;
	}

	public String getErrorMessageId() 
	{
		return error;
	}

	public String getWarningMessageId() 
	{
		return error;
	}

	/**
	 * This is the defaultAnswer upon validateData returning a Status.WARNING. Since the only warning is about 
	 * the platform already existing, I think we should return false, because returning true immediately may result
	 * in a loss of user data.
	 */
	public boolean getDefaultAnswer() 
	{
		return false;
	}

}
