/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.servicemix.eip;

import javax.jbi.messaging.ExchangeStatus;
import javax.jbi.messaging.InOnly;
import javax.jbi.messaging.InOut;
import javax.jbi.messaging.NormalizedMessage;
import javax.xml.namespace.QName;

import org.w3c.dom.Node;

import org.apache.servicemix.eip.patterns.ContentBasedRouter;
import org.apache.servicemix.eip.support.RoutingRule;
import org.apache.servicemix.eip.support.XPathPredicate;
import org.apache.servicemix.jbi.jaxp.SourceTransformer;
import org.apache.servicemix.jbi.messaging.NormalizedMessageImpl;
import org.apache.servicemix.tck.ReceiverComponent;

public class ContentBasedRouterTest extends AbstractEIPTest {

    private ContentBasedRouter router;

    protected void setUp() throws Exception {
        super.setUp();

        router = new ContentBasedRouter();
        router.setRules(new RoutingRule[] {
            new RoutingRule(
                    new XPathPredicate("/hello/@id = '1'"),
                    createServiceExchangeTarget(new QName("target1"))),
            new RoutingRule(
                    new XPathPredicate("/hello/@id = '2'"),
                    createServiceExchangeTarget(new QName("target2"))),
            new RoutingRule(
                    null,
                    createServiceExchangeTarget(new QName("target3")))
        });
        configurePattern(router);
        activateComponent(router, "router");
    }
    
    public void testInOnly() throws Exception {
        ReceiverComponent rec1 = activateReceiver("target1");
        ReceiverComponent rec2 = activateReceiver("target2");
        ReceiverComponent rec3 = activateReceiver("target3");
        
        InOnly me = client.createInOnlyExchange();
        me.setService(new QName("router"));
        me.getInMessage().setContent(createSource("<hello id='1' />"));
        client.sendSync(me);
        assertEquals(ExchangeStatus.DONE, me.getStatus());
        
        rec1.getMessageList().assertMessagesReceived(1); 
        rec2.getMessageList().assertMessagesReceived(0);
        rec3.getMessageList().assertMessagesReceived(0);

        rec1.getMessageList().flushMessages();
        rec2.getMessageList().flushMessages();
        rec3.getMessageList().flushMessages();

        me = client.createInOnlyExchange();
        me.setService(new QName("router"));
        me.getInMessage().setContent(createSource("<hello id='2' />"));
        client.sendSync(me);
        assertEquals(ExchangeStatus.DONE, me.getStatus());
        
        rec1.getMessageList().assertMessagesReceived(0);
        rec2.getMessageList().assertMessagesReceived(1);
        rec3.getMessageList().assertMessagesReceived(0);

        rec1.getMessageList().flushMessages();
        rec2.getMessageList().flushMessages();
        rec3.getMessageList().flushMessages();

        me = client.createInOnlyExchange();
        me.setService(new QName("router"));
        me.getInMessage().setContent(createSource("<hello id='3' />"));
        client.sendSync(me);
        assertEquals(ExchangeStatus.DONE, me.getStatus());
        
        rec1.getMessageList().assertMessagesReceived(0);
        rec2.getMessageList().assertMessagesReceived(0);
        rec3.getMessageList().assertMessagesReceived(1);
    }

    public void testInOut() throws Exception {
        activateComponent(new ReturnMockComponent("<from1/>"), "target1");
        activateComponent(new ReturnMockComponent("<from2/>"), "target2");
        activateComponent(new ReturnMockComponent("<from3/>"), "target3");
        
        InOut me = client.createInOutExchange();
        me.setService(new QName("router"));
        me.getInMessage().setContent(createSource("<hello id='1' />"));
        client.sendSync(me);
        assertEquals(ExchangeStatus.ACTIVE, me.getStatus());
        Node node = new SourceTransformer().toDOMNode(me.getOutMessage());
        assertEquals("from1", node.getFirstChild().getNodeName());
        client.done(me);
        
        me = client.createInOutExchange();
        me.setService(new QName("router"));
        me.getInMessage().setContent(createSource("<hello id='2' />"));
        client.sendSync(me);
        assertEquals(ExchangeStatus.ACTIVE, me.getStatus());
        node = new SourceTransformer().toDOMNode(me.getOutMessage());
        assertEquals("from2", node.getFirstChild().getNodeName());
        client.done(me);
        
        me = client.createInOutExchange();
        me.setService(new QName("router"));
        me.getInMessage().setContent(createSource("<hello id='3' />"));
        client.sendSync(me);
        assertEquals(ExchangeStatus.ACTIVE, me.getStatus());
        node = new SourceTransformer().toDOMNode(me.getOutMessage());
        assertEquals("from3", node.getFirstChild().getNodeName());
        client.done(me);
    }

    public void testForwardOperation() throws Exception {
        router.setForwardOperation(true);
        ReceiverComponent rec = activateReceiver("target1");

        InOnly me = client.createInOnlyExchange();
        me.setService(new QName("router"));
        me.setOperation(new QName("operation"));
        me.getInMessage().setContent(createSource("<hello id='1' />"));
        client.sendSync(me);
        assertEquals(ExchangeStatus.DONE, me.getStatus());

        rec.getMessageList().assertMessagesReceived(1);

        NormalizedMessageImpl msg = (NormalizedMessageImpl) rec.getMessageList().flushMessages().get(0);
        assertEquals(new QName("operation"), msg.getExchange().getOperation());

    }

}
