/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.felix.framework;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.felix.framework.monitor.MonitoringService;
import org.apache.felix.framework.util.FelixConstants;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleEvent;
import org.osgi.framework.BundleListener;
import org.osgi.framework.SynchronousBundleListener;
import org.osgi.framework.wiring.BundleWiring;

public class MonitorServiceImpl implements MonitoringService, SynchronousBundleListener {

    private final Felix felix;
    private final ThreadGroup rootThreadGroup;
    private final Map<String, ExecutorService> executors;
    private final Map<String, ThreadGroup> threadGroups;
    private final Map<ThreadGroup, Boolean> allThreadGroups;
    private final Map<ClassLoader, Boolean> classLoaders;

    public MonitorServiceImpl(Felix felix) {
        this.felix = felix;
        rootThreadGroup = new ThreadGroup("bundles");
        executors = new HashMap<String, ExecutorService>();
        threadGroups = new HashMap<String, ThreadGroup>();
        allThreadGroups = new WeakHashMap<ThreadGroup, Boolean>();
        classLoaders = new WeakHashMap<ClassLoader, Boolean>();
    }

    public Future<?> runInContext(final Bundle bundle, Runnable run) {
        return felix.runInContext((BundleImpl) bundle, run);
    }
    
    public <T> Future<T> runInContext(Bundle bundle, Callable<T> callable) {
        return felix.runInContext((BundleImpl) bundle, callable);
    }

    public ExecutorService getExecutor(Bundle bundle) {
        ExecutorService executor;
        synchronized (executors) {
            String key = bundle.getSymbolicName() + "-" + bundle.getVersion();
            executor = executors.get(key);
            if (executor == null) {
                String timeoutStr = (String) felix.getConfig().get(FelixConstants.FELIX_THREADING_TIMEOUT);
                long timeout = timeoutStr != null ? Long.parseLong(timeoutStr) : 60L;
                executor = new ThreadPoolExecutor(0, Integer.MAX_VALUE,
                        timeout, TimeUnit.SECONDS,
                        new SynchronousQueue<Runnable>(),
                        new MonitoringThreadFactory(bundle));
            }
            executors.put(key, executor);
        }
        return executor;
    }

    class MonitoringThreadFactory implements ThreadFactory {
        private final Bundle bundle;
        private final ClassLoader classLoader;
        private final ThreadGroup group;
        private final AtomicInteger threadNumber = new AtomicInteger(1);
        private final String namePrefix;

        MonitoringThreadFactory(Bundle bundle) {
            this.bundle = bundle;
            this.classLoader = bundle.adapt(BundleWiring.class).getClassLoader();
            group = getThreadGroup(bundle);
            namePrefix = "pool-" +
                    bundle.getSymbolicName() + "-" + bundle.getVersion().toString() +
                    "-thread-";
        }

        public Thread newThread(Runnable r) {
            Thread t = new Thread(group, r,
                    namePrefix + threadNumber.getAndIncrement(),
                    0);
            if (t.isDaemon())
                t.setDaemon(false);
            if (t.getPriority() != Thread.NORM_PRIORITY)
                t.setPriority(Thread.NORM_PRIORITY);
            t.setContextClassLoader(classLoader);
            return t;
        }
    }

    public ThreadGroup getThreadGroup(Bundle bundle) {
        synchronized (threadGroups) {
            String key = bundle.getSymbolicName() + "-" + bundle.getVersion();
            ThreadGroup tg = threadGroups.get(key);
            if (tg == null) {
                tg = new ThreadGroup(rootThreadGroup, key);
                threadGroups.put(key, tg);
                allThreadGroups.put(tg, true);
            }
            return tg;
        }
    }

    public Collection<ThreadGroup> getThreadGroups() {
        synchronized (threadGroups) {
            return allThreadGroups.keySet();
        }
    }

    void uninstallBundle(Bundle bundle) {
        String key = bundle.getSymbolicName() + "-" + bundle.getVersion();
        synchronized (executors) {
            ExecutorService executor = executors.remove(key);
            if (executor != null) {
                executor.shutdown();
            }
        }
        synchronized (threadGroups) {
            ThreadGroup tg = threadGroups.remove(key);
            if (tg != null) {
                tg.setDaemon(true);
                try {
                    tg.destroy();
                } catch (Throwable t) {
                    // Ignore
                }
            }
        }
    }

    public Collection<ClassLoader> getClassLoaders() {
        synchronized (classLoaders) {
            return classLoaders.keySet();
        }
    }

    void addClassLoader(ClassLoader classLoader) {
        synchronized (classLoaders) {
            classLoaders.put(classLoader, true);
        }
    }
    
    public void start() {
        felix.addBundleListener(felix, this);
    }

    public void destroy() {
        synchronized (executors) {
            for (ExecutorService ex : executors.values()) {
                ex.shutdown();
            }
            executors.clear();
        }
        synchronized (threadGroups) {
            for (ThreadGroup tg : threadGroups.values()) {
                tg.setDaemon(true);
                try {
                    tg.destroy();
                } catch (Throwable t) {
                    // Ignore
                }
            }
            threadGroups.clear();
        }
        synchronized (classLoaders) {
            classLoaders.clear();
        }
    }

    public void bundleChanged(BundleEvent event) {
        if (event.getType() == BundleEvent.UPDATED
                || event.getType() == BundleEvent.UNINSTALLED
                || event.getType() == BundleEvent.UNRESOLVED) {
            uninstallBundle(event.getBundle());
        }
    }
}
