﻿//  ------------------------------------------------------------------------------------
//  Copyright (c) 2015 Red Hat, Inc.
//  All rights reserved.
//
//  Licensed under the Apache License, Version 2.0 (the ""License""); you may not use this
//  file except in compliance with the License. You may obtain a copy of the License at
//  http://www.apache.org/licenses/LICENSE-2.0
//
//  THIS CODE IS PROVIDED *AS IS* BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION ANY IMPLIED WARRANTIES OR
//  CONDITIONS OF TITLE, FITNESS FOR A PARTICULAR PURPOSE, MERCHANTABLITY OR
//  NON-INFRINGEMENT.
//
//  See the Apache Version 2.0 License for specific language governing permissions and
//  limitations under the License.
//  ------------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Amqp;
using Amqp.Framing;
using Amqp.Types;

// The ReceiveSelector.cs example demonstrates how to set up filters 
// on the receiver connection to the broker.
// 
// The ReceiverLink is created with a filter map of selector filters.
// In this example a filter named 'f1' filters on the string property
// 'colour = 'green''. Messages must have a property named 'colour' 
// whose value is 'green' before the message is passed to this receiver.
//

namespace ReceiverSelector
{
    class ReceiverSelector
    {
        //
        // Sample invocation: ReceiveSelector [brokerUrl [serviceName [filterSpec]]]
        //
        static void Main(string[] args)
        {
            //
            // on_message
            //
            MessageCallback onMessage = (link, message) =>
            {
                Console.WriteLine("Message(Properties={0}, ApplicationProperties={1}, Body={2}",
                                  message.Properties, message.ApplicationProperties, message.Body);
                link.Accept(message);
            };

            //
            // start
            //
            String urlString    = args.Length >= 1 ? args[0] : "amqp://guest:guest@localhost:5672";
            String amqpAddress  = args.Length >= 2 ? args[1] : "examples";
            String filterString = args.Length >= 3 ? args[2] : "colour = 'green'";

            Address urlAddress = new Address(urlString);

            Connection connection = new Connection(urlAddress);
            Session session = new Session(connection);

            Map filters = new Map();
            filters.Add(new Symbol("f1"),
                        new DescribedValue(
                            new Symbol("apache.org:selector-filter:string"),
                            filterString));

            OnAttached onAttached = (l, a) => {};

            ReceiverLink receiver = new ReceiverLink(session,
                                                     "Selected_Recv-receiver",
                                                     new Source()
                                                     {
                                                         Address = amqpAddress,
                                                         FilterSet = filters
                                                     }, onAttached);
            receiver.Start(10, onMessage);

            //
            // run
            //
            Console.WriteLine("Press enter key to exit...");
            Console.ReadLine();

            //
            // shutdown
            //
            receiver.Close();
            session.Close();
            connection.Close();
        }
    }
}
