/*
 * #%L
 * Fuse EAP :: Installer
 * %%
 * Copyright (C) 2015 RedHat
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
package org.jboss.fuse.eap.installer.internal;

import java.io.File;
import java.io.FilenameFilter;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.Locale;

import org.wildfly.extras.patch.installer.AbstractInstaller;
import org.wildfly.extras.patch.installer.Version;

/**
 * Main class that installs the fuse-eap-distro.
 */
public final class Main {

    public static void main(String[] originalArgs) throws Exception {
        AbstractInstaller installer = new AbstractInstaller(){
            private boolean upgradingFromOldVersion = false;

            @Override
            public String getJarName() {
                return "fuse-eap-installer.jar";
            }

            // [TODO: ENTESB-5503] Remove this when we're confident everyone has upgraded to fuse patch >= 2.4.0
            @Override
            protected boolean mustInstallPatchDistro(Path fusePatchPath) {
                boolean mustInstall = super.mustInstallPatchDistro(fusePatchPath);

                if (isWindows()) {
                    if (mustInstall) {
                        // No patch distro is installed yet
                        return true;
                    }

                    // Special handling for upgrade from fuse-patch versions less than 2.4.0 on Windows
                    // https://github.com/wildfly-extras/fuse-patch/issues/150
                    // https://issues.jboss.org/browse/ENTESB-5503
                    File moduleBase = fusePatchPath.resolve("main").toFile();
                    if (moduleBase.exists()) {
                        String[] patchJars = moduleBase.list(new FilenameFilter() {
                            @Override
                            public boolean accept(File dir, String name) {
                                return name.matches("fuse-patch-core-[0-9.]+.jar");
                            }
                        });

                        if (patchJars.length == 0) {
                            throw new IllegalStateException("Unable to determine current fuse patch version");
                        }

                        Version currentVersion = Version.get(patchJars[0].replace("fuse-patch-core-", "").replace(".jar", ""));
                        Version minimumVersion = Version.get("2.4.0");

                        if (currentVersion.compareTo(minimumVersion) == -1) {
                            upgradingFromOldVersion = true;
                            return true;
                        }
                    }

                    // Delegate to the usual process of fuse patch performing a self upgrade
                    return false;
                }

                return mustInstall;
            }

            // [TODO: ENTESB-5503] Remove this when we're confident everyone has upgraded to fuse patch >= 2.4.0
            @Override
            protected void runCommand(String cmd) throws Exception {
                Version installerVersion = getVersion();

                if (isWindows() && upgradingFromOldVersion && cmd.endsWith("update fuse-patch-distro-wildfly")) {
                    // If fuse patch >= 2.4.0 is already installed, we can delegate distro updating to fuse patch. Else we suppress this command
                    Path fusePatchJar = eapHome.resolve(Paths.get("modules/system/layers/fuse/org/wildfly/extras/patch/main/fuse-patch-core-" + installerVersion + ".jar"));
                    if (fusePatchJar.toFile().exists()) {
                        // Force to ensure patch metadata is updated
                        super.runCommand(cmd + " --force");
                    }
                } else {
                    super.runCommand(cmd);
                }
            }

            private boolean isWindows() {
                return System.getProperty("os.name").toLowerCase(Locale.US).contains("win");
            }
        };

        installer.main(new LinkedList<>(Arrays.asList(originalArgs)));
    }
}
