/*
 * Copyright 2019, EnMasse authors.
 * License: Apache License 2.0 (see the file LICENSE or http://apache.org/licenses/LICENSE-2.0.html).
 */

package io.enmasse.controller;

import io.enmasse.address.model.Phase;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.enmasse.address.model.AddressSpace;
import io.enmasse.address.model.AddressSpaceBuilder;

import java.util.Objects;

/**
 * Abstract base class for finalizer controllers.
 */
public abstract class AbstractFinalizerController implements Controller {

    private static final Logger log = LoggerFactory.getLogger(AbstractFinalizerController.class.getName());

    protected static interface Result {
        public AddressSpace getAddressSpace();

        public boolean isFinalized();

        public static Result waiting(final AddressSpace addressSpace) {
            return new Result() {
                @Override
                public boolean isFinalized() {
                    return false;
                }

                @Override
                public AddressSpace getAddressSpace() {
                    return addressSpace;
                }
            };
        }

        public static Result completed(final AddressSpace addressSpace) {
            return new Result() {
                @Override
                public boolean isFinalized() {
                    return true;
                }

                @Override
                public AddressSpace getAddressSpace() {
                    return addressSpace;
                }
            };
        }
    }

    protected final String id;

    public AbstractFinalizerController(final String id) {
        this.id = id;
    }

    /**
     * Process the finalizer logic.
     *
     * @param addressSpace The address space to work on.
     * @return The result of the process. Must never return {@code null}.
     */
    protected abstract Result processFinalizer(AddressSpace addressSpace);

    @Override
    public ReconcileResult reconcileAnyState(final AddressSpace addressSpace) throws Exception {

        log.debug("Reconcile finalizer - id: {}, addressSpace: {}/{} -> {} ({})",
                this.id, addressSpace.getMetadata().getNamespace(), addressSpace.getMetadata().getName(), addressSpace.getMetadata().getDeletionTimestamp(),
                addressSpace.getMetadata().getFinalizers());

        // if we are not deleted ...
        if (!Controller.isDeleted(addressSpace)) {
            // ... we ensure we are added to the list of finalizers.
            AddressSpace modified = ensureFinalizer(addressSpace);

            // Persist finalizer state so that we are sure to be cleaned up if it is deleted
            if (!Objects.equals(modified.getMetadata().getFinalizers(), addressSpace.getMetadata().getFinalizers())) {
                return ReconcileResult.createRequeued(modified, true);
            } else {
                return ReconcileResult.create(modified);
            }
        }

        // if we are deleted, set phase to Terminating
        addressSpace.getStatus().setPhase(Phase.Terminating);

        // if we are deleted, and no longer have the finalizer ...
        if (!addressSpace.getMetadata().getFinalizers().contains(this.id)) {
            // ... we have nothing to do.
            return ReconcileResult.create(addressSpace);
        }

        // process the finalizer
        final Result result = processFinalizer(addressSpace);

        // if we finished finalizing ...
        if (result == null || result.isFinalized()) {
            // ... remove ourselves from the list.
            return ReconcileResult.createRequeued(removeFinalizer(result == null ? addressSpace : result.getAddressSpace()), true);
        }

        // we still need to wait and will try again.
        return ReconcileResult.create(result.getAddressSpace());

    }

    /**
     * Remove the finalizer from the list.
     *
     * @param addressSpace The original address space.
     * @return A modified copy of the original, not having the finalizer set.
     */
    protected AddressSpace removeFinalizer(final AddressSpace addressSpace) {

        return new AddressSpaceBuilder(addressSpace)
                .editOrNewMetadata()
                .removeFromFinalizers(this.id)
                .endMetadata()
                .build();

    }

    /**
     * Ensure the finalizer is in the list.
     *
     * @param addressSpace The original address space.
     * @return A modified copy of the original, having the finalizer set.
     */
    protected AddressSpace ensureFinalizer(final AddressSpace addressSpace) {

        // if the finalizer list does contain our id ...
        if (addressSpace.getMetadata().getFinalizers().contains(this.id)) {
            // ... we are done.
            return addressSpace;
        }

        // ... otherwise we add ourselves to the list.
        return new AddressSpaceBuilder(addressSpace)
                .editOrNewMetadata()
                .addNewFinalizer(this.id)
                .endMetadata()
                .build();

    }
}
