/*
 * #%L
 * Wildfly Camel :: Testsuite
 * %%
 * Copyright (C) 2013 - 2014 RedHat
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

package org.wildfly.camel.test.activemq;

import java.net.URI;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.MessageConsumer;
import javax.jms.MessageListener;
import javax.jms.MessageProducer;
import javax.jms.Session;
import javax.jms.TextMessage;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.activemq.ActiveMQConnectionFactory;
import org.apache.activemq.broker.BrokerService;
import org.apache.activemq.broker.TransportConnector;
import org.apache.camel.CamelContext;
import org.apache.camel.PollingConsumer;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.impl.DefaultCamelContext;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.arquillian.test.api.ArquillianResource;
import org.jboss.as.arquillian.api.ServerSetup;
import org.jboss.as.arquillian.api.ServerSetupTask;
import org.jboss.as.arquillian.container.ManagementClient;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.wildfly.extension.camel.CamelAware;

@CamelAware
@RunWith(Arquillian.class)
@ServerSetup({ActiveMQHttpIntegrationTest.ActiveMQHttpSetupTask.class})
public class ActiveMQHttpIntegrationTest {
    
    static Logger LOG = LoggerFactory.getLogger(ActiveMQHttpIntegrationTest.class);
    
    private static final String QUEUE_NAME = "testQueue";
    
    @ArquillianResource
    private InitialContext context;

    static class ActiveMQHttpSetupTask implements ServerSetupTask {

        private BrokerService broker;
        
        @Override
        public void setup(ManagementClient managementClient, String containerId) throws Exception {
            
            broker = new BrokerService();
            broker.setPersistent(false);
            broker.setUseJmx(false);
            
            TransportConnector connector = broker.addConnector("http://localhost:1180");
            URI connectUri = connector.getConnectUri();
            
            LOG.info("TransportConnector: " + connectUri);
            
            broker.start();
            broker.waitUntilStarted();
        }

        @Override
        public void tearDown(ManagementClient managementClient, String containerId) throws Exception {
            if (broker != null) {
                broker.stop();
            }
        }
    }

    @Deployment
    public static JavaArchive createDeployment() {
        return ShrinkWrap.create(JavaArchive.class, "camel-activemq-http-tests.jar");
    }

    @Before
    public void setUp() throws Exception {
        ActiveMQConnectionFactory connFactory = new ActiveMQConnectionFactory("http://localhost:1180");
        context.bind("ActiveMQHttpConnectionFactory", connFactory);
    }

    @After
    public void tearDown() throws Exception {
        context.unbind("ActiveMQHttpConnectionFactory");
    }

    @Test
    public void testSendMessage() throws Exception {
        CamelContext camelctx = new DefaultCamelContext();
        camelctx.addRoutes(new RouteBuilder() {
            @Override
            public void configure() throws Exception {
                fromF("activemq:queue:%s?connectionFactory=java:/ActiveMQHttpConnectionFactory", QUEUE_NAME).
                transform(simple("Hello ${body}")).
                to("seda:end");
            }
        });

        PollingConsumer pollingConsumer = camelctx.getEndpoint("seda:end").createPollingConsumer();
        pollingConsumer.start();

        camelctx.start();
        try {
            ConnectionFactory connectionFactory = lookupConnectionFactory();
            Connection con = connectionFactory.createConnection();
            try {
                sendMessage(con, "Kermit");
                String result = pollingConsumer.receive(3000).getIn().getBody(String.class);
                Assert.assertEquals("Hello Kermit", result);
            } finally {
                con.close();
            }
        } finally {
            camelctx.stop();
        }
    }

    @Test
    public void testReceiveMessage() throws Exception {
        CamelContext camelctx = new DefaultCamelContext();
        camelctx.addRoutes(new RouteBuilder() {
            @Override
            public void configure() throws Exception {
                from("direct:start").
                transform(simple("Hello ${body}")).
                toF("activemq:queue:%s?connectionFactory=java:/ActiveMQHttpConnectionFactory", QUEUE_NAME);
            }
        });

        final StringBuffer result = new StringBuffer();
        final CountDownLatch latch = new CountDownLatch(1);

        camelctx.start();
        try {
            ConnectionFactory connectionFactory = lookupConnectionFactory();
            Connection con = connectionFactory.createConnection();
            try {
                receiveMessage(con, message -> {
                    TextMessage text = (TextMessage) message;
                    try {
                        result.append(text.getText());
                    } catch (JMSException ex) {
                        result.append(ex.getMessage());
                    }
                    latch.countDown();
                });

                ProducerTemplate producer = camelctx.createProducerTemplate();
                producer.asyncSendBody("direct:start", "Kermit");

                Assert.assertTrue(latch.await(10, TimeUnit.SECONDS));
                Assert.assertEquals("Hello Kermit", result.toString());
            } finally {
                con.close();
            }
        } finally {
            camelctx.stop();
        }
    }

    private ConnectionFactory lookupConnectionFactory() throws NamingException {
        return (ConnectionFactory) context.lookup("java:/ActiveMQHttpConnectionFactory");
    }

    private void sendMessage(Connection connection, String message) throws Exception {
        Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
        Destination destination = session.createQueue(QUEUE_NAME);
        MessageProducer producer = session.createProducer(destination);
        TextMessage msg = session.createTextMessage(message);
        producer.send(msg);
        connection.start();
    }

    private void receiveMessage(Connection connection, MessageListener listener) throws Exception {
        Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
        Destination destination = session.createQueue(QUEUE_NAME);
        MessageConsumer consumer = session.createConsumer(destination);
        consumer.setMessageListener(listener);
        connection.start();
    }
}
