#!/bin/bash

checkout_repo() {
  local repo="${1:-}"

  if [ -z "${repo}" ]; then
    echo "Error: No repository defined"
    exit 1
  fi

  echo "Checking out ${repo} ..."
  git clone git@github.com:jboss-fuse/${repo}.git .
  if [ $? != 0 ]; then
    echo "Error: Failed to checkout repository successfully"
    exit 1
  fi

  #
  # Checkout current branch
  # Branch to a new version branch
  #
  git checkout -b "fuse-${PRODUCT_VERSION}-openshift-rhel-8" "origin/${BRANCH}"
  if [ $? != 0 ]; then
    echo "Error: Failed to branch and checkout successfully"
    exit 1
  fi
}

upgrade_docker_versions () {
  local dockerfile="Dockerfile"
  if [ -f ${dockerfile} ]; then
    echo "------"
    echo -n "Upgrading version label in ${dockerfile} ... "
    sed -i 's/version=\".*\"/version=\"'${TAG_VERSION}'\"/' ${dockerfile}
    sed -i "s/fuse-java-openshift-rhel8:.*/fuse-java-openshift-rhel8:${TAG_VERSION}/" ${dockerfile}
    echo "done"
  fi
}

review_commit() {
  local component="${1:-}"
  local dir="${2:-}"

  git add .
  git commit -m "Bumps version to ${PRODUCT_VERSION} and ${TAG_VERSION}"
  if [ $? == 0 ]; then
    #
    # Manually check before pushing
    #
    echo "Please review the latest commit for ${component} at ${dir} before pushing it."
  else
    echo "No commit made for ${component} at ${dir}. Probably already up-to-date?"
  fi
}

while getopts ":b:p:" opt; do
  case "${opt}" in
    b) BRANCH=${OPTARG}
      ;;
    p)
      PRODUCT_VERSION=${OPTARG}
      ;;
    :)
      echo "ERROR: Option -$OPTARG requires an argument"
      exit 1
      ;;
    \?)
      echo "ERROR: Invalid option -$OPTARG"
      exit 1
      ;;
  esac
done
shift $((OPTIND-1))

if [ -z "${BRANCH}" ] || [ -z "${PRODUCT_VERSION}" ]; then
  echo "Error: ${0} -b <branch> -p <product-version>"
  exit 1
fi

. ./bump-version-functions.sh

TAG_VERSION=$(echo "${PRODUCT_VERSION} - 6.0" | bc)

#######################################
#
# Upgrade the fuse-online-operator repository
#
#######################################

FO_OPERATOR="fuse-online-operator"
FO_OPERATOR_DIR=$(mktemp -d -t ${FO_OPERATOR}-XXXXXX)
pushd ${FO_OPERATOR_DIR} > /dev/null
checkout_repo ${FO_OPERATOR}

#
# Upgrade build config file for operator
#
# - build/conf/config.yaml
#    * Version
#    * Image tags
#  - Dockerfile:
#    * Label version
#
op_config="build/conf/config.yaml"
if [ -f ${op_config} ]; then
  echo "------"
  echo -n "Upgrading ${op_config} ... "
  sed -i "s/Version: \".*\"/Version: \"${PRODUCT_VERSION}\"/" ${op_config}
  sed -i 's~fuse7/fuse\(.*\):.*~fuse7/fuse\1:'${TAG_VERSION}'\"~g' ${op_config}
  echo "done"
fi

upgrade_docker_versions

echo "------"
echo "WARNING: The UPGRADE_VERSION variable in the Dockerfile may need updating but this needs checking manually"

review_commit "${FO_OPERATOR}" "${FO_OPERATOR_DIR}"
popd > /dev/null

continue

#######################################
#
# Upgrade the fuse-ignite-upgrade repository
#
#######################################

FI_IGNITE="fuse-ignite-upgrade"
FI_IGNITE_DIR=$(mktemp -d -t ${FI_IGNITE}-XXXXXX)
pushd ${FI_IGNITE_DIR} > /dev/null
checkout_repo ${FI_IGNITE}

#  - Dockerfile
#    * LABEL version
upgrade_docker_versions

review_commit "${FI_IGNITE}" "${FI_IGNITE_DIR}"
popd > /dev/null

continue

#######################################
#
# Upgrade the fuse-ignite-server repository
#
#######################################

FI_IGNITE="fuse-ignite-server"
FI_IGNITE_DIR=$(mktemp -d -t ${FI_IGNITE}-XXXXXX)
pushd ${FI_IGNITE_DIR} > /dev/null
checkout_repo ${FI_IGNITE}

#  - Dockerfile
#    * LABEL version
#    * FROM Tag
upgrade_docker_versions


review_commit "${FI_IGNITE}" "${FI_IGNITE_DIR}"
popd > /dev/null

continue

#######################################
#
# Upgrade the fuse-ignite-meta repository
#
#######################################

FI_IGNITE="fuse-ignite-meta"
FI_IGNITE_DIR=$(mktemp -d -t ${FI_IGNITE}-XXXXXX)
pushd ${FI_IGNITE_DIR} > /dev/null
checkout_repo ${FI_IGNITE}

#  - Dockerfile
#    * LABEL version
#    * FROM Tag
upgrade_docker_versions


review_commit "${FI_IGNITE}" "${FI_IGNITE_DIR}"
popd > /dev/null

continue

#######################################
#
# Upgrade the fuse-ignite-s2i repository
#
#######################################

FI_IGNITE="fuse-ignite-s2i"
FI_IGNITE_DIR=$(mktemp -d -t ${FI_IGNITE}-XXXXXX)
pushd ${FI_IGNITE_DIR} > /dev/null
checkout_repo ${FI_IGNITE}

#  - Dockerfile
#    * LABEL version
#    * FROM Tag
upgrade_docker_versions


review_commit "${FI_IGNITE}" "${FI_IGNITE_DIR}"
popd > /dev/null

continue

#######################################
#
# Upgrade the fuse-ignite-ui repository
#
#######################################

FI_IGNITE="fuse-ignite-ui"
FI_IGNITE_DIR=$(mktemp -d -t ${FI_IGNITE}-XXXXXX)
pushd ${FI_IGNITE_DIR} > /dev/null
checkout_repo ${FI_IGNITE}

#  - Dockerfile
#    * LABEL version
#    * FROM Tag
upgrade_docker_versions


review_commit "${FI_IGNITE}" "${FI_IGNITE_DIR}"
popd > /dev/null

continue
