/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.yesmail.gwt.rolodex.client;

import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.Timer;
import com.google.gwt.user.client.ui.*;

/**
 * A RolodexCard is represented by a single image in the card stack.
 * @see RolodexCardBundle 
 */
public class RolodexCard extends Widget implements SourcesClickEvents {

  /**
   * This guy's job is to animate a card through its transition (slide).
   */
  static class Slider {

    final static int TICK_DURATION = 20;

    final static double HEAD_START = 0.3;

    Timer slideTimer;

    RolodexCard card;

    int totalDistanceToCover;

    int startX;

    int finalX;

    boolean running;

    long slideStart;

    Slider(final int duration) {
      slideTimer = new Timer() {
        public void run() {
          long currentTime = System.currentTimeMillis();
          long spentTime = currentTime - slideStart;

          if (spentTime > duration) {
            stopRunningSlide();
            return;
          }

          double percentComplete = (double) spentTime / (double) duration;
          // this is a smoothing technique so it will look like the cards settle
          // in to place and it helps hide the fact that we don't do a real
          // rotation since the first part goes so fast
          double adjustedPercent = (1 - Math.pow(1 - percentComplete, 3));
          int newLeft =
              (int) (startX + (totalDistanceToCover * adjustedPercent));
          card.setLeft(newLeft);
        }
      };
    }

    void startSlide(final RolodexCard card, final int finalX) {
      if (running) stopRunningSlide();

      this.slideStart = System.currentTimeMillis();
      this.finalX = finalX;
      this.startX = card.getLeftAsInt();
      this.totalDistanceToCover = finalX - startX;
      if (totalDistanceToCover == 0) return;

      this.startX += (totalDistanceToCover) * HEAD_START;
      card.setLeft(startX);
      this.totalDistanceToCover = finalX - startX;

      this.card = card;
      slideTimer.scheduleRepeating(TICK_DURATION);
      running = true;
    }

    void stopRunningSlide() {
      if (!running) return;
      // jump the current slide to the endpoint and stop the timer
      card.setLeft(finalX);
      slideTimer.cancel();
      running = false;
    }
  }

  protected ClickListenerCollection clickListeners;

  protected boolean expanded = false;

  protected AbstractImagePrototype expandedImagePrototype, collapsedLeftImagePrototype, collapsedRightImagePrototype;

  protected int expandedWidth, collapsedWidth, heightOffset;

  protected Image image;

  protected RolodexPanel panel;

  protected Slider primarySlider, secondarySlider;

  /**
   * You don't generally want to create one of these guys but instead grab
   * one from a RolodexCardBundle.
   */
  public RolodexCard(
      AbstractImagePrototype expandedImagePrototype,
      AbstractImagePrototype collapsedLeftImagePrototype,
      AbstractImagePrototype collapsedRightImagePrototype,
      int expandedWidth, int collapsedWidth, int heightOffset) {
    this.expandedImagePrototype = expandedImagePrototype;
    this.collapsedLeftImagePrototype = collapsedLeftImagePrototype;
    this.collapsedRightImagePrototype = collapsedRightImagePrototype;
    this.expandedWidth = expandedWidth;
    this.collapsedWidth = collapsedWidth;
    this.heightOffset = heightOffset;
    this.image = collapsedLeftImagePrototype.createImage();

    setElement(image.getElement());
    DOM.setStyleAttribute(getElement(), "position", "absolute");

    sinkEvents(Event.MOUSEEVENTS);

    setWidth(collapsedWidth);
  }

  public void addClickListener(ClickListener listener) {
    if (clickListeners == null) {
      clickListeners = new ClickListenerCollection();
    }
    clickListeners.add(listener);
  }

  protected void collapseLeft(int left, int zIndex, boolean animate) {
    moveCard(zIndex, collapsedLeftImagePrototype, animate, left, collapsedWidth,
        false);
  }

  protected void collapseRight(int left, int zIndex, boolean animate) {
    moveCard(zIndex, collapsedRightImagePrototype, animate, left,
        collapsedWidth, false);
  }

  protected void expandLeft(int left, int zIndex, boolean animate) {
    if (expanded) return;
    moveCard(zIndex, expandedImagePrototype, animate, left, expandedWidth,
        true);
  }

  protected void expandRight(int left, int zIndex, boolean animate) {
    if (expanded) return;
    if (animate) setLeft(left +
        (panel.collapsedPrecedingEntryWidth - panel.collapsedEntryWidth));
    moveCard(zIndex, expandedImagePrototype, animate, left, expandedWidth,
        true);
  }

  public void fireClickListeners() {
    if (clickListeners != null) {
      clickListeners.fireClick(this);
    }
  }

  public AbstractImagePrototype getCollapsedLeftImagePrototype() {
    return collapsedLeftImagePrototype;
  }

  public AbstractImagePrototype getCollapsedRightImagePrototype() {
    return collapsedRightImagePrototype;
  }

  public int getCollapsedWidth() {
    return collapsedWidth;
  }

  public AbstractImagePrototype getExpandedImagePrototype() {
    return expandedImagePrototype;
  }

  public int getExpandedWidth() {
    return expandedWidth;
  }

  protected int getLeftAsInt() {
    return DOM.getIntStyleAttribute(getElement(), "left");
  }

  public RolodexPanel getPanel() {
    return panel;
  }

  public void onBrowserEvent(Event event) {
    switch (DOM.eventGetType(event)) {
      case Event.ONCLICK:
        if (clickListeners != null) {
          clickListeners.fireClick(this);
        }
        break;
      case Event.ONMOUSEOVER:
        onMouseOver();
        break;
    }
  }

  public void removeClickListener(ClickListener listener) {
    if (clickListeners != null) {
      clickListeners.remove(listener);
    }
  }

  public void setCollapsedLeftImagePrototype(
      AbstractImagePrototype collapsedLeftImagePrototype) {
    this.collapsedLeftImagePrototype = collapsedLeftImagePrototype;
  }

  public void setCollapsedRightImagePrototype(
      AbstractImagePrototype collapsedRightImagePrototype) {
    this.collapsedRightImagePrototype = collapsedRightImagePrototype;
  }

  public void setCollapsedWidth(int collapsedWidth) {
    this.collapsedWidth = collapsedWidth;
  }

  public void setExpandedImagePrototype(
      AbstractImagePrototype expandedImagePrototype) {
    this.expandedImagePrototype = expandedImagePrototype;
  }

  public void setExpandedWidth(int expandedWidth) {
    this.expandedWidth = expandedWidth;
  }

  protected void setImage(AbstractImagePrototype imagePrototype) {
    imagePrototype.applyTo(image);
  }

  public void setLeft(int left) {
    DOM.setStyleAttribute(getElement(), "left", left + "px");
  }

  public void setPanel(RolodexPanel panel) {
    this.panel = panel;
  }

  public void setPrimarySlider(Slider primarySlider) {
    this.primarySlider = primarySlider;
  }

  public void setSecondarySlider(Slider secondarySlider) {
    this.secondarySlider = secondarySlider;
  }

  public void setTop(int top) {
    DOM.setStyleAttribute(getElement(), "top", getAdjustedTop(top) + "px");
  }

  public void setWidth(int newWidth) {
    DOM.setStyleAttribute(getElement(), "width", "" + newWidth);
  }

  public void setZIndex(int zIndex) {
    DOM.setStyleAttribute(getElement(), "zIndex", "" + zIndex);
  }


  protected void onMouseOver() {
    panel.setPreviousCard(panel.getSelectedCard());
    panel.setSelectedCard(this);
    primarySlider.stopRunningSlide();
    panel.placeEntries(true);
  }


  private int getAdjustedTop(int top) {
    return top + heightOffset;
  }

  private void moveCard(
      int zIndex, AbstractImagePrototype imagePrototype, boolean animate,
      int left, int newWidth, boolean expand) {
    // todo: set attribute string would be nice here since there are like 5 seperate set style attribute calls
    setWidth(newWidth);
    setZIndex(zIndex);
    setImage(imagePrototype);
    expanded = expand;

    if (animate) {
      if (expand) {
        primarySlider.startSlide(this, left);
      } else {
        secondarySlider.startSlide(this, left);
      }
    } else {
      setLeft(left);
    }
  }
}
