package com.redhat.installer.asconfiguration.processpanel.postinstallation;

import com.izforge.izpack.LocaleDatabase;
import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.util.AbstractUIProcessHandler;
import com.redhat.installer.logging.InstallationLogger;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.ScopedMock;
import org.powermock.api.mockito.PowerMockito;
import org.powermock.core.classloader.annotations.PrepareForTest;
import org.powermock.modules.junit4.PowerMockRunner;
import org.wildfly.security.auth.realm.FileSystemSecurityRealm;
import org.wildfly.security.auth.server.ModifiableRealmIdentity;
import org.wildfly.security.authz.Attributes;
import org.wildfly.security.credential.Credential;
import org.wildfly.security.credential.PasswordCredential;
import org.wildfly.security.password.interfaces.ClearPassword;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.logging.Logger;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.when;

@RunWith(PowerMockRunner.class)
@PrepareForTest({InstallFilesystemRealm.class, FileSystemSecurityRealm.class})
public class InstallFilesystemRealmTest {

    public static final String PATH = "/any/random/path";
    @Mock
    AbstractUIProcessHandler handlerMock;

    @Mock
    AutomatedInstallData idataMock;

    @Mock
    LocaleDatabase localeDatabaseMock;

    @Mock
    FileSystemSecurityRealm realmMock;

    @Mock
    ModifiableRealmIdentity identityMock;

    @Captor
    ArgumentCaptor<Path> pathCaptor;

    @Captor
    ArgumentCaptor<Principal> principalCaptor;

    @Captor
    ArgumentCaptor<Collection<? extends Credential>> credentialsCaptor;

    @Captor
    ArgumentCaptor<Attributes> attributesCaptor;

    private static final List<ScopedMock> mockedSettings = new ArrayList<>();

    @BeforeClass
    public static void classSetup() {
        mockedSettings.add(mockStatic(AutomatedInstallData.class));
        mockedSettings.add(mockStatic(InstallationLogger.class));
    }

    @AfterClass
    public static void classTearDown() {
        mockedSettings.forEach(ScopedMock::close);
    }

    @Before
    public void setup() throws Exception {
        when(AutomatedInstallData.getInstance()).thenReturn(idataMock);
        when(InstallationLogger.getLogger()).thenReturn(mock(Logger.class));

        PowerMockito.whenNew(FileSystemSecurityRealm.class).withArguments(any()).thenReturn(realmMock);

        when(localeDatabaseMock.getString(anyString())).thenReturn("");
        when(realmMock.getRealmIdentityForUpdate(any(Principal.class))).thenReturn(identityMock);
        when(identityMock.exists()).thenReturn(false);
        idataMock.langpack = localeDatabaseMock;
    }

    @Test
    public void shouldCreateRealm() throws Exception {
        InstallFilesystemRealm install = new InstallFilesystemRealm();
        install.run(handlerMock, new String[] { "path=" + PATH });

        PowerMockito.verifyNew(FileSystemSecurityRealm.class).withArguments(Paths.get(PATH));
    }

    @Test
    public void shouldAddIdentities() throws Exception {
        InstallFilesystemRealm install = new InstallFilesystemRealm();
        String identityName = "identity1";
        String password = "p4ssw0rd";
        install.run(handlerMock, new String[] {
                "path=" + PATH,
                "identity=" + identityName + ":" + password + ":role1,role2,role3"
        });

        Mockito.verify(realmMock).getRealmIdentityForUpdate(principalCaptor.capture());
        assertEquals(identityName, principalCaptor.getValue().getName());

        Mockito.verify(identityMock).setCredentials(credentialsCaptor.capture());
        PasswordCredential passwordCredential = credentialsCaptor.getValue().toArray(new PasswordCredential[]{})[0];
        assertEquals(password, passwordFrom(passwordCredential));

        Mockito.verify(identityMock).setAttributes(attributesCaptor.capture());
        assertTrue(attributesCaptor.getValue().containsKey("role"));
        String[] roles = attributesCaptor.getValue().get("role").toArray(new String[]{});
        for (int i = 0; i < roles.length; i++) {
            assertEquals("role" + (i + 1), roles[i]);
        }
    }

    private String passwordFrom(PasswordCredential passwordCredential) {
        return new String(passwordCredential.getPassword(ClearPassword.class).getPassword());
    }
}