/*
 * IzPack - Copyright 2001-2008 Julien Ponge, All Rights Reserved.
 *
 * http://izpack.org/
 * http://izpack.codehaus.org/
 *
 * Copyright 2005 Marc Eppelmann
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.izforge.izpack.util;

import com.izforge.izpack.installer.AutomatedInstallData;

import java.io.*;
import java.net.URL;
import java.nio.file.InvalidPathException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.regex.Pattern;
import java.util.stream.Collectors;


/**
 * Provides general global file utility methods
 *
 * @author marc.eppelmann
 */
public class FileUtil {
    //~ Constructors ***********************************************************************

    /**
     * Creates a new FileUtil object.
     */
    public FileUtil() {
    }

    //~ Methods ****************************************************************************

    /**
     * Gets the content from a File as StringArray List.
     *
     * @param fileName A file to read from.
     * @return List of individual line of the specified file. List may be empty but not
     * null.
     * @throws IOException
     */
    static ArrayList getFileContent(String fileName)
            throws IOException {
        ArrayList result = new ArrayList();

        File aFile = new File(fileName);

        if (!aFile.isFile()) {
            //throw new IOException( fileName + " is not a regular File" );
            return result; // None
        }

        BufferedReader reader = null;

        try {
            reader = new BufferedReader(new FileReader(aFile));
        } catch (FileNotFoundException e1) {
            // TODO handle Exception
            e1.printStackTrace();

            return result;
        }

        String aLine = null;

        while ((aLine = reader.readLine()) != null) {
            result.add(aLine + "\n");
        }

        reader.close();

        return result;
    }

    /**
     * Searches case sensitively, and returns true if the given SearchString occurs in the
     * first File with the given Filename.
     *
     * @param aFileName     A files name
     * @param aSearchString the string search for
     * @return true if found in the file otherwise false
     */
    static boolean fileContains(String aFileName, String aSearchString) {
        return (fileContains(aFileName, aSearchString, false));
    }

    /**
     * Tests if the given File contains the given Search String
     *
     * @param aFileName             A files name
     * @param aSearchString         the String to search for
     * @param caseInSensitiveSearch If false the Search is casesensitive
     * @return true if found in the file otherwise false
     */
    static boolean fileContains(String aFileName, String aSearchString,
                                boolean caseInSensitiveSearch) {
        boolean result = false;

        String searchString = caseInSensitiveSearch
                ? aSearchString.toLowerCase() : aSearchString;

        ArrayList fileContent = new ArrayList();

        try {
            fileContent = getFileContent(aFileName);
        } catch (IOException e) {
            // TODO handle Exception
            e.printStackTrace();
        }

        for (Object aFileContent : fileContent) {
            String currentline = (String) aFileContent;

            if (caseInSensitiveSearch) {
                currentline = currentline.toLowerCase();
            }

            if (currentline.contains(searchString)) {
                result = true;

                break;
            }
        }

        return result;
    }

    /**
     * Gets file date and time.
     *
     * @param url The URL of the file for which date and time will be returned.
     * @return Returns long value which is the date and time of the file. If any error
     * occures returns -1 (=no file date and time available).
     */
    public static long getFileDateTime(URL url) {
        if (url == null) {
            return -1;
        }

        String fileName = url.getFile();
        if (fileName.charAt(0) == '/' || fileName.charAt(0) == '\\') {
            fileName = fileName.substring(1, fileName.length());
        }

        try {
            File file = new File(fileName);
            // File name must be a file or a directory.
            if (!file.isDirectory() && !file.isFile()) {
                return -1;
            }

            return file.lastModified();
        } catch (java.lang.Exception e) {   // Trap all Exception based exceptions and return -1.
            return -1;
        }
    }

    public static String[] getFileNames(String dirPath) throws Exception {
        return getFileNames(dirPath, null);
    }

    static String[] getFileNames(String dirPath, FilenameFilter fileNameFilter) throws Exception {
        String fileNames[] = null;
        File dir = new File(dirPath);
        if (dir.isDirectory()) {
            if (fileNameFilter != null) {
                fileNames = dir.list(fileNameFilter);
            } else {
                fileNames = dir.list();
            }
        }
        return fileNames;
    }

    /**
     * Test if a path len is valid, intended to be used for validation
     * There is no file system agnostic approach to testing this without creating the file
     * Remove newly created testfiles to avoid conflicts with the installer in later steps
     *
     * @param dirPath
     * @return
     */
    public static boolean isPathValid(String dirPath) throws WarningIzpackPathException, ErrorIzpackPathException {
        AbstractPathValidator validator;

        // TODO: account for filenames within the install so we don't have other problems during installation
        // solution: allow installer spec file to specify value of longest absolute path to go under $INSTALL_PATH so
        // we can subtract that value from the INSTALL_PATH's absolute length to ensure viable installation
        if (OsVersion.IS_WINDOWS) {
            validator = new WindowsPathValidator(dirPath);
        } else {
            validator = new UnixPathValidator(dirPath);
        }

        return validator.isValidPath();
    }

    /**
     * Method that checks directories based upon an offset. For instance, on some windows platforms, the absolute file
     * path cannot exceed 260 characters; If we know the longest path that appears in a pack, we can make sure that the
     * install path is not 260 - longestPath in the packs we care about, ensuring that installation will not fail due to
     * long path lengths.
     *
     * @param dirPath
     * @param absoluteOffset
     * @return
     */
    public static boolean isPathValidOffset(String dirPath, int absoluteOffset) throws WarningIzpackPathException, ErrorIzpackPathException{
        AbstractPathValidator validator;

        if (OsVersion.IS_WINDOWS){
            validator = new WindowsPathValidator(dirPath);
        } else {
            validator = new UnixPathValidator(dirPath);
        }
        validator.setAbsoluteOffset(absoluteOffset);
        return validator.isValidPath();
    }

    static abstract class AbstractPathValidator {
        AutomatedInstallData idata;
        String pathToValidate;
        int maxFileNameLength = 255;
        int softAbsoluteLength;
        int maxAbsoluteLength;
        int absoluteOffset = 0;

        AbstractPathValidator(String path) {
            idata = AutomatedInstallData.getInstance();
            pathToValidate = path;
        }

        boolean isValidPath() throws ErrorIzpackPathException, WarningIzpackPathException {
            return validAbsolutePathLength() && validFileNameLength() && noReservedNames() && noInvalidPathCharacters();
        }

        protected abstract boolean noReservedNames();

        protected abstract String[] getInvalidChars();

        private void setAbsoluteOffset(int offset) {
            absoluteOffset = offset;
        }

        /**
         * Uses values from respective platforms to validate absolute path length
         *
         * @return
         */
        private boolean validAbsolutePathLength() throws ErrorIzpackPathException, WarningIzpackPathException {
            String error = idata.langpack.getString("UserInputPanel.file.invalidfile.message");
            String warningContinue = idata.langpack.getString("installer.warning") + ": %s " + idata.langpack.getString("usermsg.continue")+"?";
            boolean hasOffset = absoluteOffset != 0;
            if (hasOffset){
                if (pathToValidate.length() > maxAbsoluteLength - absoluteOffset){
                    throw new ErrorIzpackPathException(
                            pathToValidate.substring(0, 40) + "...",
                            String.format(error,
                                    String.format(idata.langpack.getString("pathvalidator.precomputed.absolute.length.error"),
                                            maxAbsoluteLength - absoluteOffset)));
                } else if (pathToValidate.length() > softAbsoluteLength - absoluteOffset){
                    throw new WarningIzpackPathException(
                            pathToValidate.substring(0, 40) + "...",
                            String.format(warningContinue,
                                    String.format(idata.langpack.getString("pathvalidator.precomputed.absolute.length.warning"),
                                            softAbsoluteLength - absoluteOffset)));
                }
            } else {
                if (pathToValidate.length() > maxAbsoluteLength){
                    throw new ErrorIzpackPathException(
                            pathToValidate.substring(0, 40) + "...",
                            String.format(error,
                                    String.format(idata.langpack.getString("pathvalidator.absolute.length.error"),
                                            maxAbsoluteLength)));
                } else if (pathToValidate.length() > softAbsoluteLength){
                    throw new WarningIzpackPathException(
                            pathToValidate.substring(0, 40) + "...",
                            String.format(warningContinue,
                                    String.format(idata.langpack.getString("pathvalidator.absolute.length.warning"),
                                            softAbsoluteLength)));
                }
            }
            return true;
        }

        private boolean validFileNameLength() throws ErrorIzpackPathException {
            String[] pathComponents = pathToValidate.split(Pattern.quote(File.separator));
            for (String component : pathComponents) {
                if (component.length() > maxFileNameLength) {
                    throw new ErrorIzpackPathException(
                            pathToValidate.substring(0, 40) + "...",
                            String.format(idata.langpack.getString("UserInputPanel.file.invalidfile.message"),
                                    String.format(idata.langpack.getString("pathvalidator.filename.length.error"),
                                            maxFileNameLength)));
                }
            }
            return true;
        }

        String invalidCharString() {
            return Arrays.stream(getInvalidChars())
                    .filter(c -> !"  ".equals(c))
                    .distinct()
                    .map(c -> "\"" + c +"\"")
                    .collect(Collectors.joining(", "));
        }

        protected abstract boolean noInvalidPathCharacters() throws ErrorIzpackPathException;

    }

    static class WindowsPathValidator extends AbstractPathValidator {
        String[] reservedNames = new String[]{
                "CON",
                "PRN",
                "AUX",
                "NUL",
                "COM1",
                "COM2",
                "COM3",
                "COM4",
                "COM5",
                "COM6",
                "COM7",
                "COM8",
                "COM9",
                "LPT1",
                "LPT2",
                "LPT3",
                "LPT4",
                "LPT5",
                "LPT6",
                "LPT7",
                "LPT8",
                "LPT9"};

        String[] invalidChars = new String[]{
                "<",
                ">",
                ":",
                "\"",
                "/",
                "\\\\",
                "|",
                "?",
                "*",
                "\\ ",
                "  ",
                "%"
        };

        WindowsPathValidator(String path) {
            super(path);
            softAbsoluteLength = 259;
            maxAbsoluteLength = 32760;
        }

        @Override
        protected boolean noReservedNames() {
            for (String name : reservedNames) {
                for (String pathPart : pathToValidate.split(Pattern.quote(File.separator))) {
                    if (pathPart.equalsIgnoreCase(name)) {
                        throw new ErrorIzpackPathException(
                                pathToValidate,
                                String.format(idata.langpack.getString("UserInputPanel.file.invalidfile.message"),
                                        String.format(idata.langpack.getString("pathvalidator.reserved.name.error"),
                                                name)));
                    }
                }
            }
            return true;
        }

        @Override
        protected String[] getInvalidChars() {
            return invalidChars;
        }

        @Override
        protected boolean noInvalidPathCharacters() throws ErrorIzpackPathException {
            for (String character : getInvalidChars()) {
                // use substring(2) to skip the Drive letter designation in windows.
                int index = pathToValidate.substring(2).indexOf(character);
                if (index != -1) {
                    throw new ErrorIzpackPathException(
                            pathToValidate,
                            String.format(idata.langpack.getString("UserInputPanel.file.invalidfile.message"),
                                    String.format(idata.langpack.getString("TargetPanel.invalid"),
                                            invalidCharString())));
                }
            }
            Paths.get(pathToValidate);
            return true;
        }
    }

    static class UnixPathValidator extends AbstractPathValidator {
        String[] invalidChars = {
                "\\0",
                "//",
                ";",
                "\\",
                "\"",
                "  ",
                "?",
                "%",
                ":"
        };

        UnixPathValidator(String path) {
            super(path);
            softAbsoluteLength = 4095;
            maxAbsoluteLength = 4095;
        }

        @Override
        protected boolean noReservedNames() {
            return true;
        }

        @Override
        protected String[] getInvalidChars() {
            return invalidChars;
        }

        @Override
        protected boolean noInvalidPathCharacters() throws ErrorIzpackPathException {
            for (String character : getInvalidChars()) {
                int index = pathToValidate.indexOf(character);
                if (index != -1) {
                    throw new ErrorIzpackPathException(
                            pathToValidate,
                            String.format(idata.langpack.getString("UserInputPanel.file.invalidfile.message"),
                                    String.format(idata.langpack.getString("TargetPanel.invalid"), invalidCharString())));
                }
            }
            Paths.get(pathToValidate);
            return true;
        }
    }
}
