/******************************************************************************* 
 * Copyright (c) 2014 Red Hat, Inc. 
 * Distributed under license by Red Hat, Inc. All rights reserved. 
 * This program is made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, 
 * and is available at http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Red Hat, Inc. - initial API and implementation 
 ******************************************************************************/ 
package org.jboss.tools.jst.web.ui.palette.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.eclipse.jface.resource.ImageDescriptor;
import org.jboss.tools.jst.web.kb.taglib.IHTMLLibraryVersion;
import org.jboss.tools.jst.web.ui.palette.internal.html.ILibraryPaletteItem;
import org.jboss.tools.jst.web.ui.palette.internal.html.IPaletteCategory;
import org.jboss.tools.jst.web.ui.palette.internal.html.IPaletteGroup;
import org.jboss.tools.jst.web.ui.palette.internal.html.IPaletteItem;
import org.jboss.tools.jst.web.ui.palette.internal.html.IPaletteVersionGroup;
import org.jboss.tools.jst.web.ui.palette.internal.html.impl.PaletteModelImpl;

/**
 * This class returns categories, versions, and items of Palette.
 * Each returned item contains category name, version name, item name,
 * and can compute result generated by wizard with default settings.
 * 
 * @author Viacheslav Kabanovich
 *
 */
public class PaletteManager {
	private static PaletteManager instance = new PaletteManager();

	public static PaletteManager getInstance() {
		return instance;
	}
	
	private PaletteModelImpl model = new PaletteModelImpl();

	private PaletteManager() {}

	/**
	 * Returns array of category names available in Palette.
	 * Currently available are
	 * 		JQueryConstants.JQM_CATEGORY
	 * 		HTMLConstants.HTML_CATEGORY
	 * @return
	 */
	public String[] getCategories() {
		return model.getPaletteGroups();
	}

	/**
	 * Returns the default icon (16x16) of the palette group
	 * @param groupName
	 * @return
	 */
	public ImageDescriptor getImageDescriptor(String category) {
		return model.getPaletteGroup(category).getImageDescriptor();
	}

	/**
	 * Returns list of versions supported by Palette for a given category.
	 * Currently available are
	 * 		For JQueryConstants.JQM_CATEGORY versions are listed in JQueryMobileVersion
	 * 		For HTMLConstants.HTML_CATEGORY  versions are listed in HTMLVersion
	 * @param groupName
	 * @return
	 */
	public IHTMLLibraryVersion[] getVersions(String groupName) {
		return model.getPaletteGroup(groupName).getVersions();
	}

	/**
	 * Returns runnable objects for palette items.
	 * Not supposed to return Library Palette Items
	 * Available versions
	 * 		For JQueryConstants.JQM_CATEGORY versions are listed in JQueryMobileVersion
	 * 		For HTMLConstants.HTML_CATEGORY  versions are listed in HTMLVersion
	 * @param groupName
	 * @param version
	 * @return
	 */
	public Collection<RunnablePaletteItem> getItems(String groupName, IHTMLLibraryVersion version) {
		List<RunnablePaletteItem> list = new ArrayList<RunnablePaletteItem>();
		IPaletteGroup group = model.getPaletteGroup(groupName);
		for(IPaletteVersionGroup versionGroup : group.getPaletteVersionGroups()){
			if(version.equals(versionGroup.getVersion())){
				for(IPaletteCategory category : versionGroup.getCategories()){
					for(IPaletteItem item : category.getItems()){
						if(!(item instanceof ILibraryPaletteItem)){
							list.add(new RunnablePaletteItem(item));
						}
					}
				}
			}
		}
		return list;
	}
}
