"use strict";
/*********************************************************************
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCommandSuccessMessage = exports.initializeContext = exports.sleep = exports.getImageTag = exports.isStableVersion = exports.base64Decode = exports.generatePassword = exports.isOpenshiftPlatformFamily = exports.isKubernetesPlatformFamily = exports.getClusterClientCommand = exports.OPENSHIFT_CLI = exports.KUBERNETES_CLI = void 0;
const commandExists = require("command-exists");
const constants_1 = require("./constants");
exports.KUBERNETES_CLI = 'kubectl';
exports.OPENSHIFT_CLI = 'oc';
function getClusterClientCommand() {
    const clusterClients = [exports.KUBERNETES_CLI, exports.OPENSHIFT_CLI];
    for (const command of clusterClients) {
        if (commandExists.sync(command)) {
            return command;
        }
    }
    throw new Error('No cluster CLI client is installed.');
}
exports.getClusterClientCommand = getClusterClientCommand;
function isKubernetesPlatformFamily(platform) {
    return platform === 'k8s' || platform === 'minikube' || platform === 'microk8s';
}
exports.isKubernetesPlatformFamily = isKubernetesPlatformFamily;
function isOpenshiftPlatformFamily(platform) {
    return platform === 'openshift' || platform === 'minishift' || platform === 'crc';
}
exports.isOpenshiftPlatformFamily = isOpenshiftPlatformFamily;
function generatePassword(passwodLength, charactersSet = '') {
    let dictionary;
    if (!charactersSet) {
        const ZERO_CHAR_CODE = 48;
        const NINE_CHAR_CODE = 57;
        const A_CHAR_CODE = 65;
        const Z_CHAR_CODE = 90;
        const a_CHAR_CODE = 97;
        const z_CHAR_CODE = 122;
        const ranges = [[ZERO_CHAR_CODE, NINE_CHAR_CODE], [A_CHAR_CODE, Z_CHAR_CODE], [a_CHAR_CODE, z_CHAR_CODE]];
        dictionary = [];
        for (let range of ranges) {
            for (let charCode = range[0]; charCode <= range[1]; charCode++) {
                dictionary.push(String.fromCharCode(charCode));
            }
        }
    }
    else {
        dictionary = [...charactersSet];
    }
    let generatedPassword = '';
    for (let i = 0; i < passwodLength; i++) {
        const randomIndex = Math.floor(Math.random() * dictionary.length);
        generatedPassword += dictionary[randomIndex];
    }
    return generatedPassword;
}
exports.generatePassword = generatePassword;
function base64Decode(arg) {
    return Buffer.from(arg, 'base64').toString('ascii');
}
exports.base64Decode = base64Decode;
/**
 * Indicates if stable version of `crwctl` is used.
 */
function isStableVersion(flags) {
    const operatorImage = flags['che-operator-image'] || constants_1.DEFAULT_CHE_OPERATOR_IMAGE;
    const cheVersion = getImageTag(operatorImage);
    return cheVersion !== 'nightly' && cheVersion !== 'latest' && !flags['catalog-source-yaml'] && !flags['catalog-source-name'];
}
exports.isStableVersion = isStableVersion;
/**
 * Returns the tag of the image.
 */
function getImageTag(image) {
    let entries = image.split('@');
    if (entries.length === 2) {
        // digest
        return entries[1];
    }
    entries = image.split(':');
    // tag
    return entries[1];
}
exports.getImageTag = getImageTag;
function sleep(ms) {
    // tslint:disable-next-line no-string-based-set-timeout
    return new Promise(resolve => setTimeout(resolve, ms));
}
exports.sleep = sleep;
/**
 * Initialize command context.
 */
function initializeContext() {
    const ctx = {};
    ctx.highlightedMessages = [];
    ctx.starTime = Date.now();
    return ctx;
}
exports.initializeContext = initializeContext;
/**
 * Returns command success message with execution time.
 */
function getCommandSuccessMessage(command, ctx) {
    if (ctx.starTime) {
        if (!ctx.endTime) {
            ctx.endTime = Date.now();
        }
        const workingTimeInSeconds = Math.round((ctx.endTime - ctx.starTime) / 1000);
        const minutes = Math.floor(workingTimeInSeconds / 60);
        const seconds = (workingTimeInSeconds - minutes * 60) % 60;
        const minutesToStr = minutes.toLocaleString([], { minimumIntegerDigits: 2 });
        const secondsToStr = seconds.toLocaleString([], { minimumIntegerDigits: 2 });
        return `Command ${command.id} has completed successfully in ${minutesToStr}:${secondsToStr}.`;
    }
    return `Command ${command.id} has completed successfully.`;
}
exports.getCommandSuccessMessage = getCommandSuccessMessage;
//# sourceMappingURL=util.js.map