__author__ = 'slaskawi@redhat.com'

import sys
import logging
import fileinput
import re
import os
import fnmatch


class VersionUpdater:
    def __init__(self, configuration):
        self._configuration = configuration

    def update_version(self):
        product_version = self._configuration.get_product_release_version()
        infinispan_full_version = self._configuration.get_full_infinispan_release_version()
        camel_full_version = self._configuration.get_full_camel_release_version()
        quickstarts_full_version = product_version + "." + \
            self._configuration.get_release_version_modifier()

        productized_repo_location = self._configuration.get_productization_repo_location()
        packaging_repo_location = self._configuration.get_packaging_repo_location()
        camel_productized_repo_location = self._configuration.get_camel_productization_repo_location()
        quickstarts_repo_location = self._configuration.get_quickstarts_repo_location()

        self.replace_in_file(productized_repo_location + "/core/src/main/java/org/infinispan/Version.java",
                             "SNAPSHOT = \w+;$",
                             "SNAPSHOT = false;")
        self.replace_in_file(productized_repo_location + "/core/src/main/java/org/infinispan/Version.java",
                             "MAJOR = \"\d\";$",
                             "MAJOR = \"" + self._configuration.get_infinispan_release_version_major() + "\";")
        self.replace_in_file(productized_repo_location + "/core/src/main/java/org/infinispan/Version.java",
                             "MINOR = \"\d\";$",
                             "MINOR = \"" + self._configuration.get_infinispan_release_version_minor() + "\";")
        self.replace_in_file(productized_repo_location + "/core/src/main/java/org/infinispan/Version.java",
                             "MICRO = \"\d\";$",
                             "MICRO = \"" + self._configuration.get_infinispan_release_version_micro() + "\";")
        self.replace_in_file(productized_repo_location + "/core/src/main/java/org/infinispan/Version.java",
                             "MODIFIER = \"\w+\";$",
                             "MODIFIER = \"" + self._configuration.get_release_version_modifier() + "\";")

        self._reversion_project(productized_repo_location, infinispan_full_version)
        self._reversion_project(camel_productized_repo_location, camel_full_version)
        self._reversion_project(quickstarts_repo_location, quickstarts_full_version)

        self.replace_in_file(packaging_repo_location + "/pom.xml",
                             "<version.org.infinispan>.*</version.org.infinispan>",
                             "<version.org.infinispan>" + infinispan_full_version + "</version.org.infinispan>")
        self.replace_in_file(packaging_repo_location + "/pom.xml",
                             r"<version>\d\.\d\.\d</version>",
                             "<version>" + product_version + "</version>")
        self.replace_in_file(camel_productized_repo_location + "/parent/pom.xml",
                             "<infinispan-version>.*</infinispan-version>",
                             "<infinispan-version>" + infinispan_full_version + "</infinispan-version>")
        self.replace_in_file(quickstarts_repo_location + "/pom.xml",
                             "<product.milestone>.*</product.milestone>",
                             "<product.milestone>" + self._configuration.get_version_modifier() +
                             "</product.milestone>")

        for pom in self._find('pom.xml', quickstarts_repo_location):
            self.replace_in_file(pom,
                             "<version.org.infinispan>.*</version.org.infinispan>",
                             "<version.org.infinispan>" + infinispan_full_version +
                             "</version.org.infinispan>")

    def change_versions_back_to_snapshot(self):
        camel_snapshot_version = self._configuration.get_full_camel_snapshot_version()
        camel_productized_repo_location = self._configuration.get_camel_productization_repo_location()
        self._reversion_project(camel_productized_repo_location, camel_snapshot_version)
        
        quickstarts_snapshot_version = self._configuration.get_product_release_version() + \
        "-redhat-SNAPSHOT"
        quickstarts_repo_location = self._configuration.get_quickstarts_repo_location()
        self._reversion_project(quickstarts_repo_location, quickstarts_snapshot_version)
        self.replace_in_file(quickstarts_repo_location + "/pom.xml",
                             "<product.milestone>.*</product.milestone>",
                             "<product.milestone>Final-SNAPSHOT</product.milestone>")        
        

    def replace_in_file(self, file, searchExp, replaceExp, count=1):
        for line in fileinput.input(file, inplace=1):
            if re.search(searchExp, line) and count > 0:
                logging.debug("Replacing: %s to: %s in file: %s", searchExp, replaceExp, file)
                line = re.sub(searchExp, replaceExp, line)
                count -= 1
            sys.stdout.write(line)

    def prepare_clients(self):
        if str(self._configuration.get_skip_clients()) == "False":
            scratch = ""
            if self._configuration.get_dry_run() == "true":
                scratch = "--scratch"

            for client_dir in [self._configuration.get_cpp_repo_location(), self._configuration.get_dotnet_repo_location()]:
                cwd = os.getcwd()

                os.chdir(client_dir)
                logging.debug("Preparing " + client_dir + " using command: ./jdg-release.sh --prepare " +
                               self._configuration.get_version_modifier() + " " + scratch)
                os.system("./jdg-release.sh --prepare " + self._configuration.get_version_modifier() + " " +  scratch)
                os.chdir(cwd)
        else:
            logging.info("C++/C# Client building skipped")

    def _reversion_project(self, repository_location, target_version):
        packaging_repo_location = self._configuration.get_packaging_repo_location()
        os.system(
            "./" + packaging_repo_location + "/bin/reversion.sh --new-revision " + target_version + " --pom-location " + repository_location + "/pom.xml")
            
    def _find(self, pattern, path):
        result = []
        for root, dirs, files in os.walk(path):
            for name in files:
                if fnmatch.fnmatch(name, pattern):
                    result.append(os.path.join(root, name))
        return result




