package org.infinispan.jcache;

import java.lang.annotation.Annotation;
import javax.cache.annotation.CacheKeyGenerator;
import javax.cache.annotation.CacheKeyInvocationContext;
import javax.cache.annotation.GeneratedCacheKey;

/**
 * Custom implementation of {@link javax.cache.annotation.CacheKeyGenerator}. Compatible with methods providing one key parameter.
 *
 * @author Matej Cimbora
 */
public class JCacheCustomKeyGenerator implements CacheKeyGenerator {

   @Override
   public GeneratedCacheKey generateCacheKey(CacheKeyInvocationContext<? extends Annotation> cacheKeyInvocationContext) {
      if (cacheKeyInvocationContext.getKeyParameters().length != 1) {
         throw new IllegalArgumentException("Composed keys are not supported.");
      }
      return new CustomGeneratedCacheKey(cacheKeyInvocationContext.getKeyParameters()[0].getValue());
   }

   public static class CustomGeneratedCacheKey implements GeneratedCacheKey {

      private Object value;

      public CustomGeneratedCacheKey(Object value) {
         if (value == null) {
            throw new IllegalArgumentException("Value needs to be specified.");
         }
         this.value = value;
      }

      @Override
      public boolean equals(Object obj) {
         if (this == obj)
            return true;
         if (obj == null)
            return false;
         if (getClass() != obj.getClass())
            return false;
         CustomGeneratedCacheKey other = (CustomGeneratedCacheKey) obj;
         if (value == null) {
            if (other.value != null)
               return false;
         } else if (!value.equals(other.value))
            return false;
         return true;
      }

      @Override
      public int hashCode() {
         final int prime = 31;
         int result = 1;
         result = prime * result + ((value == null) ? 0 : value.hashCode());
         return result;
      }

      @Override
      public String toString() {
         return "CustomGeneratedCacheKey{" +
               "value=" + value +
               '}';
      }
   }
}
