package org.infinispan.jmx;

import static org.infinispan.factories.KnownComponentNames.TIMEOUT_SCHEDULE_EXECUTOR;
import static org.infinispan.test.TestingUtil.checkMBeanOperationParameterNaming;
import static org.infinispan.test.TestingUtil.existsObject;
import static org.infinispan.test.TestingUtil.extractGlobalComponent;
import static org.infinispan.test.TestingUtil.getCacheManagerObjectName;
import static org.infinispan.test.TestingUtil.getCacheObjectName;
import static org.infinispan.test.TestingUtil.getMethodSpecificJmxDomain;
import static org.testng.AssertJUnit.assertEquals;
import static org.testng.AssertJUnit.assertFalse;
import static org.testng.AssertJUnit.assertNotNull;
import static org.testng.AssertJUnit.assertTrue;

import java.lang.reflect.Method;
import java.util.Arrays;

import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.management.ServiceNotFoundException;

import org.infinispan.commons.jmx.PerThreadMBeanServerLookup;
import org.infinispan.configuration.cache.ConfigurationBuilder;
import org.infinispan.executors.LazyInitializingScheduledExecutorService;
import org.infinispan.manager.CacheContainer;
import org.infinispan.manager.EmbeddedCacheManager;
import org.infinispan.test.Exceptions;
import org.infinispan.test.SingleCacheManagerTest;
import org.infinispan.test.fwk.TestCacheManagerFactory;
import org.testng.annotations.Test;

/**
 * Tests whether the attributes defined by DefaultCacheManager work correct.
 *
 * @author Mircea.Markus@jboss.com
 * @author Galder Zamarreño
 * @since 4.0
 */
@Test(groups = "functional", testName = "jmx.CacheManagerMBeanTest")
public class CacheManagerMBeanTest extends SingleCacheManagerTest {

   public static final String JMX_DOMAIN = CacheManagerMBeanTest.class.getSimpleName();

   private MBeanServer server;
   private ObjectName name;

   @Override
   protected EmbeddedCacheManager createCacheManager() throws Exception {
      cacheManager = TestCacheManagerFactory.createCacheManagerEnforceJmxDomain(JMX_DOMAIN, true, false, true);
      name = getCacheManagerObjectName(JMX_DOMAIN);
      server = PerThreadMBeanServerLookup.getThreadMBeanServer();
      server.invoke(name, "startCache", new Object[]{}, new String[]{});
      return cacheManager;
   }

   public void testJmxOperations() throws Exception {
      assertEquals("1", server.getAttribute(name, "CreatedCacheCount"));
      assertEquals("0", server.getAttribute(name, "DefinedCacheCount"));
      assertEquals("[]", server.getAttribute(name, "DefinedCacheNames"));
      assertEquals("1", server.getAttribute(name, "RunningCacheCount"));

      //now define some new caches
      cacheManager.defineConfiguration("a", new ConfigurationBuilder().build());
      cacheManager.defineConfiguration("b", new ConfigurationBuilder().build());
      cacheManager.defineConfiguration("c", new ConfigurationBuilder().build());
      assertEquals("1", server.getAttribute(name, "CreatedCacheCount"));
      assertEquals("3", server.getAttribute(name, "DefinedCacheCount"));
      assertEquals("1", server.getAttribute(name, "RunningCacheCount"));
      String attribute = (String) server.getAttribute(name, "DefinedCacheConfigurationNames");
      String names[] = attribute.substring(1, attribute.length()-1).split(",");
      assertTrue(Arrays.binarySearch(names, "a") >= 0);
      assertTrue(Arrays.binarySearch(names, "b") >= 0);
      assertTrue(Arrays.binarySearch(names, "c") >= 0);

      //now start some caches
      server.invoke(name, "startCache", new Object[]{"a"}, new String[]{String.class.getName()});
      server.invoke(name, "startCache", new Object[]{"b"}, new String[]{String.class.getName()});
      assertEquals("3", server.getAttribute(name, "CreatedCacheCount"));
      assertEquals("3", server.getAttribute(name, "DefinedCacheCount"));
      assertEquals("3", server.getAttribute(name, "RunningCacheCount"));
      attribute = (String) server.getAttribute(name, "DefinedCacheNames");
      assertTrue(attribute.contains("a("));
      assertTrue(attribute.contains("b("));
      assertTrue(attribute.contains("c("));
   }

   public void testJmxOperationMetadata() throws Exception {
      checkMBeanOperationParameterNaming(name);
   }

   public void testInvokeJmxOperationNotExposed() throws Exception {
      Exceptions.expectException(MBeanException.class, ServiceNotFoundException.class, () -> server.invoke(name, "stop", new Object[]{}, new String[]{}));
   }

   public void testJmxRegistrationAtStartupAndStop(Method m) throws Exception {
      final String otherJmxDomain = getMethodSpecificJmxDomain(m, JMX_DOMAIN);
      CacheContainer otherContainer = TestCacheManagerFactory.createCacheManagerEnforceJmxDomain(otherJmxDomain, true, false, true);
      ObjectName otherName = getCacheManagerObjectName(otherJmxDomain);
      try {
         assertEquals("0", server.getAttribute(otherName, "CreatedCacheCount"));
      } finally {
         otherContainer.stop();
      }

      Exceptions.expectException(InstanceNotFoundException.class, () -> server.getAttribute(otherName, "CreatedCacheCount"));
   }

   public void testCustomCacheManagerName(Method m) throws Exception {
      final String otherJmxDomain = getMethodSpecificJmxDomain(m, JMX_DOMAIN);
      CacheContainer otherContainer = TestCacheManagerFactory.createCacheManagerEnforceJmxDomain(otherJmxDomain, "Hibernate2LC", true, false, true);
      ObjectName otherName = getCacheManagerObjectName(otherJmxDomain, "Hibernate2LC");
      try {
         assertEquals("0", server.getAttribute(otherName, "CreatedCacheCount"));
      } finally {
         otherContainer.stop();
      }
   }

   public void testAddressInformation() throws Exception {
      assertEquals("local", server.getAttribute(name, "NodeAddress"));
      assertEquals("local", server.getAttribute(name, "ClusterMembers"));
      assertEquals("local", server.getAttribute(name, "PhysicalAddresses"));
      assertEquals(1, server.getAttribute(name, "ClusterSize"));
   }

   @Test(dependsOnMethods="testJmxOperations")
   public void testCacheMBeanUnregisterOnRemove() throws Exception {
      cacheManager.defineConfiguration("test", new ConfigurationBuilder().build());
      assertNotNull(cacheManager.getCache("test"));
      ObjectName cacheMBean = getCacheObjectName(JMX_DOMAIN, "test(local)");
      assertTrue(existsObject(cacheMBean));
      cacheManager.administration().removeCache("test");
      assertFalse(existsObject(cacheMBean));
   }

   public void testExecutorMBeans() throws Exception {
      LazyInitializingScheduledExecutorService timeoutExecutor =
         extractGlobalComponent(cacheManager, LazyInitializingScheduledExecutorService.class,
                                TIMEOUT_SCHEDULE_EXECUTOR);
      timeoutExecutor.submit(() -> {});

      ObjectName objectName =
         getCacheManagerObjectName(JMX_DOMAIN, "DefaultCacheManager", TIMEOUT_SCHEDULE_EXECUTOR);
      assertTrue(existsObject(objectName));
      assertEquals(1, server.getAttribute(objectName, "PoolSize"));
      assertEquals(Integer.MAX_VALUE, server.getAttribute(objectName, "MaximumPoolSize"));
   }

}
