"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const client_node_1 = require("@kubernetes/client-node");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const execa = require("execa");
const fs = require("fs");
const Listr = require("listr");
const os = require("os");
const path = require("path");
const che_1 = require("../../api/che");
const kube_1 = require("../../api/kube");
const common_flags_1 = require("../../common-flags");
const che_2 = require("../../tasks/che");
const util_1 = require("../../util");
class Inject extends command_1.Command {
    constructor() {
        super(...arguments);
        // Holds cluster CLI tool name: kubectl or oc
        this.command = util_1.getClusterClientCommand();
    }
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Inject);
            const notifier = require('node-notifier');
            const cheTasks = new che_2.CheTasks(flags);
            const tasks = new Listr([], { renderer: flags['listr-renderer'] });
            tasks.add(cheTasks.verifyCheNamespaceExistsTask(flags, this));
            tasks.add(cheTasks.verifyWorkspaceRunTask(flags, this));
            tasks.add([
                {
                    title: `Verify if container ${flags.container} exists`,
                    enabled: () => flags.container !== undefined,
                    task: (ctx) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        if (!(yield this.containerExists(flags.chenamespace, ctx.pod, flags.container))) {
                            this.error(`The specified container "${flags.container}" doesn't exist. The configuration cannot be injected.`);
                        }
                    })
                },
                {
                    title: 'Injecting configurations',
                    skip: () => {
                        if (!flags.kubeconfig) {
                            return 'Currently, only injecting a kubeconfig is supported. Please, specify flag -k';
                        }
                    },
                    task: () => this.injectKubeconfigTasks(flags)
                },
            ]);
            try {
                yield tasks.run();
            }
            catch (err) {
                this.error(err);
            }
            notifier.notify({
                title: 'crwctl',
                message: `Command ${this.id} has completed.`
            });
        });
    }
    injectKubeconfigTasks(flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const kubeContext = flags['kube-context'];
            let contextToInject;
            const kh = new kube_1.KubeHelper(flags);
            if (kubeContext) {
                contextToInject = kh.getContext(kubeContext);
                if (!contextToInject) {
                    this.error(`Context ${kubeContext} is not found in the source kubeconfig`);
                }
            }
            else {
                const currentContext = yield kh.currentContext();
                contextToInject = kh.getContext(currentContext);
            }
            const che = new che_1.CheHelper(flags);
            const tasks = new Listr({ exitOnError: false, concurrent: true });
            const containers = flags.container ? [flags.container] : yield che.getWorkspacePodContainers(flags.chenamespace, flags.workspace);
            for (const cont of containers) {
                // che-machine-exec container is very limited for a security reason.
                // We cannot copy file into it.
                if (cont.startsWith('che-machine-exec')) {
                    continue;
                }
                tasks.add({
                    title: `injecting kubeconfig into container ${cont}`,
                    task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        try {
                            if (yield this.canInject(flags.chenamespace, ctx.pod, cont)) {
                                yield this.injectKubeconfig(flags.chenamespace, ctx.pod, cont, contextToInject);
                                task.title = `${task.title}...done.`;
                            }
                            else {
                                task.skip('the container doesn\'t support file injection');
                            }
                        }
                        catch (error) {
                            task.skip(error.message);
                        }
                    })
                });
            }
            return tasks;
        });
    }
    /**
     * Tests whether a file can be injected into the specified container.
     */
    canInject(namespace, pod, container) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { exitCode } = yield execa(`${this.command} exec ${pod} -n ${namespace} -c ${container} -- tar --version `, { timeout: 10000, reject: false, shell: true });
            if (exitCode === 0) {
                return true;
            }
            else {
                return false;
            }
        });
    }
    /**
     * Copies the local kubeconfig into the specified container.
     * If returns, it means injection was completed successfully. If throws an error, injection failed
     */
    injectKubeconfig(cheNamespace, workspacePod, container, contextToInject) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { stdout } = yield execa(`${this.command} exec ${workspacePod} -n ${cheNamespace} -c ${container} env | grep ^HOME=`, { timeout: 10000, shell: true });
            let containerHomeDir = stdout.split('=')[1];
            if (!containerHomeDir.endsWith('/')) {
                containerHomeDir += '/';
            }
            if (yield this.fileExists(cheNamespace, workspacePod, container, `${containerHomeDir}.kube/config`)) {
                throw new Error('kubeconfig already exists in the target container');
            }
            yield execa(`${this.command} exec ${workspacePod} -n ${cheNamespace} -c ${container} -- mkdir ${containerHomeDir}.kube -p`, { timeout: 10000, shell: true });
            const kc = new client_node_1.KubeConfig();
            kc.loadFromDefault();
            const kubeConfigPath = path.join(os.tmpdir(), 'che-kubeconfig');
            const cluster = kc.getCluster(contextToInject.cluster);
            if (!cluster) {
                throw new Error(`Context ${contextToInject.name} has no cluster object`);
            }
            const user = kc.getUser(contextToInject.user);
            if (!user) {
                throw new Error(`Context ${contextToInject.name} has no user object`);
            }
            // Despite oc has --kubeconfig flag it actually does nothing, so we need to use --config instead
            const configPathFlag = this.command === util_1.OPENSHIFT_CLI ? '--config' : '--kubeconfig';
            const setClusterArgs = ['config', configPathFlag, kubeConfigPath, 'set-cluster', cluster.name, `--server=${cluster.server}`];
            // Prepare CA certificate file
            if (cluster.caFile) {
                setClusterArgs.push(`--certificate-authority=${cluster.caFile}`);
                setClusterArgs.push('--embed-certs=true');
            }
            else if (cluster.caData) {
                const caFile = path.join(os.tmpdir(), 'cluster-ca-file.pem');
                // Write caData into a file and pass it as the parameter
                fs.writeFileSync(caFile, cluster.caData, 'utf8');
                setClusterArgs.push(`--certificate-authority=${caFile}`);
                setClusterArgs.push('--embed-certs=true');
            }
            yield execa(this.command, setClusterArgs, { timeout: 10000 });
            const setCredentialsArgs = ['config', configPathFlag, kubeConfigPath, 'set-credentials', user.name];
            if (user.certFile) {
                setCredentialsArgs.push(`--client-certificate=${user.certFile}`);
            }
            if (user.keyFile) {
                setCredentialsArgs.push(`--client-key=${user.keyFile}`);
            }
            if (user.certFile || user.keyFile) {
                setCredentialsArgs.push('--embed-certs=true');
            }
            yield execa(this.command, setCredentialsArgs, { timeout: 10000 });
            yield execa(this.command, ['config', configPathFlag, kubeConfigPath, 'set-context', contextToInject.name, `--cluster=${contextToInject.cluster}`, `--user=${contextToInject.user}`, `--namespace=${cheNamespace}`], { timeout: 10000 });
            yield execa(this.command, ['config', configPathFlag, kubeConfigPath, 'use-context', contextToInject.name], { timeout: 10000 });
            yield execa(this.command, ['cp', kubeConfigPath, `${cheNamespace}/${workspacePod}:${containerHomeDir}.kube/config`, '-c', container], { timeout: 10000 });
            return;
        });
    }
    fileExists(namespace, pod, container, file) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { exitCode } = yield execa(`${this.command} exec ${pod} -n ${namespace} -c ${container} -- test -e ${file}`, { timeout: 10000, reject: false, shell: true });
            if (exitCode === 0) {
                return true;
            }
            else {
                return false;
            }
        });
    }
    containerExists(namespace, pod, container) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { stdout } = yield execa(this.command, ['get', 'pods', `${pod}`, '-n', `${namespace}`, '-o', 'jsonpath={.spec.containers[*].name}'], { timeout: 10000 });
            return stdout.split(' ').some(c => c === container);
        });
    }
}
exports.default = Inject;
Inject.description = 'inject configurations and tokens in a workspace';
Inject.flags = {
    help: command_1.flags.help({ char: 'h' }),
    kubeconfig: command_1.flags.boolean({
        char: 'k',
        description: 'Inject the local Kubernetes configuration'
    }),
    workspace: flags_1.string({
        char: 'w',
        description: 'Target workspace. Can be omitted if only one workspace is running'
    }),
    container: flags_1.string({
        char: 'c',
        description: 'Target container. If not specified, configuration files will be injected in all containers of a workspace pod',
        required: false
    }),
    'kube-context': flags_1.string({
        description: 'Kubeconfig context to inject',
        required: false
    }),
    chenamespace: common_flags_1.cheNamespace,
    'listr-renderer': common_flags_1.listrRenderer
};
//# sourceMappingURL=inject.js.map