"use strict";
/**
 * Copyright (c) 2019-2021 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCheApiEndpoint = exports.getLoginData = exports.CheServerLoginManager = exports.isPasswordLoginData = exports.isOcUserTokenLoginData = exports.isRefreshTokenLoginData = void 0;
const tslib_1 = require("tslib");
const axios_1 = require("axios");
const fs = require("fs-extra");
const https = require("https");
const path = require("path");
const querystring = require("querystring");
const common_flags_1 = require("../common-flags");
const util_1 = require("../util");
const che_1 = require("./che");
const che_api_client_1 = require("./che-api-client");
const context_1 = require("./context");
const kube_1 = require("./kube");
function isRefreshTokenLoginData(loginData) {
    return Boolean(loginData.refreshToken);
}
exports.isRefreshTokenLoginData = isRefreshTokenLoginData;
function isOcUserTokenLoginData(loginData) {
    return Boolean(loginData.subjectToken);
}
exports.isOcUserTokenLoginData = isOcUserTokenLoginData;
function isPasswordLoginData(loginData) {
    return Boolean(loginData.password);
}
exports.isPasswordLoginData = isPasswordLoginData;
const REQUEST_TIMEOUT_MS = 10000;
const LOGIN_DATA_FILE_NAME = 'che-login-config.json';
let loginContext;
/**
 * Che server login sessions manager. Singleton.
 * Uses refresh tokens for authentication.
 * Usually, just using of getLoginData function is suitable.
 */
class CheServerLoginManager {
    constructor(dataFilePath) {
        this.dataFilePath = dataFilePath;
        this.loginData = {};
        this.readLoginData();
        this.apiUrl = this.loginData.lastLoginUrl || '';
        this.username = this.loginData.lastUserName || '';
        // Remove outdated login records
        this.removeExpiredLogins();
        // Make axios ignore untrusted certificate error for self-signed certificate case.
        const httpsAgent = new https.Agent({ rejectUnauthorized: false });
        this.axios = axios_1.default.create({
            httpsAgent,
        });
    }
    /**
     * Returns Che server login sessions manager.
     */
    static getInstance() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const ctx = context_1.ChectlContext.get();
            const configDir = ctx[context_1.ChectlContext.CONFIG_DIR];
            if (!fs.existsSync(configDir)) {
                fs.mkdirsSync(configDir);
            }
            const dataFilePath = path.join(configDir, LOGIN_DATA_FILE_NAME);
            if (loginContext && loginContext.dataFilePath === dataFilePath) {
                return loginContext;
            }
            loginContext = new CheServerLoginManager(dataFilePath);
            return loginContext;
        });
    }
    /**
     * Checks whether login credentials exists for given server and user.
     * @param apiUrl API URL of the Che server
     * @param username username
     */
    hasLoginFor(apiUrl, username) {
        apiUrl = che_api_client_1.CheApiClient.normalizeCheApiEndpointUrl(apiUrl);
        if (username) {
            return Boolean(this.getLoginRecord(apiUrl, username));
        }
        else {
            return Boolean(this.loginData.logins[apiUrl]);
        }
    }
    getCurrentLoginInfo() {
        return { cheApiEndpoint: this.apiUrl, username: this.username };
    }
    getCurrentServerApiUrl() {
        return this.apiUrl;
    }
    getAllLogins() {
        this.removeExpiredLogins();
        const allLogins = new Map();
        for (const [apiUrl, serverLogins] of Object.entries(this.loginData.logins)) {
            allLogins.set(apiUrl, [...Object.keys(serverLogins)]);
        }
        return allLogins;
    }
    /**
     * Logins user in specified instance of Che Server.
     * Makes this login data default context.
     * If a context with the same data already exists it will be replaced.
     * If provided data is invalid, exception will be thrown.
     * Returns username of the login.
     * @param apiUrl Che server API URL
     * @param loginRecord user credentials
     */
    setLoginContext(apiUrl, loginRecord) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            apiUrl = che_api_client_1.CheApiClient.normalizeCheApiEndpointUrl(apiUrl);
            const cheKeycloakSettings = yield this.retrieveKeycloakSettings(apiUrl);
            // Check whether provided login credentials valid and get refresh token.
            const keycloakAuthData = yield this.keycloakAuth(apiUrl, loginRecord, cheKeycloakSettings);
            const now = (Date.now() / 1000);
            let refreshTokenExpiresIn = keycloakAuthData.refresh_expires_in ? keycloakAuthData.refresh_expires_in : keycloakAuthData.expires_in;
            if (typeof refreshTokenExpiresIn === 'string') {
                refreshTokenExpiresIn = parseFloat(refreshTokenExpiresIn);
            }
            const refreshTokenLoginRecord = {
                refreshToken: keycloakAuthData.refresh_token,
                expires: now + refreshTokenExpiresIn,
            };
            const username = isPasswordLoginData(loginRecord) ? loginRecord.username :
                yield this.getCurrentUserName(cheKeycloakSettings, keycloakAuthData.access_token);
            // Delete outdated logins as config file will be rewritten
            this.removeExpiredLogins();
            // Credentials are valid, make them current
            this.setCurrentLoginContext(apiUrl, username, refreshTokenLoginRecord);
            // Save changes permanently
            this.saveLoginData();
            return username;
        });
    }
    /**
     * Changes current login.
     */
    switchLoginContext(apiUrl, username) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            // Get rid of outdated credentials before trying to switch current login
            this.removeExpiredLogins();
            apiUrl = che_api_client_1.CheApiClient.normalizeCheApiEndpointUrl(apiUrl);
            const loginRecord = this.getLoginRecord(apiUrl, username);
            if (!loginRecord) {
                throw new Error(`User "${username}" is not logged in on "${apiUrl}" server`);
            }
            // Ensure the server is reachable and credentials are still valid
            const keycloakAuthData = yield this.keycloakAuth(apiUrl, loginRecord);
            // Update refresh token
            loginRecord.refreshToken = keycloakAuthData.refresh_token;
            this.setCurrentLoginContext(apiUrl, username, loginRecord);
            this.saveLoginData();
        });
    }
    /**
     * Logouts user from specified Che server.
     * If no parameters given current login session will be deleted.
     * @param apiUrl Che server API URL
     * @param username username on the given server
     */
    deleteLoginContext(apiUrl, username) {
        if (!this.loginData.logins) {
            return;
        }
        if (!apiUrl) {
            if (!this.apiUrl) {
                // Not logged in
                return;
            }
            // Delete current login context
            return this.deleteLoginContext(this.apiUrl, this.username);
        }
        apiUrl = che_api_client_1.CheApiClient.normalizeCheApiEndpointUrl(apiUrl);
        if (!username) {
            // Delete all logins on the server
            delete this.loginData.logins[apiUrl];
        }
        else {
            // Delete specific login record if any
            const serverLogins = this.loginData.logins[apiUrl];
            if (!serverLogins) {
                // No logins for specified server
                return;
            }
            delete serverLogins[username];
            if (Object.keys(serverLogins).length < 1) {
                // Delete server without logins
                delete this.loginData.logins[apiUrl];
            }
        }
        if (apiUrl === this.apiUrl) {
            // Current login info should be deleted
            this.loginData.lastLoginUrl = this.apiUrl = '';
            this.loginData.lastUserName = this.username = '';
        }
        this.removeExpiredLogins();
        this.saveLoginData();
    }
    readLoginData() {
        if (fs.existsSync(this.dataFilePath)) {
            this.loginData = JSON.parse(fs.readFileSync(this.dataFilePath).toString());
        }
        else {
            this.loginData = {};
        }
        if (!this.loginData.logins) {
            this.loginData.logins = {};
        }
        if (!this.loginData.version) {
            // So far there is only one existing file format
            this.loginData.version = 'v1';
        }
    }
    saveLoginData() {
        this.loginData.lastLoginUrl = this.apiUrl;
        this.loginData.lastUserName = this.username;
        fs.writeFileSync(this.dataFilePath, JSON.stringify(this.loginData));
    }
    /**
     * Searches for login data by API URL and user name.
     * Returns undefined if nothing found by given keys.
     */
    getLoginRecord(apiUrl, username) {
        const serverLogins = this.loginData.logins[apiUrl];
        if (!serverLogins) {
            return;
        }
        return serverLogins[username];
    }
    /**
     * Sets current login credentials by given API URL and username.
     * If loginRecord is provided, then a new credentials are added, replacing existing if any.
     * This method doesn't check credentials validity.
     * Returns true if operation was successful.
     */
    setCurrentLoginContext(apiUrl, username, loginRecord) {
        if (!loginRecord) {
            // Find existing login context and make current
            loginRecord = this.getLoginRecord(apiUrl, username);
            if (!loginRecord) {
                return false;
            }
        }
        else {
            // Set given login config as current
            let serverLogins = this.loginData.logins[apiUrl];
            if (!serverLogins) {
                serverLogins = {};
                this.loginData.logins[apiUrl] = serverLogins;
            }
            serverLogins[username] = loginRecord;
        }
        this.apiUrl = apiUrl;
        this.username = username;
        return true;
    }
    removeExpiredLogins() {
        if (!this.loginData.logins) {
            return;
        }
        const now = Date.now() / 1000;
        for (const [apiUrl, serverLogins] of Object.entries(this.loginData.logins)) {
            for (const [username, loginRecord] of Object.entries(serverLogins)) {
                if (loginRecord.expires <= now) {
                    // Token is expired, delete it
                    delete serverLogins[username];
                }
            }
            if (Object.keys(serverLogins).length < 1) {
                // Delete server without logins
                delete this.loginData.logins[apiUrl];
            }
        }
        // Check if current login is still present
        if (!this.getLoginRecord(this.apiUrl, this.username)) {
            this.loginData.lastLoginUrl = this.apiUrl = '';
            this.loginData.lastUserName = this.username = '';
        }
    }
    retrieveKeycloakSettings(apiUrl) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const cheApi = che_api_client_1.CheApiClient.getInstance(apiUrl);
            const keycloakSettings = yield cheApi.getKeycloakSettings();
            if (!keycloakSettings) {
                // Single user mode
                throw new Error(`Authentication is not supported on the server: "${apiUrl}"`);
            }
            return keycloakSettings;
        });
    }
    /**
     * Returns new Keycloak access token for current login session.
     * Updates session timeout.
     */
    getNewAccessToken() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.apiUrl || !this.username) {
                throw new Error('Login context is not set. Please login first.');
            }
            const loginRecord = this.getLoginRecord(this.apiUrl, this.username);
            if (!loginRecord) {
                // Should never happen
                throw new Error('Invalid login state');
            }
            const keycloakAuthData = yield this.keycloakAuth(this.apiUrl, loginRecord);
            // Update refresh token
            loginRecord.refreshToken = keycloakAuthData.refresh_token;
            this.removeExpiredLogins();
            this.setCurrentLoginContext(this.apiUrl, this.username, loginRecord);
            this.saveLoginData();
            return keycloakAuthData.access_token;
        });
    }
    keycloakAuth(apiUrl, loginRecord, cheKeycloakSettings) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!cheKeycloakSettings) {
                cheKeycloakSettings = yield this.retrieveKeycloakSettings(apiUrl);
            }
            if (isPasswordLoginData(loginRecord)) {
                return this.getKeycloakAuthDataByUserNameAndPassword(cheKeycloakSettings, loginRecord.username, loginRecord.password);
            }
            else {
                if (isRefreshTokenLoginData(loginRecord)) {
                    return this.getKeycloakAuthDataByRefreshToken(cheKeycloakSettings, loginRecord.refreshToken);
                }
                else if (isOcUserTokenLoginData(loginRecord)) {
                    return this.getKeycloakAuthDataByOcToken(cheKeycloakSettings, loginRecord.subjectToken, loginRecord.subjectIssuer);
                }
                else {
                    // Should never happen
                    throw new Error('Token is not provided');
                }
            }
        });
    }
    getKeycloakAuthDataByUserNameAndPassword(cheKeycloakSettings, username, password) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const keycloakTokenUrl = cheKeycloakSettings['che.keycloak.token.endpoint'];
            const data = {
                client_id: cheKeycloakSettings['che.keycloak.client_id'],
                grant_type: 'password',
                username,
                password,
            };
            const headers = {
                'Content-Type': 'application/x-www-form-urlencoded',
            };
            try {
                const response = yield this.axios.post(keycloakTokenUrl, querystring.stringify(data), { headers, timeout: REQUEST_TIMEOUT_MS });
                if (!response || response.status !== 200 || !response.data) {
                    throw new Error('E_BAD_RESP_KEYCLOAK');
                }
                return response.data;
            }
            catch (error) {
                let message = error.message;
                if (error && error.response && error.response.data && error.response.data.error_description) {
                    message = error.response.data.error_description;
                }
                throw new Error(`Failed to get access token from ${keycloakTokenUrl}. Cause: ${message}`);
            }
        });
    }
    getKeycloakAuthDataByRefreshToken(cheKeycloakSettings, refreshToken) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const data = {
                client_id: cheKeycloakSettings['che.keycloak.client_id'],
                grant_type: 'refresh_token',
                refresh_token: refreshToken,
            };
            return this.requestKeycloakAuth(cheKeycloakSettings['che.keycloak.token.endpoint'], data);
        });
    }
    getKeycloakAuthDataByOcToken(cheKeycloakSettings, subjectToken, subjectIssuer) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const data = {
                client_id: cheKeycloakSettings['che.keycloak.client_id'],
                grant_type: 'urn:ietf:params:oauth:grant-type:token-exchange',
                subject_token: subjectToken,
                subject_issuer: subjectIssuer,
            };
            return this.requestKeycloakAuth(cheKeycloakSettings['che.keycloak.token.endpoint'], data);
        });
    }
    requestKeycloakAuth(keycloakTokenUrl, requestData) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const headers = {
                'Content-Type': 'application/x-www-form-urlencoded',
            };
            try {
                const response = yield this.axios.post(keycloakTokenUrl, querystring.stringify(requestData), { headers, timeout: REQUEST_TIMEOUT_MS });
                if (!response || response.status !== 200 || !response.data) {
                    throw new Error('E_BAD_RESP_KEYCLOAK');
                }
                return response.data;
            }
            catch (error) {
                let message = error.message;
                if (error && error.response && error.response.data && error.response.data.error_description) {
                    message = error.response.data.error_description;
                }
                throw new Error(`Failed to get the access token from ${keycloakTokenUrl}. Cause: ${message}`);
            }
        });
    }
    getCurrentUserName(cheKeycloakSettings, accessToken) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = cheKeycloakSettings['che.keycloak.userinfo.endpoint'];
            const headers = {
                'Content-Type': 'application/x-www-form-urlencoded',
                Authorization: `bearer ${accessToken}`,
            };
            try {
                const response = yield this.axios.get(endpoint, { headers, timeout: REQUEST_TIMEOUT_MS });
                if (!response || response.status !== 200 || !response.data) {
                    throw new Error('E_BAD_RESP_KEYCLOAK');
                }
                return response.data.preferred_username;
            }
            catch (error) {
                throw new Error(`Failed to get userdata from ${endpoint}. Cause: ${error.message}`);
            }
        });
    }
}
exports.CheServerLoginManager = CheServerLoginManager;
/**
 * Helper function to get valid credentials. Designed to be used from commands.
 * @param cheApiEndpoint user provided server API URL if any
 * @param accessToken user provied access token if any
 */
function getLoginData(cheApiEndpoint, accessToken, flags) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        if (cheApiEndpoint) {
            // User provides credential manually
            const cheApiClient = che_api_client_1.CheApiClient.getInstance(cheApiEndpoint);
            yield cheApiClient.checkCheApiEndpointUrl();
            if (!accessToken && (yield cheApiClient.isAuthenticationEnabled())) {
                throw new Error(`Parameter "--${common_flags_1.ACCESS_TOKEN_KEY}" is expected.`);
            }
            // Single user mode, proceed without token
        }
        else {
            if (accessToken !== undefined) {
                throw new Error('CodeReady Workspaces server API endpoint is required. Use \'--che-api-endpoint\' to provide it.');
            }
            // Use login manager to get Che API URL and token
            const loginManager = yield CheServerLoginManager.getInstance();
            cheApiEndpoint = loginManager.getCurrentServerApiUrl();
            if (!cheApiEndpoint) {
                cheApiEndpoint = yield getCheApiEndpoint(flags);
                const cheApiClient = che_api_client_1.CheApiClient.getInstance(cheApiEndpoint);
                if (yield cheApiClient.isAuthenticationEnabled()) {
                    throw new Error('There is no active login session. Please use "auth:login" first.');
                }
                else {
                    return { cheApiEndpoint, accessToken };
                }
            }
            accessToken = yield loginManager.getNewAccessToken();
        }
        return { cheApiEndpoint, accessToken };
    });
}
exports.getLoginData = getLoginData;
/**
 * Gets cheApiEndpoint for the given namespace.
 */
function getCheApiEndpoint(flags) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const kube = new kube_1.KubeHelper(flags);
        const namespace = yield util_1.findWorkingNamespace(flags);
        if (!(yield kube.hasReadPermissionsForNamespace(namespace))) {
            throw new Error('Please provide server API URL argument');
        }
        // Retrieve API URL from routes
        const cheHelper = new che_1.CheHelper(flags);
        return (yield cheHelper.cheURL(namespace)) + '/api';
    });
}
exports.getCheApiEndpoint = getCheApiEndpoint;
//# sourceMappingURL=che-login-manager.js.map