"use strict";
/**
 * Copyright (c) 2019-2021 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CommonPlatformTasks = void 0;
const tslib_1 = require("tslib");
const http = require("http");
const https = require("https");
var CommonPlatformTasks;
(function (CommonPlatformTasks) {
    /**
     * Checks whether cluster on which Che is being deployed accessible.
     * Requires flags.domain to be set.
     */
    function getPingClusterTask(flags) {
        return {
            title: 'Check if cluster accessible',
            skip: () => true,
            task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const domain = flags.domain;
                if (!domain) {
                    task.title = `${task.title}... "--domain" flag is not set. Cannot check cluster availability.'`;
                    return;
                }
                if (!(yield checkHttpServer(domain, 80)) && !(yield checkHttpsServer(domain, 443))) {
                    throw new Error(`Cannot reach cluster at "${domain}". To skip this check add "--skip-cluster-availability-check" flag.`);
                }
                task.title = `${task.title}... ok`;
            }),
        };
    }
    CommonPlatformTasks.getPingClusterTask = getPingClusterTask;
    /**
     * Sends request to given server to check if it is accessible.
     */
    function checkHttpServer(host, port) {
        return new Promise(resolve => {
            http.get({ host, port }, response => {
                // It is ok to have 404, but not 5xx
                if (response.statusCode && response.statusCode / 100 < 5) {
                    resolve(true);
                }
                resolve(false);
            }).on('error', () => {
                resolve(false);
            });
        });
    }
    function checkHttpsServer(host, port) {
        return new Promise(resolve => {
            https.get({ host, port }, response => {
                // It is ok to have 404, but not 5xx
                if (response.statusCode && response.statusCode / 100 < 5) {
                    resolve(true);
                }
                resolve(false);
            }).on('error', (err) => {
                if (err.code === 'UNABLE_TO_VERIFY_LEAF_SIGNATURE') {
                    // Connection is estabilished but the server has self-signed certificate, it's ok.
                    resolve(true);
                }
                resolve(false);
            });
        });
    }
})(CommonPlatformTasks = exports.CommonPlatformTasks || (exports.CommonPlatformTasks = {}));
//# sourceMappingURL=common-platform-tasks.js.map