"use strict";
/**
 * Copyright (c) 2019-2021 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.KubeHelper = void 0;
const tslib_1 = require("tslib");
const client_node_1 = require("@kubernetes/client-node");
const axios_1 = require("axios");
const cli_ux_1 = require("cli-ux");
const execa = require("execa");
const fs = require("fs");
const https = require("https");
const lodash_1 = require("lodash");
const net = require("net");
const stream_1 = require("stream");
const constants_1 = require("../constants");
const util_1 = require("../util");
const version_1 = require("./version");
const AWAIT_TIMEOUT_S = 30;
class KubeHelper {
    constructor(flags) {
        this.API_EXTENSIONS_V1BETA1 = 'apiextensions.k8s.io/v1beta1';
        this.podWaitTimeout = (flags && flags.k8spodwaittimeout) ? parseInt(flags.k8spodwaittimeout, 10) : constants_1.DEFAULT_K8S_POD_WAIT_TIMEOUT;
        this.podReadyTimeout = (flags && flags.k8spodreadytimeout) ? parseInt(flags.k8spodreadytimeout, 10) : constants_1.DEFAULT_K8S_POD_WAIT_TIMEOUT;
        this.podDownloadImageTimeout = (flags && flags.k8spoddownloadimagetimeout) ? parseInt(flags.k8spoddownloadimagetimeout, 10) : constants_1.DEFAULT_K8S_POD_WAIT_TIMEOUT;
        this.podErrorRecheckTimeout = (flags && flags.spoderrorrechecktimeout) ? parseInt(flags.spoderrorrechecktimeout, 10) : constants_1.DEFAULT_K8S_POD_ERROR_RECHECK_TIMEOUT;
        this.kubeConfig = new client_node_1.KubeConfig();
        this.kubeConfig.loadFromDefault();
    }
    createNamespace(namespaceName, labels) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            const namespaceObject = {
                apiVersion: 'v1',
                kind: 'Namespace',
                metadata: {
                    labels,
                    name: namespaceName,
                },
            };
            try {
                yield k8sCoreApi.createNamespace(namespaceObject);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteAllServices(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sApi.listNamespacedService(namespace);
                if (res && res.response && res.response.statusCode === 200) {
                    const serviceList = res.body;
                    yield serviceList.items.forEach((service) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        try {
                            yield k8sApi.deleteNamespacedService(service.metadata.name, namespace);
                        }
                        catch (error) {
                            if (error.response.statusCode !== 404) {
                                throw error;
                            }
                        }
                    }));
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    applyResource(yamlPath, opts = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const command = `kubectl apply -f ${yamlPath} ${opts}`;
            yield execa(command, { timeout: 30000, shell: true });
        });
    }
    getServicesBySelector(labelSelector = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sCoreApi.listNamespacedService(namespace, 'true', undefined, undefined, undefined, labelSelector);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            throw new Error('ERR_LIST_SERVICES');
        });
    }
    waitForService(selector, namespace = '', intervalMs = 500, timeoutMs = 30000) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                const currentServices = yield this.getServicesBySelector(selector, namespace);
                if (currentServices && currentServices.items.length > 0) {
                    return;
                }
                yield cli_ux_1.cli.wait(intervalMs);
            }
            throw new Error(`ERR_TIMEOUT: Timeout set to waiting for service ${timeoutMs}`);
        });
    }
    serviceAccountExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                const { body } = yield k8sApi.readNamespacedServiceAccount(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    createServiceAccount(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            const sa = new client_node_1.V1ServiceAccount();
            sa.metadata = new client_node_1.V1ObjectMeta();
            sa.metadata.name = name;
            sa.metadata.namespace = namespace;
            try {
                return yield k8sCoreApi.createNamespacedServiceAccount(namespace, sa);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    waitServiceAccount(name, namespace, timeout = AWAIT_TIMEOUT_S) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                // Set up watcher
                const watcher = new client_node_1.Watch(this.kubeConfig);
                const request = yield watcher
                    .watch(`/api/v1/namespaces/${namespace}/serviceaccounts`, {}, (_phase, obj) => {
                    const serviceAccount = obj;
                    // Filter other service accounts in the given namespace
                    if (serviceAccount && serviceAccount.metadata && serviceAccount.metadata.name === name) {
                        // The service account is present, stop watching
                        if (request) {
                            request.abort();
                        }
                        // Release awaiter
                        resolve();
                    }
                }, error => {
                    if (error) {
                        reject(error);
                    }
                });
                // Automatically stop watching after timeout
                const timeoutHandler = setTimeout(() => {
                    request.abort();
                    reject(`Timeout reached while waiting for "${name}" service account.`);
                }, timeout * 1000);
                // Request service account, for case if it is already exist
                const serviceAccount = yield this.getSecret(name, namespace);
                if (serviceAccount) {
                    // Stop watching
                    request.abort();
                    clearTimeout(timeoutHandler);
                    // Relese awaiter
                    resolve();
                }
            }));
        });
    }
    deleteServiceAccount(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                yield k8sCoreApi.deleteNamespacedServiceAccount(name, namespace);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    createServiceAccountFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlServiceAccount = this.safeLoadFromYamlFile(filePath);
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                return yield k8sCoreApi.createNamespacedServiceAccount(namespace, yamlServiceAccount);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceServiceAccountFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlServiceAccount = this.safeLoadFromYamlFile(filePath);
            if (!yamlServiceAccount || !yamlServiceAccount.metadata || !yamlServiceAccount.metadata.name) {
                throw new Error(`Service account read from ${filePath} must have name specified.`);
            }
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                return yield k8sCoreApi.replaceNamespacedServiceAccount(yamlServiceAccount.metadata.name, namespace, yamlServiceAccount);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    roleExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const { body } = yield k8sRbacAuthApi.readNamespacedRole(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    clusterRoleExist(name = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const { body } = yield k8sRbacAuthApi.readClusterRole(name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    getClusterRole(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const { body } = yield k8sRbacAuthApi.readClusterRole(name);
                return body;
            }
            catch (_a) {
                return;
            }
        });
    }
    getRole(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const res = yield k8sRbacAuthApi.readNamespacedRole(name, namespace);
                return res.body;
            }
            catch (e) {
                if (e.statusCode === 404) {
                    return;
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    listRoles(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const res = yield k8sRbacAuthApi.listNamespacedRole(namespace);
                return res.body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createRoleFrom(yamlRole, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const res = yield k8sRbacAuthApi.createNamespacedRole(namespace, yamlRole);
                return res.response.statusCode;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createRoleFromFile(filePath, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlRole = this.safeLoadFromYamlFile(filePath);
            return this.createRoleFrom(yamlRole, namespace);
        });
    }
    replaceRoleFrom(yamlRole, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            if (!yamlRole.metadata || !yamlRole.metadata.name) {
                throw new Error('Role object requires name');
            }
            try {
                const res = yield k8sRbacAuthApi.replaceNamespacedRole(yamlRole.metadata.name, namespace, yamlRole);
                return res.response.statusCode;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceRoleFromFile(filePath, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlRole = this.safeLoadFromYamlFile(filePath);
            return this.replaceRoleFrom(yamlRole, namespace);
        });
    }
    listClusterRoles() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const res = yield k8sRbacAuthApi.listClusterRole();
                return res.body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createClusterRoleFrom(yamlClusterRole, clusterRoleName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            if (!yamlClusterRole.metadata) {
                yamlClusterRole.metadata = {};
            }
            if (clusterRoleName) {
                yamlClusterRole.metadata.name = clusterRoleName;
            }
            else if (!yamlClusterRole.metadata.name) {
                throw new Error('Role name is not specified');
            }
            try {
                const res = yield k8sRbacAuthApi.createClusterRole(yamlClusterRole);
                return res.response.statusCode;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createClusterRoleFromFile(filePath, clusterRoleName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlClusterRole = this.safeLoadFromYamlFile(filePath);
            return this.createClusterRoleFrom(yamlClusterRole, clusterRoleName);
        });
    }
    replaceClusterRoleFrom(yamlClusterRole, clusterRoleName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            if (!yamlClusterRole.metadata) {
                yamlClusterRole.metadata = {};
            }
            if (clusterRoleName) {
                yamlClusterRole.metadata.name = clusterRoleName;
            }
            else if (!yamlClusterRole.metadata.name) {
                throw new Error('Role name is not specified');
            }
            try {
                const res = yield k8sRbacAuthApi.replaceClusterRole(yamlClusterRole.metadata.name, yamlClusterRole);
                return res.response.statusCode;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceClusterRoleFromFile(filePath, clusterRoleName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlClusterRole = this.safeLoadFromYamlFile(filePath);
            return this.replaceClusterRoleFrom(yamlClusterRole, clusterRoleName);
        });
    }
    addClusterRoleRule(name, apiGroups, resources, verbs) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            const clusterRole = yield this.getClusterRole(name);
            if (clusterRole) {
                // Clean up metadata, otherwise replace role call will fail
                clusterRole.metadata = {};
                clusterRole.metadata.name = name;
                // Add new policy
                const additionaRule = new client_node_1.V1PolicyRule();
                additionaRule.apiGroups = apiGroups;
                additionaRule.resources = resources;
                additionaRule.verbs = verbs;
                if (clusterRole.rules) {
                    clusterRole.rules.push(additionaRule);
                }
                try {
                    const { body } = yield k8sRbacAuthApi.replaceClusterRole(name, clusterRole);
                    return body;
                }
                catch (_a) {
                    return;
                }
            }
        });
    }
    deleteRole(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                yield k8sCoreApi.deleteNamespacedRole(name, namespace);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    getPodListByLabel(namespace, labelSelector) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                const { body: podList } = yield k8sCoreApi.listNamespacedPod(namespace, undefined, undefined, undefined, undefined, labelSelector);
                return podList.items;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteClusterRole(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                yield k8sCoreApi.deleteClusterRole(name);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    listRoleBindings(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const res = yield k8sRbacAuthApi.listNamespacedRoleBinding(namespace);
                return res.body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    roleBindingExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                yield k8sRbacAuthApi.readNamespacedRoleBinding(name, namespace);
                return true;
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    return false;
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    isMutatingWebhookConfigurationExists(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAdmissionApi = this.kubeConfig.makeApiClient(client_node_1.AdmissionregistrationV1Api);
            try {
                yield k8sAdmissionApi.readMutatingWebhookConfiguration(name);
                return true;
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    return false;
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getMutatingWebhookConfiguration(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAdmissionApi = this.kubeConfig.makeApiClient(client_node_1.AdmissionregistrationV1Api);
            try {
                const res = yield k8sAdmissionApi.readMutatingWebhookConfiguration(name);
                return res.body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    isValidatingWebhookConfigurationExists(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAdmissionApi = this.kubeConfig.makeApiClient(client_node_1.AdmissionregistrationV1Api);
            try {
                yield k8sAdmissionApi.readValidatingWebhookConfiguration(name);
                return true;
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    return false;
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteValidatingWebhookConfiguration(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAdmissionApi = this.kubeConfig.makeApiClient(client_node_1.AdmissionregistrationV1Api);
            try {
                yield k8sAdmissionApi.deleteValidatingWebhookConfiguration(name);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    deleteMutatingWebhookConfiguration(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAdmissionApi = this.kubeConfig.makeApiClient(client_node_1.AdmissionregistrationV1Api);
            try {
                yield k8sAdmissionApi.deleteMutatingWebhookConfiguration(name);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    listClusterRoleBindings(labelSelector, fieldSelector) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const res = yield k8sRbacAuthApi.listClusterRoleBinding(undefined, undefined, undefined, fieldSelector, labelSelector);
                return res.body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    clusterRoleBindingExist(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const { body } = yield k8sRbacAuthApi.readClusterRoleBinding(name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    createAdminRoleBinding(name = '', serviceAccount = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            const rb = new client_node_1.V1RoleBinding();
            rb.metadata = new client_node_1.V1ObjectMeta();
            rb.metadata.name = name;
            rb.metadata.namespace = namespace;
            rb.roleRef = new client_node_1.V1RoleRef();
            rb.roleRef.kind = 'ClusterRole';
            rb.roleRef.name = 'admin';
            const subject = new client_node_1.V1Subject();
            subject.kind = 'ServiceAccount';
            subject.name = serviceAccount;
            subject.namespace = namespace;
            rb.subjects = [subject];
            try {
                return yield k8sRbacAuthApi.createNamespacedRoleBinding(namespace, rb);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createRoleBindingFrom(yamlRoleBinding, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const response = yield k8sRbacAuthApi.createNamespacedRoleBinding(namespace, yamlRoleBinding);
                return response.body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createRoleBindingFromFile(filePath, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlRoleBinding = this.safeLoadFromYamlFile(filePath);
            return this.createRoleBindingFrom(yamlRoleBinding, namespace);
        });
    }
    replaceRoleBindingFrom(yamlRoleBinding, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!yamlRoleBinding.metadata || !yamlRoleBinding.metadata.name) {
                throw new Error('RoleBinding object requires name');
            }
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const response = yield k8sRbacAuthApi.replaceNamespacedRoleBinding(yamlRoleBinding.metadata.name, namespace, yamlRoleBinding);
                return response.body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceRoleBindingFromFile(filePath, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlRoleBinding = this.safeLoadFromYamlFile(filePath);
            return this.replaceRoleBindingFrom(yamlRoleBinding, namespace);
        });
    }
    createClusterRoleBindingFrom(yamlClusterRoleBinding) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!yamlClusterRoleBinding.metadata || !yamlClusterRoleBinding.metadata.name) {
                throw new Error('ClusterRoleBinding object requires name');
            }
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                return yield k8sRbacAuthApi.createClusterRoleBinding(yamlClusterRoleBinding);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createClusterRoleBinding(name, saName, saNamespace = '', roleName = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const clusterRoleBinding = {
                apiVersion: 'rbac.authorization.k8s.io/v1',
                metadata: {
                    name: `${name}`,
                },
                subjects: [
                    {
                        kind: 'ServiceAccount',
                        name: `${saName}`,
                        namespace: `${saNamespace}`,
                    },
                ],
                roleRef: {
                    kind: 'ClusterRole',
                    name: `${roleName}`,
                    apiGroup: 'rbac.authorization.k8s.io',
                },
            };
            return this.createClusterRoleBindingFrom(clusterRoleBinding);
        });
    }
    replaceClusterRoleBindingFrom(clusterRoleBinding) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!clusterRoleBinding.metadata || !clusterRoleBinding.metadata.name) {
                throw new Error('Cluster Role Binding must have name specified');
            }
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                return yield k8sRbacAuthApi.replaceClusterRoleBinding(clusterRoleBinding.metadata.name, clusterRoleBinding);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceClusterRoleBinding(name, saName, saNamespace = '', roleName = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const clusterRoleBinding = {
                apiVersion: 'rbac.authorization.k8s.io/v1',
                metadata: {
                    name: `${name}`,
                },
                subjects: [
                    {
                        kind: 'ServiceAccount',
                        name: `${saName}`,
                        namespace: `${saNamespace}`,
                    },
                ],
                roleRef: {
                    kind: 'ClusterRole',
                    name: `${roleName}`,
                    apiGroup: 'rbac.authorization.k8s.io',
                },
            };
            return this.replaceClusterRoleBindingFrom(clusterRoleBinding);
        });
    }
    deleteRoleBinding(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                yield k8sRbacAuthApi.deleteNamespacedRoleBinding(name, namespace);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    deleteClusterRoleBinding(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = this.kubeConfig.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                yield k8sRbacAuthApi.deleteClusterRoleBinding(name);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    getConfigMap(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                const { body } = yield k8sCoreApi.readNamespacedConfigMap(name, namespace);
                return this.compare(body, name) && body;
            }
            catch (_a) {
                return;
            }
        });
    }
    getConfigMapValue(name, namespace, key) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                const { body } = yield k8sCoreApi.readNamespacedConfigMap(name, namespace);
                if (body.data) {
                    return body.data[key];
                }
            }
            catch (_a) {
                return;
            }
        });
    }
    createConfigMapFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlConfigMap = this.safeLoadFromYamlFile(filePath);
            return this.createNamespacedConfigMap(namespace, yamlConfigMap);
        });
    }
    createNamespacedConfigMap(namespace, configMap) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                const { body } = yield k8sCoreApi.createNamespacedConfigMap(namespace, configMap);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    patchConfigMap(name, patch, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(PatchedK8sApi);
            try {
                return yield k8sCoreApi.patchNamespacedConfigMap(name, namespace, patch);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteConfigMap(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                yield k8sCoreApi.deleteNamespacedConfigMap(name, namespace);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    replaceNamespacedConfigMap(name, namespace, configMap) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                const { body } = yield k8sCoreApi.replaceNamespacedConfigMap(name, namespace, configMap);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getNamespace(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                const { body } = yield k8sApi.readNamespace(namespace);
                return body;
            }
            catch (_a) {
            }
        });
    }
    hasReadPermissionsForNamespace(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.AuthorizationV1Api);
            const accessReview = new client_node_1.V1SelfSubjectAccessReview();
            accessReview.spec = new client_node_1.V1SelfSubjectAccessReviewSpec();
            accessReview.spec.resourceAttributes = {
                group: '',
                name: 'access-to-che-namespace',
                namespace,
                resource: 'namespaces',
                verb: 'get',
            };
            try {
                const { body } = yield k8sApi.createSelfSubjectAccessReview(accessReview);
                return body.status.allowed;
            }
            catch (error) {
                if (error.response && error.response.body) {
                    if (error.response.body.code === 403) {
                        return false;
                    }
                }
                throw this.wrapK8sClientError(error);
            }
        });
    }
    readNamespacedPod(podName, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sCoreApi.readNamespacedPod(podName, namespace);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (_a) {
                return;
            }
        });
    }
    patchCustomResource(name, namespace, patch, resourceAPIGroup, resourceAPIVersion, resourcePlural) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            // It is required to patch content-type, otherwise request will be rejected with 415 (Unsupported media type) error.
            const requestOptions = {
                headers: {
                    'content-type': 'application/merge-patch+json',
                },
            };
            try {
                const res = yield k8sCoreApi.patchNamespacedCustomObject(resourceAPIGroup, resourceAPIVersion, namespace, resourcePlural, name, patch, undefined, undefined, undefined, requestOptions);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    patchNamespacedPod(name, namespace, patch) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            // It is required to patch content-type, otherwise request will be rejected with 415 (Unsupported media type) error.
            const requestOptions = {
                headers: {
                    'content-type': 'application/strategic-merge-patch+json',
                },
            };
            try {
                const res = yield k8sCoreApi.patchNamespacedPod(name, namespace, patch, undefined, undefined, undefined, undefined, requestOptions);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (_a) {
                return;
            }
        });
    }
    podsExistBySelector(selector, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            let res;
            try {
                res = yield k8sCoreApi.listNamespacedPod(namespace, undefined, undefined, undefined, undefined, selector);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body || !res.body.items) {
                throw new Error(`Get pods by selector "${selector}" returned an invalid response`);
            }
            return (res.body.items.length > 0);
        });
    }
    /**
     * Returns pod waiting state.
     */
    getPodWaitingState(namespace, selector, desiredPhase) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const pods = yield this.getPodListByLabel(namespace, selector);
            if (!pods.length) {
                return;
            }
            for (const pod of pods) {
                if (pod.status && pod.status.phase === desiredPhase && pod.status.containerStatuses) {
                    for (const status of pod.status.containerStatuses) {
                        if (status.state && status.state.waiting && status.state.waiting.message && status.state.waiting.reason) {
                            return status.state.waiting;
                        }
                    }
                }
            }
        });
    }
    /**
     * Returns pod last terminated state.
     */
    getPodLastTerminatedState(namespace, selector) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const pods = yield this.getPodListByLabel(namespace, selector);
            if (!pods.length) {
                return;
            }
            for (const pod of pods) {
                if (pod.status && pod.status.containerStatuses) {
                    for (const status of pod.status.containerStatuses) {
                        if (status.lastState) {
                            return status.lastState.terminated;
                        }
                    }
                }
            }
        });
    }
    getPodCondition(namespace, selector, conditionType) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            let res;
            try {
                res = yield k8sCoreApi.listNamespacedPod(namespace, undefined, undefined, undefined, undefined, selector);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body || !res.body.items) {
                return [];
            }
            const conditions = [];
            for (const pod of res.body.items) {
                if (pod.status && pod.status.conditions) {
                    for (const condition of pod.status.conditions) {
                        if (condition.type === conditionType) {
                            conditions.push(condition);
                        }
                    }
                }
            }
            return conditions;
        });
    }
    getPodReadyConditionStatus(selector, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            let res;
            try {
                res = yield k8sCoreApi.listNamespacedPod(namespace, undefined, undefined, undefined, undefined, selector);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body || !res.body.items) {
                throw new Error(`Get pods by selector "${selector}" returned an invalid response.`);
            }
            if (res.body.items.length < 1) {
                // No pods found by the specified selector. So, it's not ready.
                return 'False';
            }
            if (res.body.items.length > 1) {
                // Several pods found, rolling update?
                return;
            }
            if (!res.body.items[0].status || !res.body.items[0].status.conditions || !(res.body.items[0].status.conditions.length > 0)) {
                return;
            }
            const conditions = res.body.items[0].status.conditions;
            for (const condition of conditions) {
                if (condition.type === 'Ready') {
                    return condition.status;
                }
            }
        });
    }
    waitForPodReady(selector, namespace = '', intervalMs = 500, timeoutMs = this.podReadyTimeout) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                const readyStatus = yield this.getPodReadyConditionStatus(selector, namespace);
                if (readyStatus === 'True') {
                    return;
                }
                yield cli_ux_1.cli.wait(intervalMs);
            }
            throw new Error(`ERR_TIMEOUT: Timeout set to pod ready timeout ${this.podReadyTimeout}`);
        });
    }
    waitUntilPodIsDeleted(selector, namespace = '', intervalMs = 500, timeoutMs = this.podReadyTimeout) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                const pods = yield this.listNamespacedPod(namespace, undefined, selector);
                if (!pods.items.length) {
                    return;
                }
                yield cli_ux_1.cli.wait(intervalMs);
            }
            throw new Error('ERR_TIMEOUT: Waiting until pod is deleted took too long.');
        });
    }
    deletePod(name, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            this.kubeConfig.loadFromDefault();
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                return yield k8sCoreApi.deleteNamespacedPod(name, namespace);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    // make sure that flag is specified for command that it's invoked
    waitLatestReplica(deploymentName, namespace = '', intervalMs = 500, timeoutMs = this.podWaitTimeout) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                const deployment = yield this.getDeployment(deploymentName, namespace);
                if (!deployment) {
                    throw new Error(`Deployment ${namespace}/${deploymentName} is not found.`);
                }
                const deploymentStatus = deployment.status;
                if (!deploymentStatus) {
                    throw new Error(`Deployment ${namespace}/${deploymentName} does not have any status`);
                }
                if (deploymentStatus.unavailableReplicas && deploymentStatus.unavailableReplicas > 0) {
                    yield cli_ux_1.cli.wait(intervalMs);
                }
                else {
                    return;
                }
            }
            throw new Error(`ERR_TIMEOUT: Timeout set to pod wait timeout ${this.podWaitTimeout}`);
        });
    }
    deploymentExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.AppsV1Api);
            try {
                const { body } = yield k8sApi.readNamespacedDeployment(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    isConfigMapExists(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                yield k8sApi.readNamespacedConfigMap(name, namespace);
                return true;
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    return false;
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deploymentReady(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.AppsV1Api);
            try {
                const res = yield k8sApi.readNamespacedDeployment(name, namespace);
                return (res && res.body &&
                    res.body.status && res.body.status.readyReplicas &&
                    res.body.status.readyReplicas > 0);
            }
            catch (_a) {
                return false;
            }
        });
    }
    deploymentStopped(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.AppsV1Api);
            try {
                const res = yield k8sApi.readNamespacedDeployment(name, namespace);
                if (res && res.body && res.body.spec && res.body.spec.replicas) {
                    throw new Error(`Deployment '${name}' without replicas in spec is fetched`);
                }
                return res.body.spec.replicas === 0;
            }
            catch (_a) {
                return false;
            }
        });
    }
    isDeploymentPaused(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.AppsV1Api);
            try {
                const res = yield k8sApi.readNamespacedDeployment(name, namespace);
                if (!res || !res.body || !res.body.spec) {
                    throw new Error('E_BAD_DEPLOY_RESPONSE');
                }
                return res.body.spec.paused || false;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    pauseDeployment(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(PatchedK8sAppsApi);
            try {
                const patch = {
                    spec: {
                        paused: true,
                    },
                };
                yield k8sApi.patchNamespacedDeployment(name, namespace, patch);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    resumeDeployment(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(PatchedK8sAppsApi);
            try {
                const patch = {
                    spec: {
                        paused: false,
                    },
                };
                yield k8sApi.patchNamespacedDeployment(name, namespace, patch);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    scaleDeployment(name = '', namespace = '', replicas) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAppsApi = this.kubeConfig.makeApiClient(PatchedK8sAppsApi);
            const patch = {
                spec: {
                    replicas,
                },
            };
            let res;
            try {
                res = yield k8sAppsApi.patchNamespacedDeploymentScale(name, namespace, patch);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body) {
                throw new Error('Patch deployment scale returned an invalid response');
            }
        });
    }
    createDeployment(name, image, serviceAccount, pullPolicy, configMapEnvSource, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAppsApi = this.kubeConfig.makeApiClient(client_node_1.AppsV1Api);
            const deployment = new client_node_1.V1Deployment();
            deployment.metadata = new client_node_1.V1ObjectMeta();
            deployment.metadata.name = name;
            deployment.metadata.namespace = namespace;
            deployment.spec = new client_node_1.V1DeploymentSpec();
            deployment.spec.selector = new client_node_1.V1LabelSelector();
            deployment.spec.selector.matchLabels = { app: name };
            deployment.spec.template = new client_node_1.V1PodTemplateSpec();
            deployment.spec.template.metadata = new client_node_1.V1ObjectMeta();
            deployment.spec.template.metadata.name = name;
            deployment.spec.template.metadata.labels = { app: name };
            deployment.spec.template.spec = new client_node_1.V1PodSpec();
            deployment.spec.template.spec.serviceAccountName = serviceAccount;
            const opContainer = new client_node_1.V1Container();
            opContainer.name = name;
            opContainer.image = image;
            opContainer.imagePullPolicy = pullPolicy;
            const envFromSource = new client_node_1.V1EnvFromSource();
            envFromSource.configMapRef = new client_node_1.V1ConfigMapEnvSource();
            envFromSource.configMapRef.name = configMapEnvSource;
            opContainer.envFrom = [envFromSource];
            deployment.spec.template.spec.containers = [opContainer];
            try {
                return yield k8sAppsApi.createNamespacedDeployment(namespace, deployment);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createDeploymentFrom(yamlDeployment) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAppsApi = this.kubeConfig.makeApiClient(client_node_1.AppsV1Api);
            try {
                yield k8sAppsApi.createNamespacedDeployment(yamlDeployment.metadata.namespace, yamlDeployment);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createServiceFrom(yamlService, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                return yield k8sApi.createNamespacedService(namespace, yamlService);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceDeploymentFrom(yamlDeployment) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            // updating restartedAt to make sure that rollout will be restarted
            let annotations = yamlDeployment.spec.template.metadata.annotations;
            if (!annotations) {
                annotations = {};
                yamlDeployment.spec.template.metadata.annotations = annotations;
            }
            annotations['kubectl.kubernetes.io/restartedAt'] = new Date().toISOString();
            const k8sAppsApi = this.kubeConfig.makeApiClient(client_node_1.AppsV1Api);
            try {
                yield k8sAppsApi.replaceNamespacedDeployment(yamlDeployment.metadata.name, yamlDeployment.metadata.namespace, yamlDeployment);
            }
            catch (e) {
                if (e.response && e.response.body && e.response.body.message && e.response.body.message.toString().endsWith('field is immutable')) {
                    try {
                        yield k8sAppsApi.deleteNamespacedDeployment(yamlDeployment.metadata.name, yamlDeployment.metadata.namespace);
                        yield k8sAppsApi.createNamespacedDeployment(yamlDeployment.metadata.namespace, yamlDeployment);
                    }
                    catch (e) {
                        throw this.wrapK8sClientError(e);
                    }
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteAllDeployments(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAppsApi = this.kubeConfig.makeApiClient(client_node_1.AppsV1Api);
            try {
                yield k8sAppsApi.deleteCollectionNamespacedDeployment(namespace);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getDeploymentsBySelector(labelSelector = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAppsApi = this.kubeConfig.makeApiClient(client_node_1.AppsV1Api);
            try {
                const res = yield k8sAppsApi.listNamespacedDeployment(namespace, 'true', undefined, undefined, undefined, labelSelector);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            throw new Error('ERR_LIST_NAMESPACES');
        });
    }
    getDeployment(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAppsApi = this.kubeConfig.makeApiClient(client_node_1.AppsV1Api);
            try {
                const res = yield k8sAppsApi.readNamespacedDeployment(name, namespace);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (error) {
                if (error.response && error.response.statusCode === 404) {
                    return;
                }
                throw this.wrapK8sClientError(error);
            }
            throw new Error('ERR_GET_DEPLOYMENT');
        });
    }
    createPod(name, image, serviceAccount, restartPolicy, pullPolicy, configMapEnvSource, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            const pod = new client_node_1.V1Pod();
            pod.metadata = new client_node_1.V1ObjectMeta();
            pod.metadata.name = name;
            pod.metadata.labels = { app: name };
            pod.metadata.namespace = namespace;
            pod.spec = new client_node_1.V1PodSpec();
            pod.spec.restartPolicy = restartPolicy;
            pod.spec.serviceAccountName = serviceAccount;
            const opContainer = new client_node_1.V1Container();
            opContainer.name = name;
            opContainer.image = image;
            opContainer.imagePullPolicy = pullPolicy;
            const envFromSource = new client_node_1.V1EnvFromSource();
            envFromSource.configMapRef = new client_node_1.V1ConfigMapEnvSource();
            envFromSource.configMapRef.name = configMapEnvSource;
            opContainer.envFrom = [envFromSource];
            pod.spec.containers = [opContainer];
            try {
                return yield k8sCoreApi.createNamespacedPod(namespace, pod);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createJob(name, image, serviceAccount, namespace, backoffLimit = 0, restartPolicy = 'Never') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sBatchApi = this.kubeConfig.makeApiClient(client_node_1.BatchV1Api);
            const job = new client_node_1.V1Job();
            job.metadata = new client_node_1.V1ObjectMeta();
            job.metadata.name = name;
            job.metadata.labels = { app: name };
            job.metadata.namespace = namespace;
            job.spec = new client_node_1.V1JobSpec();
            job.spec.ttlSecondsAfterFinished = 10;
            job.spec.backoffLimit = backoffLimit;
            job.spec.template = new client_node_1.V1PodTemplateSpec();
            job.spec.template.spec = new client_node_1.V1PodSpec();
            job.spec.template.spec.serviceAccountName = serviceAccount;
            const jobContainer = new client_node_1.V1Container();
            jobContainer.name = name;
            jobContainer.image = image;
            job.spec.template.spec.restartPolicy = restartPolicy;
            job.spec.template.spec.containers = [jobContainer];
            try {
                return yield k8sBatchApi.createNamespacedJob(namespace, job);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getJob(jobName, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sBatchApi = this.kubeConfig.makeApiClient(client_node_1.BatchV1Api);
            try {
                const result = yield k8sBatchApi.readNamespacedJob(jobName, namespace);
                return result.body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    waitJob(jobName, namespace, timeout = AWAIT_TIMEOUT_S) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                // Set up watcher
                const watcher = new client_node_1.Watch(this.kubeConfig);
                const request = yield watcher
                    .watch(`/apis/batch/v1/namespaces/${namespace}/jobs/`, {}, (_phase, obj) => {
                    const job = obj;
                    // Filter other jobs in the given namespace
                    if (job && job.metadata && job.metadata.name === jobName) {
                        // Check job status
                        if (job.status && job.status.succeeded && job.status.succeeded >= 1) {
                            // Job is finished, stop watching
                            if (request) {
                                request.abort();
                            }
                            // Release awaiter
                            resolve();
                        }
                    }
                }, error => {
                    if (error) {
                        reject(error);
                    }
                });
                // Automatically stop watching after timeout
                const timeoutHandler = setTimeout(() => {
                    request.abort();
                    reject(`Timeout reached while waiting for "${jobName}" job.`);
                }, timeout * 1000);
                // Request job, for case if it is already ready
                const job = yield this.getJob(jobName, namespace);
                if (job.status && job.status.succeeded && job.status.succeeded >= 1) {
                    // Stop watching
                    request.abort();
                    clearTimeout(timeoutHandler);
                    // Relese awaiter
                    resolve();
                }
            }));
        });
    }
    deleteJob(jobName, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sBatchApi = this.kubeConfig.makeApiClient(client_node_1.BatchV1Api);
            try {
                const result = yield k8sBatchApi.deleteNamespacedJob(jobName, namespace);
                return result.body.status === 'Success';
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    compare(body, name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (body && body.metadata && body.metadata.name && body.metadata.name === name) {
                return true;
            }
            else {
                return false;
            }
        });
    }
    ingressExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sExtensionsApi = this.kubeConfig.makeApiClient(client_node_1.ExtensionsV1beta1Api);
            try {
                const { body } = yield k8sExtensionsApi.readNamespacedIngress(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    deleteAllIngresses(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sExtensionsApi = this.kubeConfig.makeApiClient(client_node_1.ExtensionsV1beta1Api);
            try {
                yield k8sExtensionsApi.deleteCollectionNamespacedIngress(namespace);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createCrdFromFile(filePath) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yaml = this.safeLoadFromYamlFile(filePath);
            if (yaml.apiVersion === this.API_EXTENSIONS_V1BETA1) {
                return this.createCrdV1Beta1(yaml);
            }
            return this.createCrdV1(yaml);
        });
    }
    createCrdV1Beta1(yaml) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.ApiextensionsV1beta1Api);
            try {
                yield k8sApi.createCustomResourceDefinition(yaml);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createCrdV1(yaml) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.ApiextensionsV1Api);
            try {
                yield k8sApi.createCustomResourceDefinition(yaml);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceCrdFromFile(filePath, resourceVersion) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yaml = this.safeLoadFromYamlFile(filePath);
            if (!yaml.metadata || !yaml.metadata.name) {
                throw new Error(`Name is not defined in: ${filePath}`);
            }
            yaml.metadata.resourceVersion = resourceVersion;
            if (yaml.apiVersion === this.API_EXTENSIONS_V1BETA1) {
                return this.replaceCrdV1Beta1(yaml);
            }
            return this.replaceCrdV1(yaml);
        });
    }
    replaceCrdV1Beta1(yaml) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.ApiextensionsV1beta1Api);
            try {
                yield k8sApi.replaceCustomResourceDefinition(yaml.metadata.name, yaml);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceCrdV1(yaml) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.ApiextensionsV1Api);
            try {
                yield k8sApi.replaceCustomResourceDefinition(yaml.metadata.name, yaml);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getCrd(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (yield this.IsAPIExtensionSupported('v1')) {
                return this.getCrdV1(name);
            }
            return this.getCrdV1beta1(name);
        });
    }
    getCrdV1(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.ApiextensionsV1Api);
            try {
                const { body } = yield k8sApi.readCustomResourceDefinition(name);
                return body;
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    return;
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getCrdV1beta1(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.ApiextensionsV1beta1Api);
            try {
                const { body } = yield k8sApi.readCustomResourceDefinition(name);
                return body;
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    return;
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getCrdStorageVersion(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const crd = yield this.getCrd(name);
            if (!crd.spec.versions) {
                // Should never happen
                return 'v1';
            }
            const version = crd.spec.versions.find((v) => v.storage);
            return version ? version.name : 'v1';
        });
    }
    deleteCrd(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (yield this.IsAPIExtensionSupported('v1')) {
                return this.deleteCrdV1(name);
            }
            return this.deleteCrdV1Beta1(name);
        });
    }
    deleteCrdV1Beta1(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.ApiextensionsV1beta1Api);
            try {
                yield k8sApi.deleteCustomResourceDefinition(name);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    deleteCrdV1(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.ApiextensionsV1Api);
            try {
                yield k8sApi.deleteCustomResourceDefinition(name);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    createCheCluster(cheClusterCR, flags, ctx, useDefaultCR) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const cheNamespace = flags.chenamespace;
            if (useDefaultCR) {
                // If CheCluster CR is not explicitly provided, then modify the default example CR
                // with values derived from the other parameters
                if (version_1.VersionHelper.isDeployingStableVersion(flags)) {
                    // Use images from operator defaults in case of a stable version
                    cheClusterCR.spec.server.cheImage = '';
                    cheClusterCR.spec.server.cheImageTag = '';
                    cheClusterCR.spec.server.pluginRegistryImage = '';
                    cheClusterCR.spec.server.devfileRegistryImage = '';
                    cheClusterCR.spec.auth.identityProviderImage = '';
                }
                const cheImage = flags.cheimage;
                if (cheImage) {
                    const [image, tag] = util_1.getImageNameAndTag(cheImage);
                    cheClusterCR.spec.server.cheImage = image;
                    cheClusterCR.spec.server.cheImageTag = tag;
                }
                if ((flags.installer === 'olm' && !flags['catalog-source-yaml']) || (flags['catalog-source-yaml'] && flags['olm-channel'] === constants_1.OLM_STABLE_CHANNEL_NAME)) {
                    // use default image tag for `olm` to install stable Che, because we don't have nightly channel for OLM catalog.
                    cheClusterCR.spec.server.cheImageTag = '';
                }
                cheClusterCR.spec.server.cheDebug = flags.debug ? flags.debug.toString() : 'false';
                if (util_1.isKubernetesPlatformFamily(flags.platform) || !cheClusterCR.spec.auth.openShiftoAuth) {
                    cheClusterCR.spec.auth.updateAdminPassword = true;
                }
                if (!cheClusterCR.spec.k8s) {
                    cheClusterCR.spec.k8s = {};
                }
                if (flags.tls) {
                    cheClusterCR.spec.server.tlsSupport = flags.tls;
                    if (!cheClusterCR.spec.k8s.tlsSecretName) {
                        cheClusterCR.spec.k8s.tlsSecretName = 'che-tls';
                    }
                }
                if (flags.domain) {
                    cheClusterCR.spec.k8s.ingressDomain = flags.domain;
                }
                const pluginRegistryUrl = flags['plugin-registry-url'];
                if (pluginRegistryUrl) {
                    cheClusterCR.spec.server.pluginRegistryUrl = pluginRegistryUrl;
                    cheClusterCR.spec.server.externalPluginRegistry = true;
                }
                const devfileRegistryUrl = flags['devfile-registry-url'];
                if (devfileRegistryUrl) {
                    cheClusterCR.spec.server.devfileRegistryUrl = devfileRegistryUrl;
                    cheClusterCR.spec.server.externalDevfileRegistry = true;
                }
                cheClusterCR.spec.storage.postgresPVCStorageClassName = flags['postgres-pvc-storage-class-name'];
                cheClusterCR.spec.storage.workspacePVCStorageClassName = flags['workspace-pvc-storage-class-name'];
                if (flags['workspace-engine'] === 'dev-workspace') {
                    cheClusterCR.spec.devWorkspace.enable = true;
                }
                // Use self-signed TLS certificate by default (for versions before 7.14.3).
                // In modern versions of Che this field is ignored.
                cheClusterCR.spec.server.selfSignedCert = true;
            }
            cheClusterCR.spec.server.cheClusterRoles = ctx.namespaceEditorClusterRoleName;
            // override default values
            if (ctx.crPatch) {
                lodash_1.merge(cheClusterCR, ctx.crPatch);
            }
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.createNamespacedCustomObject('org.eclipse.che', 'v1', cheNamespace, 'checlusters', cheClusterCR);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    patchCheCluster(name, namespace, patch) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
                const { body } = yield customObjectsApi.patchNamespacedCustomObject('org.eclipse.che', 'v1', namespace, 'checlusters', name, patch, undefined, undefined, undefined, { headers: { 'Content-Type': 'application/merge-patch+json' } });
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    /**
     * Returns `checlusters.org.eclipse.che' in the given namespace.
     */
    getCheCluster(cheNamespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return this.getCustomResource(cheNamespace, constants_1.CHE_CLUSTER_API_GROUP, constants_1.CHE_CLUSTER_API_VERSION, constants_1.CHE_CLUSTER_KIND_PLURAL);
        });
    }
    /**
     * Returns custom resource in the given namespace.
     */
    getCustomResource(namespace, resourceAPIGroup, resourceAPIVersion, resourcePlural) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.listNamespacedCustomObject(resourceAPIGroup, resourceAPIVersion, namespace, resourcePlural);
                if (!body.items) {
                    return;
                }
                const crs = body.items;
                if (crs.length === 0) {
                    return;
                }
                else if (crs.length !== 1) {
                    throw new Error(`Too many resources of type ${resourcePlural}.${resourceAPIGroup} found in the namespace '${namespace}'`);
                }
                return crs[0];
            }
            catch (e) {
                if (e.response && e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    /**
     * Deletes `checlusters.org.eclipse.che' resources in the given namespace.
     */
    getAllCheClusters() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return this.getAllCustomResources(constants_1.CHE_CLUSTER_API_GROUP, constants_1.CHE_CLUSTER_API_VERSION, constants_1.CHE_CLUSTER_KIND_PLURAL);
        });
    }
    /**
     * Returns all custom resources
     */
    getAllCustomResources(resourceAPIGroup, resourceAPIVersion, resourcePlural) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.listClusterCustomObject(resourceAPIGroup, resourceAPIVersion, resourcePlural);
                return body.items ? body.items : [];
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    // There is no CRD
                    return [];
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    /**
     * Deletes `checlusters.org.eclipse.che' resources in the given namespace.
     */
    deleteCheCluster(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return this.deleteCustomResource(namespace, constants_1.CHE_CLUSTER_API_GROUP, constants_1.CHE_CLUSTER_API_VERSION, constants_1.CHE_CLUSTER_KIND_PLURAL);
        });
    }
    /**
     * Deletes custom resources in the given namespace.
     */
    deleteCustomResource(namespace, resourceAPIGroup, resourceAPIVersion, resourcePlural) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.listNamespacedCustomObject(resourceAPIGroup, resourceAPIVersion, namespace, resourcePlural);
                if (!body.items) {
                    return;
                }
                const crs = body.items;
                for (const cr of crs) {
                    yield customObjectsApi.deleteNamespacedCustomObject(resourceAPIGroup, resourceAPIVersion, namespace, resourcePlural, cr.metadata.name);
                }
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    // There is no CRD
                    return;
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    recreateCheGroupCr(crYaml, kindPlural) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const namespace = crYaml.metadata.namespace;
            const name = crYaml.metadata.name;
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            // Delete previos CR if any
            try {
                yield customObjectsApi.getNamespacedCustomObject(constants_1.CHE_CLUSTER_API_GROUP, constants_1.CHE_CLUSTER_API_VERSION, namespace, kindPlural, name);
                try {
                    yield customObjectsApi.deleteNamespacedCustomObject(constants_1.CHE_CLUSTER_API_GROUP, constants_1.CHE_CLUSTER_API_VERSION, namespace, kindPlural, name);
                }
                catch (e) {
                    // Failed to delete old CR
                    throw this.wrapK8sClientError(e);
                }
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
                // Old CR doesn't exists, do nothing
            }
            try {
                const res = yield customObjectsApi.createNamespacedCustomObject(constants_1.CHE_CLUSTER_API_GROUP, constants_1.CHE_CLUSTER_API_VERSION, namespace, kindPlural, crYaml);
                return res.body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    recreateBackupCr(namespace, name, backupServerConfigName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const backupCr = {
                apiVersion: `${constants_1.CHE_CLUSTER_API_GROUP}/${constants_1.CHE_CLUSTER_API_VERSION}`,
                kind: 'CheClusterBackup',
                spec: {},
            };
            backupCr.metadata = new client_node_1.V1ObjectMeta();
            backupCr.metadata.name = name;
            backupCr.metadata.namespace = namespace;
            if (backupServerConfigName) {
                backupCr.spec.backupServerConfigRef = backupServerConfigName;
                backupCr.spec.useInternalBackupServer = false;
            }
            else {
                backupCr.spec.useInternalBackupServer = true;
            }
            return this.recreateCheGroupCr(backupCr, constants_1.CHE_CLUSTER_BACKUP_KIND_PLURAL);
        });
    }
    recreateRestoreCr(namespace, name, backupServerConfigName, snapshotId) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const restoreCr = {
                apiVersion: `${constants_1.CHE_CLUSTER_API_GROUP}/${constants_1.CHE_CLUSTER_API_VERSION}`,
                kind: 'CheClusterRestore',
                spec: {},
            };
            restoreCr.metadata = new client_node_1.V1ObjectMeta();
            restoreCr.metadata.name = name;
            restoreCr.metadata.namespace = namespace;
            restoreCr.spec.backupServerConfigRef = backupServerConfigName;
            restoreCr.spec.snapshotId = snapshotId;
            return this.recreateCheGroupCr(restoreCr, constants_1.CHE_CLUSTER_RESTORE_KIND_PLURAL);
        });
    }
    isPreInstalledOLM() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const apiApi = this.kubeConfig.makeApiClient(client_node_1.ApisApi);
            try {
                const { body } = yield apiApi.getAPIVersions();
                const OLMAPIGroup = body.groups.find(apiGroup => apiGroup.name === 'operators.coreos.com');
                return Boolean(OLMAPIGroup);
            }
            catch (_a) {
                return false;
            }
        });
    }
    getUsersNumber() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            let amountOfUsers;
            try {
                const { body } = yield customObjectsApi.listClusterCustomObject('user.openshift.io', 'v1', 'users');
                if (!body.items) {
                    throw new Error('Unable to get list users.');
                }
                amountOfUsers = body.items.length;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            return amountOfUsers;
        });
    }
    getOpenshiftAuthProviders() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const oAuthName = 'cluster';
                const { body } = yield customObjectsApi.getClusterCustomObject('config.openshift.io', 'v1', 'oauths', oAuthName);
                return body.spec.identityProviders;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    operatorSourceExists(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.getNamespacedCustomObject('operators.coreos.com', 'v1', namespace, 'operatorsources', name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    catalogSourceExists(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.getNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'catalogsources', name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    getOAuthClientAuthorizations(clientName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.listClusterCustomObject('oauth.openshift.io', 'v1', 'oauthclientauthorizations');
                if (!body.items) {
                    return [];
                }
                const oauthClientAuthorizations = body.items;
                return oauthClientAuthorizations.filter(o => o.clientName === clientName);
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    // There is no 'oauthclientauthorizations`
                    return [];
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteOAuthClientAuthorizations(oAuthClientAuthorizations) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const filetOauthAuthorizations = oAuthClientAuthorizations.filter((e => e.metadata && e.metadata.name));
                for (const oauthAuthorization of filetOauthAuthorizations) {
                    yield customObjectsApi.deleteClusterCustomObject('oauth.openshift.io', 'v1', 'oauthclientauthorizations', oauthAuthorization.metadata.name);
                }
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    return;
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    consoleLinkExists(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                yield customObjectsApi.getClusterCustomObject('console.openshift.io', 'v1', 'consolelinks', name);
                return true;
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    // There are no consoleLink
                    return false;
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteConsoleLink(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                yield customObjectsApi.deleteClusterCustomObject('console.openshift.io', 'v1', 'consolelinks', name);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    getCatalogSource(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.getNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'catalogsources', name);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    readCatalogSourceFromFile(filePath) {
        const catalogSource = this.safeLoadFromYamlFile(filePath);
        if (!catalogSource.metadata || !catalogSource.metadata.name) {
            throw new Error(`CatalogSource from ${filePath} must have specified metadata and name`);
        }
        return catalogSource;
    }
    createCatalogSource(catalogSource) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const namespace = catalogSource.metadata.namespace;
                const { body } = yield customObjectsApi.createNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'catalogsources', catalogSource);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    waitCatalogSource(namespace, catalogSourceName, timeout = 60) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const watcher = new client_node_1.Watch(this.kubeConfig);
                const request = yield watcher.watch(`/apis/operators.coreos.com/v1alpha1/namespaces/${namespace}/catalogsources`, { fieldSelector: `metadata.name=${catalogSourceName}` }, (_phase, obj) => {
                    resolve(obj);
                }, error => {
                    if (error) {
                        reject(error);
                    }
                });
                setTimeout(() => {
                    request.abort();
                    reject(`Timeout reached while waiting for "${catalogSourceName}" catalog source is created.`);
                }, timeout * 1000);
            }));
        });
    }
    deleteCatalogSource(namespace, catalogSourceName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                yield customObjectsApi.deleteNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'catalogsources', catalogSourceName);
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    return;
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    operatorGroupExists(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.getNamespacedCustomObject('operators.coreos.com', 'v1', namespace, 'operatorgroups', name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    createOperatorGroup(operatorGroupName, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const operatorGroup = {
                apiVersion: 'operators.coreos.com/v1',
                kind: 'OperatorGroup',
                metadata: {
                    name: operatorGroupName,
                    namespace,
                },
                spec: {
                    targetNamespaces: [namespace],
                },
            };
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.createNamespacedCustomObject('operators.coreos.com', 'v1', namespace, 'operatorgroups', operatorGroup);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteOperatorGroup(operatorGroupName, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                yield customObjectsApi.deleteNamespacedCustomObject('operators.coreos.com', 'v1', namespace, 'operatorgroups', operatorGroupName);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    createOperatorSubscription(subscription) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const namespace = subscription.metadata.namespace;
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.createNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'subscriptions', subscription);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getOperatorSubscription(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.getNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'subscriptions', name);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    operatorSubscriptionExists(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.getNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'subscriptions', name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    deleteOperatorSubscription(operatorSubscriptionName, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                yield customObjectsApi.deleteNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'subscriptions', operatorSubscriptionName);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    waitOperatorSubscriptionReadyForApproval(namespace, subscriptionName, timeout = AWAIT_TIMEOUT_S) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const watcher = new client_node_1.Watch(this.kubeConfig);
                const request = yield watcher.watch(`/apis/operators.coreos.com/v1alpha1/namespaces/${namespace}/subscriptions`, { fieldSelector: `metadata.name=${subscriptionName}` }, (_phase, obj) => {
                    const subscription = obj;
                    if (subscription.status && subscription.status.conditions) {
                        for (const condition of subscription.status.conditions) {
                            if (condition.type === 'InstallPlanPending' && condition.status === 'True') {
                                resolve(subscription.status.installplan);
                            }
                        }
                    }
                }, error => {
                    if (error) {
                        reject(error);
                    }
                });
                setTimeout(() => {
                    request.abort();
                    reject(`Timeout reached while waiting for "${subscriptionName}" subscription is ready.`);
                }, timeout * 1000);
            }));
        });
    }
    approveOperatorInstallationPlan(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const patch = {
                    spec: {
                        approved: true,
                    },
                };
                yield customObjectsApi.patchNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'installplans', name, patch, undefined, undefined, undefined, { headers: { 'Content-Type': 'application/merge-patch+json' } });
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    waitUntilOperatorIsInstalled(installPlanName, namespace, timeout = 240) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const watcher = new client_node_1.Watch(this.kubeConfig);
                const request = yield watcher.watch(`/apis/operators.coreos.com/v1alpha1/namespaces/${namespace}/installplans`, { fieldSelector: `metadata.name=${installPlanName}` }, (_phase, obj) => {
                    const installPlan = obj;
                    if (installPlan.status && installPlan.status.phase === 'Failed') {
                        const errorMessage = [];
                        for (const condition of installPlan.status.conditions) {
                            if (!condition.reason) {
                                errorMessage.push(`Reason: ${condition.reason}`);
                                errorMessage.push(!condition.message ? `Message: ${condition.message}` : '');
                            }
                        }
                        reject(errorMessage.join(' '));
                    }
                    if (installPlan.status && installPlan.status.conditions) {
                        for (const condition of installPlan.status.conditions) {
                            if (condition.type === 'Installed' && condition.status === 'True') {
                                resolve(installPlan);
                            }
                        }
                    }
                }, error => {
                    if (error) {
                        reject(error);
                    }
                });
                setTimeout(() => {
                    request.abort();
                    reject(`Timeout reached while waiting for "${installPlanName}" has go status 'Installed'.`);
                }, timeout * 1000);
            }));
        });
    }
    getCSV(csvName, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const csvs = yield this.getClusterServiceVersions(namespace);
            return csvs.items.find(item => item.metadata.name === csvName);
        });
    }
    getClusterServiceVersions(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.listNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'clusterserviceversions');
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    patchClusterServiceVersion(namespace, name, jsonPatch) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            const requestOptions = {
                headers: {
                    'content-type': 'application/json-patch+json',
                },
            };
            try {
                const response = yield customObjectsApi.patchNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'clusterserviceversions', name, jsonPatch, undefined, undefined, undefined, requestOptions);
                return response.body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteClusterServiceVersion(namespace, csvName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                yield customObjectsApi.deleteNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'clusterserviceversions', csvName);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    getPackageManifect(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.getNamespacedCustomObject('packages.operators.coreos.com', 'v1', 'default', 'packagemanifests', name);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteNamespace(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                yield k8sCoreApi.deleteNamespace(namespace);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    /**
     * Returns CRD version of Cert Manager
     */
    getCertManagerK8sApiVersion() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return this.getCrdStorageVersion('certificates.cert-manager.io');
        });
    }
    clusterIssuerExists(name, version) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                // If cluster issuers doesn't exist an exception will be thrown
                yield customObjectsApi.getClusterCustomObject('cert-manager.io', version, 'clusterissuers', name);
                return true;
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    return false;
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    isNamespacedCertificateExists(name, version, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                // If cluster issuers doesn't exist an exception will be thrown
                yield customObjectsApi.getNamespacedCustomObject('cert-manager.io', version, namespace, 'certificates', name);
                return true;
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    return false;
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteNamespacedCertificate(name, version, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                // If cluster certificates doesn't exist an exception will be thrown
                yield customObjectsApi.deleteNamespacedCustomObject('cert-manager.io', version, namespace, 'certificates', name);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    deleteNamespacedIssuer(name, version, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                yield customObjectsApi.deleteNamespacedCustomObject('cert-manager.io', version, namespace, 'issuers', name);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    listClusterIssuers(version, labelSelector) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            let res;
            try {
                res = yield customObjectsApi.listClusterCustomObject('cert-manager.io', version, 'clusterissuers', undefined, undefined, undefined, labelSelector);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body) {
                throw new Error('Unable to get cluster issuers list');
            }
            const clusterIssuersList = res.body;
            return clusterIssuersList.items || [];
        });
    }
    createCheClusterIssuer(cheClusterIssuerYamlPath, version) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            const cheClusterIssuer = this.safeLoadFromYamlFile(cheClusterIssuerYamlPath);
            try {
                yield customObjectsApi.createClusterCustomObject('cert-manager.io', version, 'clusterissuers', cheClusterIssuer);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createCertificateIssuer(cheClusterIssuerYamlPath, version, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            const certificateIssuer = this.safeLoadFromYamlFile(cheClusterIssuerYamlPath);
            try {
                yield customObjectsApi.createNamespacedCustomObject('cert-manager.io', version, namespace, 'issuers', certificateIssuer);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    isCertificateIssuerExists(name, version, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                // If issuers doesn't exist an exception will be thrown
                yield customObjectsApi.getNamespacedCustomObject('cert-manager.io', version, namespace, 'issuers', name);
                return true;
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    return false;
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createCheClusterCertificate(certificate, version) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = this.kubeConfig.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                yield customObjectsApi.createNamespacedCustomObject('cert-manager.io', version, certificate.metadata.namespace, 'certificates', certificate);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    currentContext() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return this.kubeConfig.getCurrentContext();
        });
    }
    getContext(name) {
        return this.kubeConfig.getContextObject(name);
    }
    /**
     * Retrieve the default token from the default serviceAccount.
     */
    getDefaultServiceAccountToken() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            const namespaceName = 'default';
            const saName = 'default';
            let res;
            // now get the matching secrets
            try {
                res = yield k8sCoreApi.listNamespacedSecret(namespaceName);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body) {
                throw new Error('Unable to get default service account');
            }
            const v1SecretList = res.body;
            if (!v1SecretList.items || v1SecretList.items.length === 0) {
                throw new Error(`Unable to get default service account token since there is no secret in '${namespaceName}' namespace`);
            }
            const v1DefaultSATokenSecret = v1SecretList.items.find(secret => secret.metadata.annotations &&
                secret.metadata.annotations['kubernetes.io/service-account.name'] === saName &&
                secret.type === 'kubernetes.io/service-account-token');
            if (!v1DefaultSATokenSecret) {
                throw new Error(`Secret for '${saName}' service account is not found in namespace '${namespaceName}'`);
            }
            return Buffer.from(v1DefaultSATokenSecret.data.token, 'base64').toString();
        });
    }
    checkKubeApi() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const currentCluster = this.kubeConfig.getCurrentCluster();
            if (!currentCluster) {
                throw new Error(`The current context is unknown. It should be set using '${util_1.getClusterClientCommand()} config use-context <CONTEXT_NAME>' or in another way.`);
            }
            try {
                yield this.requestKubeHealthz(currentCluster);
            }
            catch (error) {
                if (error.message && error.message.includes('E_K8S_API_UNAUTHORIZED')) {
                    const token = yield this.getDefaultServiceAccountToken();
                    yield this.requestKubeHealthz(currentCluster, token);
                }
                else {
                    throw error;
                }
            }
        });
    }
    requestKubeHealthz(currentCluster, token) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = `${currentCluster.server}/healthz`;
            try {
                const config = {
                    httpsAgent: new https.Agent({
                        rejectUnauthorized: false,
                        requestCert: true,
                    }),
                    headers: token && { Authorization: 'bearer ' + token },
                };
                const response = yield axios_1.default.get(`${endpoint}`, config);
                if (!response || response.status !== 200 || response.data !== 'ok') {
                    throw new Error('E_BAD_RESP_K8S_API');
                }
            }
            catch (error) {
                if (error.response && error.response.status === 403) {
                    throw new Error(`E_K8S_API_FORBIDDEN - Message: ${error.response.data.message}`);
                }
                if (error.response && error.response.status === 401) {
                    throw new Error(`E_K8S_API_UNAUTHORIZED - Message: ${error.response.data.message}`);
                }
                if (error.response) {
                    // The request was made and the server responded with a status code
                    // that falls out of the range of 2xx
                    throw new Error(`E_K8S_API_UNKNOWN_ERROR - Status: ${error.response.status}`);
                }
                else if (error.request) {
                    // The request was made but no response was received
                    // `error.request` is an instance of XMLHttpRequest in the browser and an instance of
                    // http.ClientRequest in node.js
                    throw new Error(`E_K8S_API_NO_RESPONSE - Endpoint: ${endpoint} - Error message: ${error.message}`);
                }
                else {
                    // Something happened in setting up the request that triggered an Error
                    throw new Error(`E_CHECTL_UNKNOWN_ERROR - Message: ${error.message}`);
                }
            }
        });
    }
    isOpenShift() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return this.IsAPIGroupSupported('apps.openshift.io');
        });
    }
    isOpenShift3() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const isAppsAPISupported = yield this.IsAPIGroupSupported('apps.openshift.io');
            const isConfigAPISupported = yield this.IsAPIGroupSupported('config.openshift.io');
            return isAppsAPISupported && !isConfigAPISupported;
        });
    }
    isOpenShift4() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const isRouteAPISupported = yield this.IsAPIGroupSupported('route.openshift.io');
            const isConfigAPISupported = yield this.IsAPIGroupSupported('config.openshift.io');
            return isRouteAPISupported && isConfigAPISupported;
        });
    }
    IsAPIExtensionSupported(version) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return this.IsAPIGroupSupported('apiextensions.k8s.io', version);
        });
    }
    IsAPIGroupSupported(name, version) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.ApisApi);
            try {
                const res = yield k8sCoreApi.getAPIVersions();
                if (!res || !res.body || !res.body.groups) {
                    return false;
                }
                const group = res.body.groups.find(g => g.name === name);
                if (!group) {
                    return false;
                }
                if (version) {
                    return Boolean(group.versions.find(v => v.version === version));
                }
                else {
                    return Boolean(group);
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getIngressHost(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sExtensionsApi = this.kubeConfig.makeApiClient(client_node_1.ExtensionsV1beta1Api);
            try {
                const res = yield k8sExtensionsApi.readNamespacedIngress(name, namespace);
                if (res && res.body &&
                    res.body.spec &&
                    res.body.spec.rules &&
                    res.body.spec.rules.length > 0) {
                    return res.body.spec.rules[0].host || '';
                }
                throw new Error('ERR_INGRESS_NO_HOST');
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getIngressProtocol(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sExtensionsApi = this.kubeConfig.makeApiClient(client_node_1.ExtensionsV1beta1Api);
            try {
                const res = yield k8sExtensionsApi.readNamespacedIngress(name, namespace);
                if (!res || !res.body || !res.body.spec) {
                    throw new Error('ERR_INGRESS_NO_HOST');
                }
                if (res.body.spec.tls && res.body.spec.tls.length > 0) {
                    return 'https';
                }
                else {
                    return 'http';
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getIngressesBySelector(labelSelector = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sV1Beta = this.kubeConfig.makeApiClient(client_node_1.ExtensionsV1beta1Api);
            try {
                const res = yield k8sV1Beta.listNamespacedIngress(namespace, 'true', undefined, undefined, undefined, labelSelector);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            throw new Error('ERR_LIST_INGRESSES');
        });
    }
    getSecret(name, namespace = 'default') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            // now get the matching secrets
            try {
                const res = yield k8sCoreApi.readNamespacedSecret(name, namespace);
                if (res && res.body && res.body) {
                    return res.body;
                }
                else {
                    return;
                }
            }
            catch (_a) {
                return;
            }
        });
    }
    /**
     * Creates a secret with given name and data.
     * Data should not be base64 encoded.
     */
    createSecret(namespace, name, data) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            const secret = new client_node_1.V1Secret();
            secret.metadata = new client_node_1.V1ObjectMeta();
            secret.metadata.name = name;
            secret.metadata.namespace = namespace;
            secret.stringData = data;
            try {
                return (yield k8sCoreApi.createNamespacedSecret(namespace, secret)).body;
            }
            catch (_a) {
                return;
            }
        });
    }
    /**
     * Deletes specified secret if any.
     */
    deleteSecret(namespace, name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                yield k8sCoreApi.deleteNamespacedSecret(name, namespace);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    /**
     * Replaces given secret or creates a new one.
     * Data should not be base64 encoded.
     */
    createOrReplaceSecret(namespace, name, data) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const existingSecret = yield this.getSecret(name, namespace);
            if (existingSecret) {
                if (existingSecret.data === data) {
                    return existingSecret;
                }
                existingSecret.data = data;
                const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
                try {
                    const res = yield k8sCoreApi.replaceNamespacedSecret(name, namespace, existingSecret);
                    return res.body;
                }
                catch (e) {
                    throw this.wrapK8sClientError(e);
                }
            }
            return this.createSecret(namespace, name, data);
        });
    }
    /**
     * Awaits secret to be present and contain non-empty data fields specified in dataKeys parameter.
     */
    waitSecret(secretName, namespace, dataKeys = [], timeout = AWAIT_TIMEOUT_S) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                // Set up watcher
                const watcher = new client_node_1.Watch(this.kubeConfig);
                const request = yield watcher
                    .watch(`/api/v1/namespaces/${namespace}/secrets/`, { fieldSelector: `metadata.name=${secretName}` }, (_phase, obj) => {
                    const secret = obj;
                    // Check all required data fields to be present
                    if (dataKeys.length > 0 && secret.data) {
                        for (const key of dataKeys) {
                            if (!secret.data[key]) {
                                // Key is missing or empty
                                return;
                            }
                        }
                    }
                    // The secret with all specified fields is present, stop watching
                    if (request) {
                        request.abort();
                    }
                    // Release awaiter
                    resolve();
                }, error => {
                    if (error) {
                        reject(error);
                    }
                });
                // Automatically stop watching after timeout
                const timeoutHandler = setTimeout(() => {
                    request.abort();
                    reject(`Timeout reached while waiting for "${secretName}" secret.`);
                }, timeout * 1000);
                // Request secret, for case if it is already exist
                const secret = yield this.getSecret(secretName, namespace);
                if (secret) {
                    // Stop watching
                    request.abort();
                    clearTimeout(timeoutHandler);
                    // Relese awaiter
                    resolve();
                }
            }));
        });
    }
    persistentVolumeClaimExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                const { body } = yield k8sCoreApi.readNamespacedPersistentVolumeClaim(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    deletePersistentVolumeClaim(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                yield k8sCoreApi.deleteNamespacedPersistentVolumeClaim(name, namespace);
            }
            catch (e) {
                if (e.response.statusCode !== 404) {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    getPersistentVolumeClaimsBySelector(labelSelector = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sCoreApi.listNamespacedPersistentVolumeClaim(namespace, 'true', undefined, undefined, undefined, labelSelector);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            throw new Error('ERR_LIST_PVCS');
        });
    }
    listNamespace() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sApi.listNamespace();
                if (res && res.body) {
                    return res.body;
                }
                else {
                    return {
                        items: [],
                    };
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    listNamespacedPod(namespace, fieldSelector, labelSelector) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = this.kubeConfig.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sApi.listNamespacedPod(namespace, undefined, undefined, undefined, fieldSelector, labelSelector);
                if (res && res.body) {
                    return res.body;
                }
                else {
                    return {
                        items: [],
                    };
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    /**
     * Reads log by chunk and writes into a file.
     */
    readNamespacedPodLog(pod, namespace, container, filename, follow) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const logHelper = new client_node_1.Log(this.kubeConfig);
                const stream = new stream_1.Writable();
                stream._write = function (chunk, encoding, done) {
                    fs.appendFileSync(filename, chunk, { encoding });
                    done();
                };
                yield logHelper.log(namespace, pod, container, stream, error => {
                    stream.end();
                    if (error) {
                        reject(error);
                    }
                    else {
                        resolve();
                    }
                }, { follow });
            }));
        });
    }
    /**
     * Forwards port, based on the example
     * https://github.com/kubernetes-client/javascript/blob/master/examples/typescript/port-forward/port-forward.ts
     */
    portForward(podName, namespace, port) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const portForwardHelper = new client_node_1.PortForward(this.kubeConfig, true);
            try {
                const server = net.createServer((socket) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield portForwardHelper.portForward(namespace, podName, [port], socket, null, socket);
                }));
                server.listen(port, 'localhost');
                return;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    /**
     * Checks if message is present and returns error with it
     * or returns error with the specified error if message is not found.
     *
     * @param e k8s error to wrap
     */
    wrapK8sClientError(e) {
        if (e.response && e.response.body) {
            return util_1.newError(e.response.body, e);
        }
        return e;
    }
    safeLoadFromYamlFile(filePath) {
        return util_1.safeLoadFromYamlFile(filePath);
    }
}
exports.KubeHelper = KubeHelper;
class PatchedK8sApi extends client_node_1.CoreV1Api {
    patchNamespacedConfigMap(...args) {
        const oldDefaultHeaders = this.defaultHeaders;
        this.defaultHeaders = Object.assign({ 'Content-Type': 'application/strategic-merge-patch+json' }, this.defaultHeaders);
        const returnValue = super.patchNamespacedConfigMap.apply(this, args);
        this.defaultHeaders = oldDefaultHeaders;
        return returnValue;
    }
}
class PatchedK8sAppsApi extends client_node_1.AppsV1Api {
    patchNamespacedDeployment(...args) {
        const oldDefaultHeaders = this.defaultHeaders;
        this.defaultHeaders = Object.assign({ 'Content-Type': 'application/strategic-merge-patch+json' }, this.defaultHeaders);
        const returnValue = super.patchNamespacedDeployment.apply(this, args);
        this.defaultHeaders = oldDefaultHeaders;
        return returnValue;
    }
    patchNamespacedDeploymentScale(...args) {
        const oldDefaultHeaders = this.defaultHeaders;
        this.defaultHeaders = Object.assign({ 'Content-Type': 'application/strategic-merge-patch+json' }, this.defaultHeaders);
        const returnValue = super.patchNamespacedDeploymentScale.apply(this, args);
        this.defaultHeaders = oldDefaultHeaders;
        return returnValue;
    }
}
//# sourceMappingURL=kube.js.map