"use strict";
/*********************************************************************
 * Copyright (c) 2019-2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const client_node_1 = require("@kubernetes/client-node");
const axios_1 = require("axios");
const cp = require("child_process");
const cli_ux_1 = require("cli-ux");
const commandExists = require("command-exists");
const fs = require("fs-extra");
const https = require("https");
const yaml = require("js-yaml");
const os = require("os");
const path = require("path");
const openshift_1 = require("../api/openshift");
const constants_1 = require("../constants");
const util_1 = require("../util");
const kube_1 = require("./kube");
class CheHelper {
    constructor(flags) {
        this.defaultCheResponseTimeoutMs = 3000;
        this.oc = new openshift_1.OpenShiftHelper();
        this.kube = new kube_1.KubeHelper(flags);
        // Make axios ignore untrusted certificate error for self-signed certificate case.
        const httpsAgent = new https.Agent({ rejectUnauthorized: false });
        this.axios = axios_1.default.create({
            httpsAgent
        });
    }
    /**
     * Finds a pod where workspace is running.
     * Rejects if no workspace is found for the given workspace ID
     * or if workspace ID wasn't specified but more than one workspace is found.
     */
    getWorkspacePodName(namespace, cheWorkspaceId) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = kube_1.KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            const res = yield k8sApi.listNamespacedPod(namespace);
            const pods = res.body.items;
            const wsPods = pods.filter(pod => pod.metadata.labels['che.workspace_id'] && pod.metadata.labels['che.original_name'] !== 'che-jwtproxy');
            if (wsPods.length === 0) {
                throw new Error('No workspace pod is found');
            }
            if (cheWorkspaceId) {
                const wsPod = wsPods.find(p => p.metadata.labels['che.workspace_id'] === cheWorkspaceId);
                if (wsPod) {
                    return wsPod.metadata.name;
                }
                throw new Error('Pod is not found for the given workspace ID');
            }
            else {
                if (wsPods.length === 1) {
                    return wsPods[0].metadata.name;
                }
                throw new Error('More than one pod with running workspace is found. Please, specify Workspace ID.');
            }
        });
    }
    getWorkspacePodContainers(namespace, cheWorkspaceId) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = kube_1.KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            const res = yield k8sApi.listNamespacedPod(namespace);
            const pods = res.body.items;
            const wsPods = pods.filter(pod => pod.metadata.labels['che.workspace_id'] && pod.metadata.labels['che.original_name'] !== 'che-jwtproxy');
            if (wsPods.length === 0) {
                throw new Error('No workspace pod is found');
            }
            if (cheWorkspaceId) {
                const wsPod = wsPods.find(p => p.metadata.labels['che.workspace_id'] === cheWorkspaceId);
                if (wsPod) {
                    return wsPod.spec.containers.map(c => c.name);
                }
                throw new Error('Pod is not found for the given workspace ID');
            }
            else {
                if (wsPods.length === 1) {
                    return wsPods[0].spec.containers.map(c => c.name);
                }
                throw new Error('More than one pod with running workspace is found. Please, specify Workspace ID.');
            }
        });
    }
    cheURL(namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!(yield this.cheNamespaceExist(namespace))) {
                throw new Error(`ERR_NAMESPACE_NO_EXIST - No namespace ${namespace} is found`);
            }
            if (yield this.kube.isOpenShift()) {
                return this.cheOpenShiftURL(namespace);
            }
            else {
                return this.cheK8sURL(namespace);
            }
        });
    }
    isSelfSignedCertificateSecretExist(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const selfSignedCertSecret = yield this.kube.getSecret(constants_1.CHE_ROOT_CA_SECRET_NAME, namespace);
            return !!selfSignedCertSecret;
        });
    }
    /**
     * Gets self-signed Che CA certificate from 'self-signed-certificate' secret.
     * If secret doesn't exist, undefined is returned.
     */
    retrieveCheCaCert(cheNamespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const cheCaSecret = yield this.kube.getSecret(constants_1.CHE_ROOT_CA_SECRET_NAME, cheNamespace);
            if (!cheCaSecret) {
                return;
            }
            if (cheCaSecret.data && cheCaSecret.data['ca.crt']) {
                return util_1.base64Decode(cheCaSecret.data['ca.crt']);
            }
            throw new Error(`Secret "${constants_1.CHE_ROOT_CA_SECRET_NAME}" has invalid format: "ca.crt" key not found in data.`);
        });
    }
    saveCheCaCert(cheCaCert, destinaton) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (destinaton && fs.existsSync(destinaton)) {
                if (fs.lstatSync(destinaton).isDirectory()) {
                    destinaton = path.join(destinaton, constants_1.DEFAULT_CA_CERT_FILE_NAME);
                }
            }
            else {
                // Fallback to default location
                destinaton = path.join(os.homedir(), constants_1.DEFAULT_CA_CERT_FILE_NAME);
            }
            fs.writeFileSync(destinaton, cheCaCert);
            return destinaton;
        });
    }
    /**
     * Retreives Keycloak admin user credentials.
     * Works only with installers which use Che CR (operator, olm).
     * Returns credentials as an array of two values: [login, password]
     * In case of an error an array with undefined values will be returned.
     */
    retrieveKeycloakAdminCredentials(cheNamespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let adminUsername;
            let adminPassword;
            const cheCluster = yield this.kube.getCheCluster(cheNamespace);
            if (!cheCluster) {
                return [];
            }
            const keycloakCredentialsSecretName = cheCluster.spec.auth.identityProviderSecret;
            if (keycloakCredentialsSecretName) {
                // Keycloak credentials are stored in secret
                const keycloakCredentialsSecret = yield this.kube.getSecret(keycloakCredentialsSecretName, cheNamespace);
                if (keycloakCredentialsSecret && keycloakCredentialsSecret.data) {
                    adminUsername = util_1.base64Decode(keycloakCredentialsSecret.data.user);
                    adminPassword = util_1.base64Decode(keycloakCredentialsSecret.data.password);
                }
            }
            else {
                // Keycloak credentials are stored in Che custom resource
                adminUsername = cheCluster.spec.auth.identityProviderAdminUserName;
                adminPassword = cheCluster.spec.auth.identityProviderPassword;
            }
            return [adminUsername, adminPassword];
        });
    }
    cheK8sURL(namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const ingress_names = ['che', 'che-ingress'];
            for (const ingress_name of ingress_names) {
                if (yield this.kube.ingressExist(ingress_name, namespace)) {
                    const protocol = yield this.kube.getIngressProtocol(ingress_name, namespace);
                    const hostname = yield this.kube.getIngressHost(ingress_name, namespace);
                    return `${protocol}://${hostname}`;
                }
            }
            throw new Error(`ERR_INGRESS_NO_EXIST - No ingress ${ingress_names} in namespace ${namespace}`);
        });
    }
    cheOpenShiftURL(namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const route_names = ['codeready', 'che-host'];
            for (const route_name of route_names) {
                if (yield this.oc.routeExist(route_name, namespace)) {
                    const protocol = yield this.oc.getRouteProtocol(route_name, namespace);
                    const hostname = yield this.oc.getRouteHost(route_name, namespace);
                    return `${protocol}://${hostname}`;
                }
            }
            throw new Error(`ERR_ROUTE_NO_EXIST - No route ${route_names} in namespace ${namespace}`);
        });
    }
    cheNamespaceExist(namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return this.kube.namespaceExist(namespace);
        });
    }
    getCheServerStatus(cheURL, responseTimeoutMs = this.defaultCheResponseTimeoutMs) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = `${cheURL}/api/system/state`;
            let response = null;
            try {
                response = yield this.axios.get(endpoint, { timeout: responseTimeoutMs });
            }
            catch (error) {
                throw this.getCheApiError(error, endpoint);
            }
            if (!response || response.status !== 200 || !response.data || !response.data.status) {
                throw new Error('E_BAD_RESP_CHE_API');
            }
            return response.data.status;
        });
    }
    startShutdown(cheURL, accessToken = '', responseTimeoutMs = this.defaultCheResponseTimeoutMs) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = `${cheURL}/api/system/stop?shutdown=true`;
            const headers = accessToken ? { Authorization: `${accessToken}` } : null;
            let response = null;
            try {
                response = yield this.axios.post(endpoint, null, { headers, timeout: responseTimeoutMs });
            }
            catch (error) {
                if (error.response && error.response.status === 409) {
                    return;
                }
                else {
                    throw this.getCheApiError(error, endpoint);
                }
            }
            if (!response || response.status !== 204) {
                throw new Error('E_BAD_RESP_CHE_API');
            }
        });
    }
    waitUntilReadyToShutdown(cheURL, intervalMs = 500, timeoutMs = 60000) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                let status = yield this.getCheServerStatus(cheURL);
                if (status === 'READY_TO_SHUTDOWN') {
                    return;
                }
                yield cli_ux_1.cli.wait(intervalMs);
            }
            throw new Error('ERR_TIMEOUT');
        });
    }
    isCheServerReady(cheURL, responseTimeoutMs = this.defaultCheResponseTimeoutMs) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const id = yield this.axios.interceptors.response.use(response => response, (error) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                if (error.config && error.response && (error.response.status === 404 || error.response.status === 503)) {
                    return this.axios.request(error.config);
                }
                return Promise.reject(error);
            }));
            try {
                yield this.axios.get(`${cheURL}/api/system/state`, { timeout: responseTimeoutMs });
                yield this.axios.interceptors.response.eject(id);
                return true;
            }
            catch (_a) {
                yield this.axios.interceptors.response.eject(id);
                return false;
            }
        });
    }
    startWorkspace(cheNamespace, workspaceId, debug, accessToken) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const cheUrl = yield this.cheURL(cheNamespace);
            let endpoint = `${cheUrl}/api/workspace/${workspaceId}/runtime`;
            if (debug) {
                endpoint += '?debug-workspace-start=true';
            }
            let response;
            const headers = {};
            if (accessToken) {
                headers.Authorization = accessToken;
            }
            try {
                response = yield this.axios.post(endpoint, undefined, { headers });
            }
            catch (error) {
                if (error.response && error.response.status === 404) {
                    throw new Error(`E_WORKSPACE_NOT_EXIST - workspace with "${workspaceId}" id doesn't exist`);
                }
                else {
                    throw this.getCheApiError(error, endpoint);
                }
            }
            if (!response || response.status !== 200 || !response.data) {
                throw new Error('E_BAD_RESP_CHE_API');
            }
        });
    }
    stopWorkspace(cheUrl, workspaceId, accessToken) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let endpoint = `${cheUrl}/api/workspace/${workspaceId}/runtime`;
            let response;
            const headers = {};
            if (accessToken) {
                headers.Authorization = accessToken;
            }
            try {
                response = yield this.axios.delete(endpoint, { headers });
            }
            catch (error) {
                if (error.response && error.response.status === 404) {
                    throw new Error(`E_WORKSPACE_NOT_EXIST - workspace with "${workspaceId}" id doesn't exist`);
                }
                else {
                    throw this.getCheApiError(error, endpoint);
                }
            }
            if (!response || response.status !== 204) {
                throw new Error('E_BAD_RESP_CHE_API');
            }
        });
    }
    createWorkspaceFromDevfile(namespace, devfilePath = '', workspaceName, accessToken = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!(yield this.cheNamespaceExist(namespace))) {
                throw new Error('E_BAD_NS');
            }
            let url = yield this.cheURL(namespace);
            let endpoint = `${url}/api/workspace/devfile`;
            let devfile;
            let response;
            const headers = { 'Content-Type': 'text/yaml' };
            if (accessToken && accessToken.length > 0) {
                headers.Authorization = `${accessToken}`;
            }
            try {
                devfile = yield this.parseDevfile(devfilePath);
                if (workspaceName) {
                    let json = yaml.load(devfile);
                    json.metadata.name = workspaceName;
                    devfile = yaml.dump(json);
                }
                response = yield this.axios.post(endpoint, devfile, { headers });
            }
            catch (error) {
                if (!devfile) {
                    throw new Error(`E_NOT_FOUND_DEVFILE - ${devfilePath} - ${error.message}`);
                }
                if (error.response) {
                    if (error.response.status === 400) {
                        throw new Error(`E_BAD_DEVFILE_FORMAT - Message: ${error.response.data.message}`);
                    }
                    if (error.response.status === 409) {
                        let message = '';
                        if (error.response.data) {
                            message = error.response.data.message;
                        }
                        throw new Error(`E_CONFLICT - Message: ${message}`);
                    }
                }
                throw this.getCheApiError(error, endpoint);
            }
            if (response && response.data) {
                return response.data;
            }
            else {
                throw new Error('E_BAD_RESP_CHE_SERVER');
            }
        });
    }
    parseDevfile(devfilePath = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (devfilePath.startsWith('http')) {
                const response = yield this.axios.get(devfilePath);
                return response.data;
            }
            else {
                return fs.readFileSync(devfilePath, 'utf8');
            }
        });
    }
    isAuthenticationEnabled(cheURL, responseTimeoutMs = this.defaultCheResponseTimeoutMs) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = `${cheURL}/api/keycloak/settings`;
            let response = null;
            try {
                response = yield this.axios.get(endpoint, { timeout: responseTimeoutMs });
            }
            catch (error) {
                if (error.response && (error.response.status === 404 || error.response.status === 503)) {
                    return false;
                }
                else {
                    throw this.getCheApiError(error, endpoint);
                }
            }
            if (!response || response.status !== 200 || !response.data) {
                throw new Error('E_BAD_RESP_CHE_API');
            }
            return true;
        });
    }
    buildDashboardURL(ideURL) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return ideURL.replace(/\/[^/|.]*\/[^/|.]*$/g, '\/dashboard\/#\/ide$&');
        });
    }
    /**
     * Finds workspace pods and reads logs from it.
     */
    readWorkspacePodLog(namespace, workspaceId, directory) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const podLabelSelector = `che.workspace_id=${workspaceId}`;
            let workspaceIsRun = false;
            const pods = yield this.kube.listNamespacedPod(namespace, undefined, podLabelSelector);
            if (pods.items.length) {
                workspaceIsRun = true;
            }
            for (const pod of pods.items) {
                for (const containerStatus of pod.status.containerStatuses) {
                    workspaceIsRun = workspaceIsRun && !!containerStatus.state && !!containerStatus.state.running;
                }
            }
            const follow = !workspaceIsRun;
            yield this.readPodLog(namespace, podLabelSelector, directory, follow);
            yield this.readNamespaceEvents(namespace, directory, follow);
            return workspaceIsRun;
        });
    }
    /**
     * Reads logs from pods that match a given selector.
     */
    readPodLog(namespace, podLabelSelector, directory, follow) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (follow) {
                yield this.watchNamespacedPods(namespace, podLabelSelector, directory);
            }
            else {
                yield this.readNamespacedPodLog(namespace, podLabelSelector, directory);
            }
        });
    }
    /**
     * Reads containers logs inside pod that match a given selector.
     */
    readNamespacedPodLog(namespace, podLabelSelector, directory) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const pods = yield this.kube.listNamespacedPod(namespace, undefined, podLabelSelector);
            for (const pod of pods.items) {
                if (!pod.status || !pod.status.containerStatuses) {
                    return;
                }
                const podName = pod.metadata.name;
                for (const containerName of this.getContainers(pod)) {
                    const fileName = this.doCreateLogFile(namespace, podName, containerName, directory);
                    yield this.doReadNamespacedPodLog(namespace, podName, containerName, fileName, false);
                }
            }
        });
    }
    /**
     * Reads all namespace events and store into a file.
     */
    readNamespaceEvents(namespace, directory, follow) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const fileName = path.resolve(directory, namespace, 'events.txt');
            fs.ensureFileSync(fileName);
            const cli = (commandExists.sync('kubectl') && 'kubectl') || (commandExists.sync('oc') && 'oc');
            if (cli) {
                const command = 'get events';
                const namespaceParam = `-n ${namespace}`;
                const watchParam = follow && '--watch' || '';
                cp.exec(`${cli} ${command} ${namespaceParam} ${watchParam} >> ${fileName}`);
            }
            else {
                throw new Error('No events are collected. \'kubectl\' or \'oc\' is required to perform the task.');
            }
        });
    }
    watchNamespacedPods(namespace, podLabelSelector, directory) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const processedContainers = new Map();
            const watcher = new client_node_1.Watch(kube_1.KubeHelper.KUBE_CONFIG);
            watcher.watch(`/api/v1/namespaces/${namespace}/pods`, {}, (_phase, obj) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const pod = obj;
                if (!pod || !pod.metadata || !pod.metadata.name) {
                    return;
                }
                const podName = pod.metadata.name;
                if (!processedContainers.has(podName)) {
                    processedContainers.set(podName, new Set());
                }
                if (!podLabelSelector || this.matchLabels(pod.metadata.labels || {}, podLabelSelector)) {
                    for (const containerName of this.getContainers(pod)) {
                        // not to read logs from the same containers twice
                        if (!processedContainers.get(podName).has(containerName)) {
                            processedContainers.get(podName).add(containerName);
                            const fileName = this.doCreateLogFile(namespace, podName, containerName, directory);
                            yield this.doReadNamespacedPodLog(namespace, pod.metadata.name, containerName, fileName, true);
                        }
                    }
                }
            }), 
            // ignore errors
            () => { });
        });
    }
    getAllWorkspaces(cheURL, accessToken) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const all = [];
            const maxItems = 30;
            let skipCount = 0;
            do {
                const workspaces = yield this.doGetWorkspaces(cheURL, skipCount, maxItems, accessToken);
                all.push(...workspaces);
                skipCount += workspaces.length;
            } while (all.length === maxItems);
            return all;
        });
    }
    /**
     * Returns list of workspaces
     */
    doGetWorkspaces(cheUrl, skipCount, maxItems, accessToken = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = `${cheUrl}/api/workspace?skipCount=${skipCount}&maxItems=${maxItems}`;
            const headers = { 'Content-Type': 'text/yaml' };
            if (accessToken && accessToken.length > 0) {
                headers.Authorization = `${accessToken}`;
            }
            try {
                const response = yield this.axios.get(endpoint, { headers });
                if (response && response.data) {
                    return response.data;
                }
                else {
                    throw new Error('E_BAD_RESP_CHE_SERVER');
                }
            }
            catch (error) {
                throw this.getCheApiError(error, endpoint);
            }
        });
    }
    /**
     * Get workspace.
     */
    getWorkspace(cheUrl, workspaceId, accessToken = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = `${cheUrl}/api/workspace/${workspaceId}`;
            const headers = { 'Content-Type': 'text/yaml' };
            if (accessToken && accessToken.length > 0) {
                headers.Authorization = `${accessToken}`;
            }
            try {
                const response = yield this.axios.get(endpoint, { headers });
                return response.data;
            }
            catch (error) {
                throw this.getCheApiError(error, endpoint);
            }
        });
    }
    /**
     * Deletes workspace.
     */
    deleteWorkspace(cheUrl, workspaceId, accessToken = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = `${cheUrl}/api/workspace/${workspaceId}`;
            const headers = {};
            if (accessToken) {
                headers.Authorization = `${accessToken}`;
            }
            try {
                yield this.axios.delete(endpoint, { headers });
            }
            catch (error) {
                if (error.response.status === 409) {
                    throw new Error(`Workspace '${workspaceId}' not found`);
                }
                throw this.getCheApiError(error, endpoint);
            }
        });
    }
    /**
     * Indicates if pod matches given labels.
     */
    matchLabels(podLabels, podLabelSelector) {
        const labels = podLabelSelector.split(',');
        for (const label of labels) {
            if (label) {
                const keyValue = label.split('=');
                if (podLabels[keyValue[0]] !== keyValue[1]) {
                    return false;
                }
            }
        }
        return true;
    }
    /**
     * Returns containers names.
     */
    getContainers(pod) {
        if (!pod.status || !pod.status.containerStatuses) {
            return [];
        }
        return pod.status.containerStatuses.map(containerStatus => containerStatus.name);
    }
    /**
     * Reads pod log from a specific container of the pod.
     */
    doReadNamespacedPodLog(namespace, podName, containerName, fileName, follow) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (follow) {
                try {
                    yield this.kube.readNamespacedPodLog(podName, namespace, containerName, fileName, follow);
                }
                catch (_a) {
                    // retry in 200ms, container might not be started
                    setTimeout(() => tslib_1.__awaiter(this, void 0, void 0, function* () { return this.doReadNamespacedPodLog(namespace, podName, containerName, fileName, follow); }), 200);
                }
            }
            else {
                yield this.kube.readNamespacedPodLog(podName, namespace, containerName, fileName, follow);
            }
        });
    }
    doCreateLogFile(namespace, podName, containerName, directory) {
        const fileName = path.resolve(directory, namespace, podName, `${containerName}.log`);
        fs.ensureFileSync(fileName);
        return fileName;
    }
    getCheApiError(error, endpoint) {
        if (error.response) {
            const status = error.response.status;
            if (status === 403) {
                return new Error(`E_CHE_API_FORBIDDEN - Endpoint: ${endpoint} - Message: ${JSON.stringify(error.response.data.message)}`);
            }
            else if (status === 401) {
                return new Error(`E_CHE_API_UNAUTHORIZED - Endpoint: ${endpoint} - Message: ${JSON.stringify(error.response.data)}`);
            }
            else if (status === 404) {
                return new Error(`E_CHE_API_NOTFOUND - Endpoint: ${endpoint} - Message: ${JSON.stringify(error.response.data)}`);
            }
            else {
                // The request was made and the server responded with a status code
                // that falls out of the range of 2xx
                return new Error(`E_CHE_API_UNKNOWN_ERROR - Endpoint: ${endpoint} -Status: ${error.response.status}`);
            }
        }
        else if (error.request) {
            // The request was made but no response was received
            // `error.request` is an instance of XMLHttpRequest in the browser and an instance of
            // http.ClientRequest in node.js
            return new Error(`E_CHE_API_NO_RESPONSE - Endpoint: ${endpoint} - Error message: ${error.message}`);
        }
        else {
            // Something happened in setting up the request that triggered an Error
            return new Error(`E_CHECTL_UNKNOWN_ERROR - Endpoint: ${endpoint} - Message: ${error.message}`);
        }
    }
}
exports.CheHelper = CheHelper;
//# sourceMappingURL=che.js.map