"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const cli_ux_1 = require("cli-ux");
const execa = require("execa");
const fs = require("fs");
const os = require("os");
const path = require("path");
const che_1 = require("../../api/che");
const che_api_client_1 = require("../../api/che-api-client");
const kube_1 = require("../../api/kube");
const common_flags_1 = require("../../common-flags");
const util_1 = require("../../util");
class Inject extends command_1.Command {
    constructor() {
        super(...arguments);
        // Holds cluster CLI tool name: kubectl or oc
        this.command = util_1.getClusterClientCommand();
    }
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Inject);
            const notifier = require('node-notifier');
            const cheHelper = new che_1.CheHelper(flags);
            let cheApiEndpoint = flags[common_flags_1.CHE_API_ENDPOINT_KEY];
            if (!cheApiEndpoint) {
                const kube = new kube_1.KubeHelper(flags);
                if (!(yield kube.hasReadPermissionsForNamespace(flags.chenamespace))) {
                    throw new Error(`CodeReady Workspaces API endpoint is required. Use flag --${common_flags_1.CHE_API_ENDPOINT_KEY} to provide it.`);
                }
                cheApiEndpoint = (yield cheHelper.cheURL(flags.chenamespace)) + '/api';
            }
            const cheApiClient = che_api_client_1.CheApiClient.getInstance(cheApiEndpoint);
            yield cheApiClient.checkCheApiEndpointUrl();
            if (!flags[common_flags_1.ACCESS_TOKEN_KEY] && (yield cheApiClient.isAuthenticationEnabled())) {
                cli_ux_1.cli.error('Authentication is enabled but \'access-token\' is not provided.\nSee more details with the --help flag.');
            }
            let workspaceId = flags.workspace;
            let workspaceNamespace = '';
            if (!workspaceId) {
                const workspaces = yield cheApiClient.getAllWorkspaces(flags[common_flags_1.ACCESS_TOKEN_KEY]);
                const runningWorkspaces = workspaces.filter(w => w.status === 'RUNNING');
                if (runningWorkspaces.length === 1) {
                    workspaceId = runningWorkspaces[0].id;
                    workspaceNamespace = runningWorkspaces[0].attributes.infrastructureNamespace;
                }
                else if (runningWorkspaces.length === 0) {
                    cli_ux_1.cli.error('There are no running workspaces. Please start workspace first.');
                }
                else {
                    cli_ux_1.cli.error('There are more than 1 running workspaces. Please, specify the workspace id by providing \'--workspace\' flag.\nSee more details with the --help flag.');
                }
            }
            else {
                const workspace = yield cheApiClient.getWorkspaceById(workspaceId, flags[common_flags_1.ACCESS_TOKEN_KEY]);
                if (workspace.status !== 'RUNNING') {
                    cli_ux_1.cli.error(`Workspace '${workspaceId}' is not running. Please start workspace first.`);
                }
                workspaceNamespace = workspace.attributes.infrastructureNamespace;
            }
            const workspacePodName = yield cheHelper.getWorkspacePodName(workspaceNamespace, workspaceId);
            if (flags.container && !(yield this.containerExists(workspaceNamespace, workspacePodName, flags.container))) {
                cli_ux_1.cli.error(`The specified container '${flags.container}' doesn't exist. The configuration cannot be injected.`);
            }
            try {
                yield this.injectKubeconfig(flags, workspaceNamespace, workspacePodName, workspaceId);
            }
            catch (err) {
                this.error(err);
            }
            notifier.notify({
                title: 'crwctl',
                message: `Command ${this.id} has completed.`
            });
        });
    }
    injectKubeconfig(flags, workspaceNamespace, workspacePodName, workspaceId) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const kubeContext = flags['kube-context'];
            let contextToInject;
            const kh = new kube_1.KubeHelper(flags);
            if (kubeContext) {
                contextToInject = kh.getContext(kubeContext);
                if (!contextToInject) {
                    this.error(`Context ${kubeContext} is not found in the source kubeconfig`);
                }
            }
            else {
                const currentContext = yield kh.currentContext();
                contextToInject = kh.getContext(currentContext);
            }
            const che = new che_1.CheHelper(flags);
            const containers = flags.container ? [flags.container] : yield che.getWorkspacePodContainers(workspaceNamespace, workspaceId);
            for (const container of containers) {
                // che-machine-exec container is very limited for a security reason.
                // We cannot copy file into it.
                if (container.startsWith('che-machine-exec') || container.startsWith('che-jwtproxy')) {
                    continue;
                }
                try {
                    if (yield this.canInject(workspaceNamespace, workspacePodName, container)) {
                        yield this.doInjectKubeconfig(workspaceNamespace, workspacePodName, container, contextToInject);
                        cli_ux_1.cli.info(`Configuration successfully injected into ${container} container`);
                    }
                }
                catch (error) {
                    cli_ux_1.cli.warn(`Failed to injected configuration into ${container} container.\nError: ${error.message}`);
                }
            }
        });
    }
    /**
     * Tests whether a file can be injected into the specified container.
     */
    canInject(namespace, pod, container) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { exitCode } = yield execa(`${this.command} exec ${pod} -n ${namespace} -c ${container} -- tar --version `, { timeout: 10000, reject: false, shell: true });
            if (exitCode === 0) {
                return true;
            }
            else {
                return false;
            }
        });
    }
    /**
     * Copies the local kubeconfig into the specified container.
     * If returns, it means injection was completed successfully. If throws an error, injection failed
     */
    doInjectKubeconfig(namespace, workspacePod, container, contextToInject) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { stdout } = yield execa(`${this.command} exec ${workspacePod} -n ${namespace} -c ${container} env | grep ^HOME=`, { timeout: 10000, shell: true });
            let containerHomeDir = stdout.split('=')[1];
            if (!containerHomeDir.endsWith('/')) {
                containerHomeDir += '/';
            }
            if (yield this.fileExists(namespace, workspacePod, container, `${containerHomeDir}.kube/config`)) {
                throw new Error('kubeconfig already exists in the target container');
            }
            yield execa(`${this.command} exec ${workspacePod} -n ${namespace} -c ${container} -- mkdir ${containerHomeDir}.kube -p`, { timeout: 10000, shell: true });
            const kubeConfigPath = path.join(os.tmpdir(), 'che-kubeconfig');
            const cluster = kube_1.KubeHelper.KUBE_CONFIG.getCluster(contextToInject.cluster);
            if (!cluster) {
                throw new Error(`Context ${contextToInject.name} has no cluster object`);
            }
            const user = kube_1.KubeHelper.KUBE_CONFIG.getUser(contextToInject.user);
            if (!user) {
                throw new Error(`Context ${contextToInject.name} has no user object`);
            }
            // Despite oc has --kubeconfig flag it actually does nothing, so we need to use --config instead
            const configPathFlag = this.command === util_1.OPENSHIFT_CLI ? '--config' : '--kubeconfig';
            const setClusterArgs = ['config', configPathFlag, kubeConfigPath, 'set-cluster', cluster.name, `--server=${cluster.server}`];
            // Prepare CA certificate file
            if (cluster.caFile) {
                setClusterArgs.push(`--certificate-authority=${cluster.caFile}`);
                setClusterArgs.push('--embed-certs=true');
            }
            else if (cluster.caData) {
                const caFile = path.join(os.tmpdir(), 'cluster-ca-file.pem');
                // Write caData into a file and pass it as the parameter
                fs.writeFileSync(caFile, cluster.caData, 'utf8');
                setClusterArgs.push(`--certificate-authority=${caFile}`);
                setClusterArgs.push('--embed-certs=true');
            }
            yield execa(this.command, setClusterArgs, { timeout: 10000 });
            const setCredentialsArgs = ['config', configPathFlag, kubeConfigPath, 'set-credentials', user.name];
            if (user.certFile) {
                setCredentialsArgs.push(`--client-certificate=${user.certFile}`);
            }
            if (user.keyFile) {
                setCredentialsArgs.push(`--client-key=${user.keyFile}`);
            }
            if (user.certFile || user.keyFile) {
                setCredentialsArgs.push('--embed-certs=true');
            }
            yield execa(this.command, setCredentialsArgs, { timeout: 10000 });
            yield execa(this.command, ['config', configPathFlag, kubeConfigPath, 'set-context', contextToInject.name, `--cluster=${contextToInject.cluster}`, `--user=${contextToInject.user}`, `--namespace=${namespace}`], { timeout: 10000 });
            yield execa(this.command, ['config', configPathFlag, kubeConfigPath, 'use-context', contextToInject.name], { timeout: 10000 });
            yield execa(this.command, ['cp', kubeConfigPath, `${namespace}/${workspacePod}:${containerHomeDir}.kube/config`, '-c', container], { timeout: 10000 });
            return;
        });
    }
    fileExists(namespace, pod, container, file) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { exitCode } = yield execa(`${this.command} exec ${pod} -n ${namespace} -c ${container} -- test -e ${file}`, { timeout: 10000, reject: false, shell: true });
            if (exitCode === 0) {
                return true;
            }
            else {
                return false;
            }
        });
    }
    containerExists(namespace, pod, container) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { stdout } = yield execa(this.command, ['get', 'pods', `${pod}`, '-n', `${namespace}`, '-o', 'jsonpath={.spec.containers[*].name}'], { timeout: 10000 });
            return stdout.split(' ').some(c => c === container);
        });
    }
}
exports.default = Inject;
Inject.description = 'inject configurations and tokens in a workspace';
Inject.flags = {
    help: command_1.flags.help({ char: 'h' }),
    kubeconfig: command_1.flags.boolean({
        char: 'k',
        description: 'Inject the local Kubernetes configuration',
        required: true
    }),
    workspace: flags_1.string({
        char: 'w',
        description: `The workspace id to inject configuration into. It can be omitted if the only one running workspace exists.
                    Use workspace:list command to get all workspaces and their statuses.`
    }),
    container: flags_1.string({
        char: 'c',
        description: 'The container name. If not specified, configuration files will be injected in all containers of the workspace pod',
        required: false
    }),
    'kube-context': flags_1.string({
        description: 'Kubeconfig context to inject',
        required: false
    }),
    [common_flags_1.CHE_API_ENDPOINT_KEY]: common_flags_1.cheApiEndpoint,
    [common_flags_1.ACCESS_TOKEN_KEY]: common_flags_1.accessToken,
    chenamespace: common_flags_1.cheNamespace,
    'skip-kubernetes-health-check': common_flags_1.skipKubeHealthzCheck
};
//# sourceMappingURL=inject.js.map