"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const cli_ux_1 = require("cli-ux");
const che_1 = require("../../api/che");
const che_api_client_1 = require("../../api/che-api-client");
const kube_1 = require("../../api/kube");
const common_flags_1 = require("../../common-flags");
class List extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(List);
            let workspaces = [];
            let cheApiEndpoint = flags[common_flags_1.CHE_API_ENDPOINT_KEY];
            if (!cheApiEndpoint) {
                const kube = new kube_1.KubeHelper(flags);
                if (!(yield kube.hasReadPermissionsForNamespace(flags.chenamespace))) {
                    throw new Error(`CodeReady Workspaces API endpoint is required. Use flag --${common_flags_1.CHE_API_ENDPOINT_KEY} to provide it.`);
                }
                const cheHelper = new che_1.CheHelper(flags);
                cheApiEndpoint = (yield cheHelper.cheURL(flags.chenamespace)) + '/api';
            }
            const cheApiClient = che_api_client_1.CheApiClient.getInstance(cheApiEndpoint);
            yield cheApiClient.checkCheApiEndpointUrl();
            workspaces = yield cheApiClient.getAllWorkspaces(flags[common_flags_1.ACCESS_TOKEN_KEY]);
            this.printWorkspaces(workspaces);
        });
    }
    printWorkspaces(workspaces) {
        const data = [];
        workspaces.forEach((workspace) => {
            data.push({
                id: workspace.id,
                name: workspace.devfile.metadata.name,
                namespace: workspace.attributes.infrastructureNamespace,
                status: workspace.status,
                created: new Date(parseInt(workspace.attributes.created, 10)).toISOString(),
                updated: workspace.attributes.updated ? new Date(parseInt(workspace.attributes.updated, 10)).toISOString() : ''
            });
        });
        cli_ux_1.cli.table(data, { id: {}, name: {}, namespace: {}, status: {}, created: {}, updated: {} });
    }
}
exports.default = List;
List.description = 'list workspaces';
List.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    [common_flags_1.CHE_API_ENDPOINT_KEY]: common_flags_1.cheApiEndpoint,
    [common_flags_1.ACCESS_TOKEN_KEY]: common_flags_1.accessToken,
    'skip-kubernetes-health-check': common_flags_1.skipKubeHealthzCheck
};
//# sourceMappingURL=list.js.map