"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/command/lib/flags");
const cli_ux_1 = require("cli-ux");
const Listrq = require("listr");
const kube_1 = require("../../api/kube");
const common_flags_1 = require("../../common-flags");
const che_1 = require("../../tasks/che");
const devfile_workspace_operator_installer_1 = require("../../tasks/component-installers/devfile-workspace-operator-installer");
const olm_1 = require("../../tasks/installers/olm");
const operator_1 = require("../../tasks/installers/operator");
const api_1 = require("../../tasks/platforms/api");
const util_1 = require("../../util");
class Delete extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const ctx = util_1.initializeContext();
            const { flags } = this.parse(Delete);
            const notifier = require('node-notifier');
            const apiTasks = new api_1.ApiTasks();
            const operatorTasks = new operator_1.OperatorTasks();
            const olmTasks = new olm_1.OLMTasks();
            const cheTasks = new che_1.CheTasks(flags);
            const devWorkspaceTasks = new devfile_workspace_operator_installer_1.DevWorkspaceTasks(flags);
            let tasks = new Listrq(undefined, { renderer: flags['listr-renderer'] });
            tasks.add(apiTasks.testApiTasks(flags, this));
            tasks.add(operatorTasks.deleteTasks(flags));
            tasks.add(olmTasks.deleteTasks(flags));
            tasks.add(cheTasks.deleteTasks(flags));
            tasks.add(devWorkspaceTasks.getUninstallTasks());
            tasks.add(cheTasks.waitPodsDeletedTasks());
            if (flags['delete-namespace']) {
                tasks.add(cheTasks.deleteNamespace(flags));
            }
            const cluster = kube_1.KubeHelper.KUBE_CONFIG.getCurrentCluster();
            if (!cluster) {
                throw new Error('Failed to get current Kubernetes cluster. Check if the current context is set via kubect/oc');
            }
            if (!flags['skip-deletion-check']) {
                const confirmed = yield cli_ux_1.cli.confirm(`You're going to remove CodeReady Workspaces server in namespace '${flags.chenamespace}' on server '${cluster ? cluster.server : ''}'. If you want to continue - press Y`);
                if (!confirmed) {
                    this.exit(0);
                }
            }
            try {
                yield tasks.run();
                cli_ux_1.cli.log(util_1.getCommandSuccessMessage(this, ctx));
            }
            catch (error) {
                cli_ux_1.cli.error(error);
            }
            notifier.notify({
                title: 'crwctl',
                message: util_1.getCommandSuccessMessage(this, ctx)
            });
            this.exit(0);
        });
    }
}
exports.default = Delete;
Delete.description = 'delete any CodeReady Workspaces related resource';
Delete.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    'dev-workspace-controller-namespace': common_flags_1.devWorkspaceControllerNamespace,
    'delete-namespace': flags_1.boolean({
        description: 'Indicates that a CodeReady Workspaces namespace will be deleted as well',
        default: false
    }),
    'deployment-name': common_flags_1.cheDeployment,
    'listr-renderer': common_flags_1.listrRenderer,
    'skip-deletion-check': flags_1.boolean({
        description: 'Skip user confirmation on deletion check',
        default: false
    }),
    'skip-kubernetes-health-check': common_flags_1.skipKubeHealthzCheck
};
//# sourceMappingURL=delete.js.map