"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const cli_ux_1 = require("cli-ux");
const Listr = require("listr");
const semver = require("semver");
const context_1 = require("../../api/context");
const kube_1 = require("../../api/kube");
const common_flags_1 = require("../../common-flags");
const constants_1 = require("../../constants");
const che_1 = require("../../tasks/che");
const devfile_workspace_operator_installer_1 = require("../../tasks/component-installers/devfile-workspace-operator-installer");
const common_tasks_1 = require("../../tasks/installers/common-tasks");
const installer_1 = require("../../tasks/installers/installer");
const api_1 = require("../../tasks/platforms/api");
const platform_1 = require("../../tasks/platforms/platform");
const util_1 = require("../../util");
class Deploy extends command_1.Command {
    setPlaformDefaults(flags, ctx) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            flags.tls = yield this.checkTlsMode(ctx);
            if (flags['self-signed-cert']) {
                this.warn('"self-signed-cert" flag is deprecated and has no effect. Autodetection is used instead.');
            }
            if (!flags.installer) {
                yield this.setDefaultInstaller(flags, ctx);
                cli_ux_1.cli.info(`› Installer type is set to: '${flags.installer}'`);
            }
            if (flags.installer === 'olm' && flags['olm-suggested-namespace']) {
                flags.chenamespace = constants_1.DEFAULT_OLM_SUGGESTED_NAMESPACE;
                cli_ux_1.cli.info(` ❕olm-suggested-namespace flag is turned on. CodeReady Workspaces will be deployed in namespace: ${constants_1.DEFAULT_OLM_SUGGESTED_NAMESPACE}.`);
            }
            if (!ctx.isChectl && flags.version) {
                // Flavors of crwctl should not use upstream repositories, so version flag is not applicable
                this.error(`${util_1.getProjectName()} does not support '--version' flag.`);
            }
            if (!flags.templates && !flags.version) {
                // Use build-in templates if no custom templates nor version to deploy specified.
                // All flavors should use embedded templates if not custom templates is given.
                flags.templates = util_1.getEmbeddedTemplatesDirectory();
            }
        });
    }
    /**
     * Checks if TLS is disabled via operator custom resource.
     * Returns true if TLS is enabled (or omitted) and false if it is explicitly disabled.
     */
    checkTlsMode(ctx) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const crPatch = ctx.crPatch;
            if (crPatch && crPatch.spec && crPatch.spec.server && crPatch.spec.server.tlsSupport === false) {
                return false;
            }
            const customCR = ctx.customCR;
            if (customCR && customCR.spec && customCR.spec.server && customCR.spec.server.tlsSupport === false) {
                return false;
            }
            return true;
        });
    }
    checkCompatibility(flags) {
        if (flags.installer === 'operator' && flags[common_flags_1.CHE_OPERATOR_CR_YAML_KEY]) {
            const ignoredFlags = [];
            flags['plugin-registry-url'] && ignoredFlags.push('--plugin-registry-url');
            flags['devfile-registry-url'] && ignoredFlags.push('--devfile-registry-url');
            flags['postgres-pvc-storage-class-name'] && ignoredFlags.push('--postgres-pvc-storage-class-name');
            flags['workspace-pvc-storage-class-name'] && ignoredFlags.push('--workspace-pvc-storage-class-name');
            flags.tls && ignoredFlags.push('--tls');
            flags.cheimage && ignoredFlags.push('--cheimage');
            flags.debug && ignoredFlags.push('--debug');
            flags.domain && ignoredFlags.push('--domain');
            flags.multiuser && ignoredFlags.push('--multiuser');
            if (ignoredFlags.length) {
                this.warn(`--${common_flags_1.CHE_OPERATOR_CR_YAML_KEY} is used. The following flag(s) will be ignored: ${ignoredFlags.join('\t')}`);
            }
        }
        if (flags.domain && !flags[common_flags_1.CHE_OPERATOR_CR_YAML_KEY] && util_1.isOpenshiftPlatformFamily(flags.platform)) {
            this.warn('"--domain" flag is ignored for Openshift family infrastructures. It should be done on the cluster level.');
        }
        if (flags.installer === 'helm') {
            if (!util_1.isKubernetesPlatformFamily(flags.platform) && flags.platform !== 'docker-desktop') {
                this.error(`🛑 Current platform is ${flags.platform}. Helm installer is only available on top of Kubernetes flavor platform (including Minikube, Docker Desktop).`);
            }
        }
        if (flags.installer === 'olm') {
            // OLM installer only checks
            if (flags.platform === 'minishift') {
                this.error(`🛑 The specified installer ${flags.installer} does not support Minishift`);
            }
            if (flags['catalog-source-name'] && flags['catalog-source-yaml']) {
                this.error('should be provided only one argument: "catalog-source-name" or "catalog-source-yaml"');
            }
            if (flags.version) {
                if (flags['starting-csv']) {
                    this.error('"starting-csv" and "version" flags are mutually exclusive. Please specify only one of them.');
                }
                if (flags['olm-channel']) {
                    this.error('"starting-csv" and "version" flags are mutually exclusive. Use "starting-csv" with "olm-channel" flag.');
                }
                if (flags['auto-update']) {
                    this.error('enabled "auto-update" flag cannot be used with version flag. Deploy latest version instead.');
                }
            }
            if (!flags['package-manifest-name'] && flags['catalog-source-yaml']) {
                this.error('you need to define "package-manifest-name" flag to use "catalog-source-yaml".');
            }
            if (!flags['olm-channel'] && flags['catalog-source-yaml']) {
                this.error('you need to define "olm-channel" flag to use "catalog-source-yaml".');
            }
        }
        else {
            // Not OLM installer
            if (flags['starting-csv']) {
                this.error('"starting-csv" flag should be used only with "olm" installer.');
            }
            if (flags['catalog-source-yaml']) {
                this.error('"catalog-source-yaml" flag should be used only with "olm" installer.');
            }
            if (flags['olm-channel']) {
                this.error('"olm-channel" flag should be used only with "olm" installer.');
            }
            if (flags['package-manifest-name']) {
                this.error('"package-manifest-name" flag should be used only with "olm" installer.');
            }
            if (flags['catalog-source-name']) {
                this.error('"catalog-source-name" flag should be used only with "olm" installer.');
            }
            if (flags['catalog-source-namespace']) {
                this.error('"package-manifest-name" flag should be used only with "olm" installer.');
            }
            if (flags['cluster-monitoring'] && flags.platform !== 'openshift') {
                this.error('"cluster-monitoring" flag should be used only with "olm" installer and "openshift" platform.');
            }
        }
        if (flags.version) {
            // Check minimal allowed version to install
            let minAllowedVersion;
            switch (flags.installer) {
                case 'olm':
                    minAllowedVersion = constants_1.MIN_OLM_INSTALLER_VERSION;
                    break;
                case 'operator':
                    minAllowedVersion = constants_1.MIN_CHE_OPERATOR_INSTALLER_VERSION;
                    break;
                case 'helm':
                    minAllowedVersion = constants_1.MIN_HELM_INSTALLER_VERSION;
                    break;
                default:
                    // Should never happen
                    minAllowedVersion = 'latest';
            }
            if (semver.gt(minAllowedVersion, flags.version)) {
                throw new Error(`This crwctl version can deploy version ${minAllowedVersion} and higher. If you need to deploy ${flags.version} or lower, download the corresponding legacy crwctl version.`);
            }
        }
    }
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Deploy);
            flags.chenamespace = flags.chenamespace || constants_1.DEFAULT_CHE_NAMESPACE;
            const ctx = yield context_1.ChectlContext.initAndGet(flags, this);
            if (!flags.batch && ctx.isChectl) {
                yield util_1.askForChectlUpdateIfNeeded();
            }
            yield this.setPlaformDefaults(flags, ctx);
            yield this.config.runHook(constants_1.DEFAULT_ANALYTIC_HOOK_NAME, { command: Deploy.id, flags });
            const cheTasks = new che_1.CheTasks(flags);
            const platformTasks = new platform_1.PlatformTasks();
            const installerTasks = new installer_1.InstallerTasks();
            const apiTasks = new api_1.ApiTasks();
            const devWorkspaceTasks = new devfile_workspace_operator_installer_1.DevWorkspaceTasks(flags);
            // Platform Checks
            let platformCheckTasks = new Listr(platformTasks.preflightCheckTasks(flags, this), ctx.listrOptions);
            // Checks if CodeReady Workspaces is already deployed
            let preInstallTasks = new Listr(undefined, ctx.listrOptions);
            preInstallTasks.add(apiTasks.testApiTasks(flags, this));
            preInstallTasks.add({
                title: '👀  Looking for an already existing CodeReady Workspaces instance',
                task: () => new Listr(cheTasks.checkIfCheIsInstalledTasks(flags, this))
            });
            preInstallTasks.add(common_tasks_1.checkChectlAndCheVersionCompatibility(flags));
            preInstallTasks.add(common_tasks_1.downloadTemplates(flags));
            let installTasks = new Listr(installerTasks.installTasks(flags, this), ctx.listrOptions);
            // Post Install Checks
            const postInstallTasks = new Listr([
                {
                    title: '✅  Post installation checklist',
                    task: () => new Listr(cheTasks.waitDeployedChe())
                },
                {
                    title: '🧪  DevWorkspace engine (experimental / technology preview) 🚨',
                    enabled: () => flags['workspace-engine'] === 'dev-workspace',
                    task: () => new Listr(devWorkspaceTasks.getInstallTasks(flags))
                },
                common_tasks_1.getRetrieveKeycloakCredentialsTask(flags),
                common_tasks_1.retrieveCheCaCertificateTask(flags),
                ...cheTasks.preparePostInstallationOutput(flags),
                common_tasks_1.getPrintHighlightedMessagesTask(),
            ], ctx.listrOptions);
            const logsTasks = new Listr([{
                    title: 'Following CodeReady Workspaces logs',
                    task: () => new Listr(cheTasks.serverLogsTasks(flags, true))
                }], ctx.listrOptions);
            try {
                yield preInstallTasks.run(ctx);
                if (ctx.isCheDeployed) {
                    let message = 'CodeReady Workspaces has been already deployed.';
                    if (!ctx.isCheReady) {
                        message += ' Use server:start command to start a stopped CodeReady Workspaces instance.';
                    }
                    cli_ux_1.cli.warn(message);
                }
                else {
                    this.checkCompatibility(flags);
                    yield platformCheckTasks.run(ctx);
                    yield logsTasks.run(ctx);
                    yield installTasks.run(ctx);
                    yield postInstallTasks.run(ctx);
                    this.log(util_1.getCommandSuccessMessage());
                }
            }
            catch (err) {
                this.error(util_1.getCommandErrorMessage(err));
            }
            util_1.notifyCommandCompletedSuccessfully();
            this.exit(0);
        });
    }
    /**
     * Sets default installer which is `olm` for OpenShift 4 with stable version of crwctl
     * and `operator` for other cases.
     */
    setDefaultInstaller(flags, ctx) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const kubeHelper = new kube_1.KubeHelper(flags);
            const isOlmPreinstalled = yield kubeHelper.isPreInstalledOLM();
            if ((flags['catalog-source-name'] || flags['catalog-source-yaml']) && isOlmPreinstalled) {
                flags.installer = 'olm';
                return;
            }
            if (flags.platform === 'openshift' && ctx.isOpenShift4 && isOlmPreinstalled) {
                flags.installer = 'olm';
            }
            else {
                flags.installer = 'operator';
            }
        });
    }
}
exports.default = Deploy;
Deploy.description = 'Deploy CodeReady Workspaces server';
Deploy.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    batch: common_flags_1.batch,
    'listr-renderer': common_flags_1.listrRenderer,
    'deployment-name': common_flags_1.cheDeployment,
    cheimage: flags_1.string({
        char: 'i',
        description: 'CodeReady Workspaces server container image',
        env: 'CHE_CONTAINER_IMAGE'
    }),
    templates: flags_1.string({
        char: 't',
        description: 'Path to the templates folder',
        env: 'CHE_TEMPLATES_FOLDER',
        exclusive: [common_flags_1.DEPLOY_VERSION_KEY],
    }),
    'devfile-registry-url': flags_1.string({
        description: 'The URL of the external Devfile registry.',
        env: 'CHE_WORKSPACE_DEVFILE__REGISTRY__URL'
    }),
    'plugin-registry-url': flags_1.string({
        description: 'The URL of the external plugin registry.',
        env: 'CHE_WORKSPACE_PLUGIN__REGISTRY__URL'
    }),
    cheboottimeout: flags_1.string({
        char: 'o',
        description: 'CodeReady Workspaces server bootstrap timeout (in milliseconds)',
        default: '40000',
        required: true,
        env: 'CHE_SERVER_BOOT_TIMEOUT'
    }),
    [common_flags_1.K8SPODWAITTIMEOUT_KEY]: common_flags_1.k8sPodWaitTimeout,
    [common_flags_1.K8SPODREADYTIMEOUT_KEY]: common_flags_1.k8sPodReadyTimeout,
    [common_flags_1.K8SPODDOWNLOADIMAGETIMEOUT_KEY]: common_flags_1.k8sPodDownloadImageTimeout,
    [common_flags_1.K8SPODERRORRECHECKTIMEOUT_KEY]: common_flags_1.k8sPodErrorRecheckTimeout,
    [common_flags_1.LOG_DIRECTORY_KEY]: common_flags_1.logsDirectory,
    multiuser: command_1.flags.boolean({
        char: 'm',
        description: `Deploys CodeReady Workspaces in multi-user mode.
 		                Note, this option is turned on by default.`,
        default: false
    }),
    tls: command_1.flags.boolean({
        char: 's',
        description: `Deprecated. Enable TLS encryption.
                    Note, this option is turned on by default.
                    To provide own certificate for Kubernetes infrastructure, 'che-tls' secret with TLS certificate must be pre-created in the configured namespace.
                    In case of providing own self-signed certificate 'self-signed-certificate' secret should be also created.
                    For OpenShift, router will use default cluster certificates.
                    Please see the docs how to deploy CodeReady Workspaces on different infrastructures: ${constants_1.DOCS_LINK_INSTALL_RUNNING_CHE_LOCALLY}`,
        hidden: true
    }),
    'self-signed-cert': command_1.flags.boolean({
        description: 'Deprecated. The flag is ignored. Self signed certificates usage is autodetected now.',
        default: false,
        hidden: true
    }),
    platform: flags_1.string({
        char: 'p',
        description: 'Type of OpenShift platform. Valid values are \"openshift\", \"crc (for CodeReady Containers)\".',
        options: ['openshift', 'crc'],
        default: 'openshift'
    }),
    installer: flags_1.string({
        char: 'a',
        description: 'Installer type. If not set, default is "olm" for OpenShift 4.x platform otherwise "operator".',
        options: ['helm', 'operator', 'olm'],
    }),
    debug: flags_1.boolean({
        description: 'Enables the debug mode for CodeReady Workspaces server. To debug CodeReady Workspaces server from localhost use \'server:debug\' command.',
        default: false
    }),
    'che-operator-image': flags_1.string({
        description: 'Container image of the operator. This parameter is used only when the installer is the operator',
    }),
    [common_flags_1.CHE_OPERATOR_CR_YAML_KEY]: common_flags_1.cheOperatorCRYaml,
    [common_flags_1.CHE_OPERATOR_CR_PATCH_YAML_KEY]: common_flags_1.cheOperatorCRPatchYaml,
    'helm-patch-yaml': flags_1.string({
        description: `Path to yaml file with Helm Chart values patch.
                    The file format is identical to values.yaml from the chart.
                    Note, Provided command line arguments take precedence over patch file.`,
        default: '',
    }),
    'workspace-pvc-storage-class-name': flags_1.string({
        description: 'persistent volume(s) storage class name to use to store CodeReady Workspaces workspaces data',
        env: 'CHE_INFRA_KUBERNETES_PVC_STORAGE__CLASS__NAME',
        default: ''
    }),
    'postgres-pvc-storage-class-name': flags_1.string({
        description: 'persistent volume storage class name to use to store CodeReady Workspaces postgres database',
        default: ''
    }),
    'skip-version-check': command_1.flags.boolean({
        description: 'Skip minimal versions check.',
        default: false
    }),
    'skip-cluster-availability-check': command_1.flags.boolean({
        description: 'Skip cluster availability check. The check is a simple request to ensure the cluster is reachable.',
        default: false
    }),
    'auto-update': command_1.flags.boolean({
        description: `Auto update approval strategy for installation CodeReady Workspaces.
                    With this strategy will be provided auto-update CodeReady Workspaces without any human interaction.
                    By default this flag is enabled.
                    This parameter is used only when the installer is 'olm'.`,
        allowNo: true,
        exclusive: ['starting-csv']
    }),
    'starting-csv': command_1.flags.string({
        description: `Starting cluster service version(CSV) for installation CodeReady Workspaces.
                    Flags uses to set up start installation version Che.
                    For example: 'starting-csv' provided with value 'eclipse-che.v7.10.0' for stable channel.
                    Then OLM will install CodeReady Workspaces with version 7.10.0.
                    Notice: this flag will be ignored with 'auto-update' flag. OLM with auto-update mode installs the latest known version.
                    This parameter is used only when the installer is 'olm'.`
    }),
    'olm-channel': flags_1.string({
        description: `Olm channel to install CodeReady Workspaces, f.e. stable.
                    If options was not set, will be used default version for package manifest.
                    This parameter is used only when the installer is the 'olm'.`,
    }),
    'package-manifest-name': flags_1.string({
        description: `Package manifest name to subscribe to CodeReady Workspaces OLM package manifest.
                    This parameter is used only when the installer is the 'olm'.`,
    }),
    'catalog-source-yaml': flags_1.string({
        description: `Path to a yaml file that describes custom catalog source for installation CodeReady Workspaces operator.
                    Catalog source will be applied to the namespace with Che operator.
                    Also you need define 'olm-channel' name and 'package-manifest-name'.
                    This parameter is used only when the installer is the 'olm'.`,
    }),
    'catalog-source-name': flags_1.string({
        description: `OLM catalog source to install CodeReady Workspaces operator.
                    This parameter is used only when the installer is the 'olm'.`
    }),
    'catalog-source-namespace': flags_1.string({
        description: `Namespace for OLM catalog source to install CodeReady Workspaces operator.
                    This parameter is used only when the installer is the 'olm'.`
    }),
    'cluster-monitoring': flags_1.boolean({
        default: false,
        hidden: false,
        description: `Enable cluster monitoring to scrape CodeReady Workspaces metrics in Prometheus.
	                  This parameter is used only when the platform is 'openshift'.`
    }),
    'olm-suggested-namespace': flags_1.boolean({
        default: true,
        allowNo: true,
        description: `Indicate to deploy CodeReady Workspaces in OLM suggested namespace: '${constants_1.DEFAULT_OLM_SUGGESTED_NAMESPACE}'.
                    Flag 'chenamespace' is ignored in this case
                    This parameter is used only when the installer is 'olm'.`
    }),
    'skip-kubernetes-health-check': common_flags_1.skipKubeHealthzCheck,
    'workspace-engine': flags_1.string({
        description: 'Workspace Engine. If not set, default is "che-server". "dev-workspace" is experimental.',
        options: ['che-server', 'dev-workspace'],
        default: 'che-server',
    }),
    'dev-workspace-controller-image': flags_1.string({
        description: 'Container image of the dev workspace controller. This parameter is used only when the workspace engine is the DevWorkspace',
        default: constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_IMAGE,
        env: 'DEV_WORKSPACE_OPERATOR_IMAGE',
    }),
    'dev-workspace-controller-namespace': common_flags_1.devWorkspaceControllerNamespace,
    telemetry: common_flags_1.CHE_TELEMETRY,
    [common_flags_1.DEPLOY_VERSION_KEY]: common_flags_1.cheDeployVersion,
};
//# sourceMappingURL=deploy.js.map