"use strict";
/*********************************************************************
 * Copyright (c) 2019-2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.CheHelper = void 0;
const tslib_1 = require("tslib");
const client_node_1 = require("@kubernetes/client-node");
const axios_1 = require("axios");
const cp = require("child_process");
const cli_ux_1 = require("cli-ux");
const commandExists = require("command-exists");
const fs = require("fs-extra");
const https = require("https");
const yaml = require("js-yaml");
const nodeforge = require("node-forge");
const os = require("os");
const path = require("path");
const rimraf = require("rimraf");
const unzipper = require("unzipper");
const openshift_1 = require("../api/openshift");
const constants_1 = require("../constants");
const util_1 = require("../util");
const che_api_client_1 = require("./che-api-client");
const context_1 = require("./context");
const kube_1 = require("./kube");
class CheHelper {
    constructor(flags) {
        this.flags = flags;
        this.defaultCheResponseTimeoutMs = 3000;
        this.oc = new openshift_1.OpenShiftHelper();
        this.kube = new kube_1.KubeHelper(flags);
        // Make axios ignore untrusted certificate error for self-signed certificate case.
        const httpsAgent = new https.Agent({ rejectUnauthorized: false });
        this.axios = axios_1.default.create({
            httpsAgent
        });
    }
    /**
     * Finds a pod where workspace is running.
     * Rejects if no workspace is found for the given workspace ID
     * or if workspace ID wasn't specified but more than one workspace is found.
     */
    getWorkspacePodName(namespace, cheWorkspaceId) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = kube_1.KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            const res = yield k8sApi.listNamespacedPod(namespace);
            const pods = res.body.items;
            const wsPods = pods.filter(pod => pod.metadata.labels['che.workspace_id'] && pod.metadata.labels['che.original_name'] !== 'che-jwtproxy');
            if (wsPods.length === 0) {
                throw new Error('No workspace pod is found');
            }
            if (cheWorkspaceId) {
                const wsPod = wsPods.find(p => p.metadata.labels['che.workspace_id'] === cheWorkspaceId);
                if (wsPod) {
                    return wsPod.metadata.name;
                }
                throw new Error('Pod is not found for the given workspace ID');
            }
            else {
                if (wsPods.length === 1) {
                    return wsPods[0].metadata.name;
                }
                throw new Error('More than one pod with running workspace is found. Please, specify Workspace ID.');
            }
        });
    }
    getWorkspacePodContainers(namespace, cheWorkspaceId) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = kube_1.KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            const res = yield k8sApi.listNamespacedPod(namespace);
            const pods = res.body.items;
            const wsPods = pods.filter(pod => pod.metadata.labels['che.workspace_id'] && pod.metadata.labels['che.original_name'] !== 'che-jwtproxy');
            if (wsPods.length === 0) {
                throw new Error('No workspace pod is found');
            }
            if (cheWorkspaceId) {
                const wsPod = wsPods.find(p => p.metadata.labels['che.workspace_id'] === cheWorkspaceId);
                if (wsPod) {
                    return wsPod.spec.containers.map(c => c.name);
                }
                throw new Error('Pod is not found for the given workspace ID');
            }
            else {
                if (wsPods.length === 1) {
                    return wsPods[0].spec.containers.map(c => c.name);
                }
                throw new Error('More than one pod with running workspace is found. Please, specify Workspace ID.');
            }
        });
    }
    cheURL(namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!(yield this.kube.getNamespace(namespace))) {
                throw new Error(`ERR_NAMESPACE_NO_EXIST - No namespace ${namespace} is found`);
            }
            const ctx = context_1.ChectlContext.get();
            if (ctx.isOpenShift) {
                return this.cheOpenShiftURL(namespace);
            }
            else {
                return this.cheK8sURL(namespace);
            }
        });
    }
    chePluginRegistryURL(namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            // provided through command line ?
            if (this.flags['plugin-registry-url']) {
                return this.flags['plugin-registry-url'];
            }
            // check
            if (!(yield this.kube.getNamespace(namespace))) {
                throw new Error(`ERR_NAMESPACE_NO_EXIST - No namespace ${namespace} is found`);
            }
            // grab URL
            const ctx = context_1.ChectlContext.get();
            if (ctx.isOpenShift) {
                return this.chePluginRegistryOpenShiftURL(namespace);
            }
            else {
                return this.chePluginRegistryK8sURL(namespace);
            }
        });
    }
    isSelfSignedCertificateSecretExist(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const selfSignedCertSecret = yield this.kube.getSecret(constants_1.CHE_ROOT_CA_SECRET_NAME, namespace);
            return !!selfSignedCertSecret;
        });
    }
    /**
     * Gets self-signed Che CA certificate from 'self-signed-certificate' secret.
     * If secret doesn't exist, undefined is returned.
     */
    retrieveCheCaCert(cheNamespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const cheCaSecretContent = yield this.getCheSelfSignedSecretContent(cheNamespace);
            if (!cheCaSecretContent) {
                return;
            }
            const pemBeginHeader = '-----BEGIN CERTIFICATE-----';
            const pemEndHeader = '-----END CERTIFICATE-----';
            const certRegExp = new RegExp(`(^${pemBeginHeader}$(?:(?!${pemBeginHeader}).)*^${pemEndHeader}$)`, 'mgs');
            const certsPem = cheCaSecretContent.match(certRegExp);
            const caCertsPem = [];
            if (certsPem) {
                for (const certPem of certsPem) {
                    const cert = nodeforge.pki.certificateFromPem(certPem);
                    const basicConstraintsExt = cert.getExtension('basicConstraints');
                    if (basicConstraintsExt && basicConstraintsExt.cA) {
                        caCertsPem.push(certPem);
                    }
                }
            }
            return caCertsPem.join('\n');
        });
    }
    /**
     * Retrieves content of Che self-signed-certificate secret or undefined if the secret doesn't exist.
     * Note, it contains certificate chain in pem format.
     */
    getCheSelfSignedSecretContent(cheNamespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const cheCaSecret = yield this.kube.getSecret(constants_1.CHE_ROOT_CA_SECRET_NAME, cheNamespace);
            if (!cheCaSecret) {
                return;
            }
            if (cheCaSecret.data && cheCaSecret.data['ca.crt']) {
                return util_1.base64Decode(cheCaSecret.data['ca.crt']);
            }
            throw new Error(`Secret "${constants_1.CHE_ROOT_CA_SECRET_NAME}" has invalid format: "ca.crt" key not found in data.`);
        });
    }
    saveCheCaCert(cheCaCert, destination) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const cheCaCertFile = this.getTargetFile(destination);
            fs.writeFileSync(cheCaCertFile, cheCaCert);
            return cheCaCertFile;
        });
    }
    /**
     * Handles certificate target location and returns string which points to the target file.
     */
    getTargetFile(destination) {
        if (!destination) {
            return path.join(os.tmpdir(), constants_1.DEFAULT_CA_CERT_FILE_NAME);
        }
        if (fs.existsSync(destination)) {
            return fs.lstatSync(destination).isDirectory() ? path.join(destination, constants_1.DEFAULT_CA_CERT_FILE_NAME) : destination;
        }
        throw new Error(`Given path \'${destination}\' doesn't exist.`);
    }
    /**
     * Retrieves Keycloak admin user credentials.
     * Works only with installers which use Che CR (operator, olm).
     * Returns credentials as an array of two values: [login, password]
     * In case of an error an array with undefined values will be returned.
     */
    retrieveKeycloakAdminCredentials(cheNamespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let adminUsername;
            let adminPassword;
            const cheCluster = yield this.kube.getCheCluster(cheNamespace);
            if (!cheCluster || cheCluster.spec.auth.externalIdentityProvider) {
                return [];
            }
            const keycloakCredentialsSecretName = cheCluster.spec.auth.identityProviderSecret;
            if (keycloakCredentialsSecretName) {
                // Keycloak credentials are stored in secret
                const keycloakCredentialsSecret = yield this.kube.getSecret(keycloakCredentialsSecretName, cheNamespace);
                if (keycloakCredentialsSecret && keycloakCredentialsSecret.data) {
                    adminUsername = util_1.base64Decode(keycloakCredentialsSecret.data.user);
                    adminPassword = util_1.base64Decode(keycloakCredentialsSecret.data.password);
                }
            }
            else {
                // Keycloak credentials are stored in Che custom resource
                adminUsername = cheCluster.spec.auth.identityProviderAdminUserName;
                adminPassword = cheCluster.spec.auth.identityProviderPassword;
            }
            return [adminUsername, adminPassword];
        });
    }
    chePluginRegistryK8sURL(namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (yield this.kube.ingressExist('plugin-registry', namespace)) {
                const protocol = yield this.kube.getIngressProtocol('plugin-registry', namespace);
                const hostname = yield this.kube.getIngressHost('plugin-registry', namespace);
                return `${protocol}://${hostname}`;
            }
            throw new Error(`ERR_INGRESS_NO_EXIST - No ingress 'plugin-registry' in namespace ${namespace}`);
        });
    }
    chePluginRegistryOpenShiftURL(namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (yield this.oc.routeExist('plugin-registry', namespace)) {
                const protocol = yield this.oc.getRouteProtocol('plugin-registry', namespace);
                const hostname = yield this.oc.getRouteHost('plugin-registry', namespace);
                return `${protocol}://${hostname}`;
            }
            throw new Error(`ERR_ROUTE_NO_EXIST - No route 'plugin-registry' in namespace ${namespace}`);
        });
    }
    cheK8sURL(namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const ingress_names = ['che', 'che-ingress'];
            for (const ingress_name of ingress_names) {
                if (yield this.kube.ingressExist(ingress_name, namespace)) {
                    const protocol = yield this.kube.getIngressProtocol(ingress_name, namespace);
                    const hostname = yield this.kube.getIngressHost(ingress_name, namespace);
                    return `${protocol}://${hostname}`;
                }
            }
            throw new Error(`ERR_INGRESS_NO_EXIST - No ingress ${ingress_names} in namespace ${namespace}`);
        });
    }
    cheOpenShiftURL(namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const route_names = ['codeready', 'che-host'];
            for (const route_name of route_names) {
                if (yield this.oc.routeExist(route_name, namespace)) {
                    const protocol = yield this.oc.getRouteProtocol(route_name, namespace);
                    const hostname = yield this.oc.getRouteHost(route_name, namespace);
                    return `${protocol}://${hostname}`;
                }
            }
            throw new Error(`ERR_ROUTE_NO_EXIST - No route ${route_names} in namespace ${namespace}`);
        });
    }
    createWorkspaceFromDevfile(cheApiEndpoint, devfilePath, workspaceName, accessToken) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let devfile;
            try {
                devfile = yield this.parseDevfile(devfilePath);
                if (workspaceName) {
                    let json = yaml.load(devfile);
                    json.metadata.name = workspaceName;
                    devfile = yaml.dump(json);
                }
            }
            catch (error) {
                if (!devfile) {
                    throw new Error(`E_NOT_FOUND_DEVFILE - ${devfilePath} - ${error.message}`);
                }
            }
            const cheApi = che_api_client_1.CheApiClient.getInstance(cheApiEndpoint);
            return cheApi.createWorkspaceFromDevfile(devfile, accessToken);
        });
    }
    parseDevfile(devfilePath = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (devfilePath.startsWith('http')) {
                const response = yield this.axios.get(devfilePath);
                return response.data;
            }
            else {
                return fs.readFileSync(devfilePath, 'utf8');
            }
        });
    }
    buildDashboardURL(ideURL) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return ideURL.replace(/\/[^/|.]*\/[^/|.]*$/g, '\/dashboard\/#\/ide$&');
        });
    }
    /**
     * Finds workspace pods and reads logs from it.
     */
    readWorkspacePodLog(namespace, workspaceId, directory) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const podLabelSelector = `che.workspace_id=${workspaceId}`;
            let workspaceIsRun = false;
            const pods = yield this.kube.listNamespacedPod(namespace, undefined, podLabelSelector);
            if (pods.items.length) {
                workspaceIsRun = true;
            }
            for (const pod of pods.items) {
                for (const containerStatus of pod.status.containerStatuses) {
                    workspaceIsRun = workspaceIsRun && !!containerStatus.state && !!containerStatus.state.running;
                }
            }
            const follow = !workspaceIsRun;
            yield this.readPodLog(namespace, podLabelSelector, directory, follow);
            yield this.readNamespaceEvents(namespace, directory, follow);
            return workspaceIsRun;
        });
    }
    /**
     * Reads logs from pods that match a given selector.
     */
    readPodLog(namespace, podLabelSelector, directory, follow) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (follow) {
                yield this.watchNamespacedPods(namespace, podLabelSelector, directory);
            }
            else {
                yield this.readNamespacedPodLog(namespace, podLabelSelector, directory);
            }
        });
    }
    /**
     * Reads containers logs inside pod that match a given selector.
     */
    readNamespacedPodLog(namespace, podLabelSelector, directory) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const pods = yield this.kube.listNamespacedPod(namespace, undefined, podLabelSelector);
            for (const pod of pods.items) {
                if (!pod.status || !pod.status.containerStatuses) {
                    return;
                }
                const podName = pod.metadata.name;
                for (const containerName of this.getContainers(pod)) {
                    const fileName = this.doCreateLogFile(namespace, podName, containerName, directory);
                    yield this.doReadNamespacedPodLog(namespace, podName, containerName, fileName, false);
                }
            }
        });
    }
    /**
     * Reads all namespace events and store into a file.
     */
    readNamespaceEvents(namespace, directory, follow) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const fileName = path.resolve(directory, namespace, 'events.txt');
            fs.ensureFileSync(fileName);
            const cli = (commandExists.sync('kubectl') && 'kubectl') || (commandExists.sync('oc') && 'oc');
            if (cli) {
                const command = 'get events';
                const namespaceParam = `-n ${namespace}`;
                const watchParam = follow && '--watch' || '';
                cp.exec(`${cli} ${command} ${namespaceParam} ${watchParam} >> ${fileName}`);
            }
            else {
                throw new Error('No events are collected. \'kubectl\' or \'oc\' is required to perform the task.');
            }
        });
    }
    watchNamespacedPods(namespace, podLabelSelector, directory) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const processedContainers = new Map();
            const watcher = new client_node_1.Watch(kube_1.KubeHelper.KUBE_CONFIG);
            return watcher.watch(`/api/v1/namespaces/${namespace}/pods`, {}, (_phase, obj) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const pod = obj;
                if (!pod || !pod.metadata || !pod.metadata.name) {
                    return;
                }
                const podName = pod.metadata.name;
                if (!processedContainers.has(podName)) {
                    processedContainers.set(podName, new Set());
                }
                if (!podLabelSelector || this.matchLabels(pod.metadata.labels || {}, podLabelSelector)) {
                    for (const containerName of this.getContainers(pod)) {
                        // not to read logs from the same containers twice
                        if (!processedContainers.get(podName).has(containerName)) {
                            processedContainers.get(podName).add(containerName);
                            const fileName = this.doCreateLogFile(namespace, podName, containerName, directory);
                            yield this.doReadNamespacedPodLog(namespace, pod.metadata.name, containerName, fileName, true);
                        }
                    }
                }
            }), 
            // ignore errors
            () => { });
        });
    }
    /**
     * Wait until workspace is in 'Active` state.
     */
    waitNamespaceActive(namespaceName, intervalMs = 500, timeoutMs = 60000) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                const namespace = yield this.kube.getNamespace(namespaceName);
                if (namespace && namespace.status && namespace.status.phase && namespace.status.phase === 'Active') {
                    return;
                }
                yield cli_ux_1.cli.wait(intervalMs);
            }
            throw new Error(`ERR_TIMEOUT: ${namespaceName} is not 'Active'.`);
        });
    }
    /**
     * Indicates if pod matches given labels.
     */
    matchLabels(podLabels, podLabelSelector) {
        const labels = podLabelSelector.split(',');
        for (const label of labels) {
            if (label) {
                const keyValue = label.split('=');
                if (podLabels[keyValue[0]] !== keyValue[1]) {
                    return false;
                }
            }
        }
        return true;
    }
    /**
     * Returns containers names.
     */
    getContainers(pod) {
        if (!pod.status || !pod.status.containerStatuses) {
            return [];
        }
        return pod.status.containerStatuses.map(containerStatus => containerStatus.name);
    }
    /**
     * Reads pod log from a specific container of the pod.
     */
    doReadNamespacedPodLog(namespace, podName, containerName, fileName, follow) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (follow) {
                try {
                    yield this.kube.readNamespacedPodLog(podName, namespace, containerName, fileName, follow);
                }
                catch (_a) {
                    // retry in 200ms, container might not be started
                    setTimeout(() => tslib_1.__awaiter(this, void 0, void 0, function* () { return this.doReadNamespacedPodLog(namespace, podName, containerName, fileName, follow); }), 200);
                }
            }
            else {
                yield this.kube.readNamespacedPodLog(podName, namespace, containerName, fileName, follow);
            }
        });
    }
    doCreateLogFile(namespace, podName, containerName, directory) {
        const fileName = path.resolve(directory, namespace, podName, `${containerName}.log`);
        fs.ensureFileSync(fileName);
        return fileName;
    }
    /**
     * Gets install templates for given installer.
     * @param installer Che installer
     * @param url link to zip archive with sources of Che operator
     * @param destDir destination directory into which the templates should be unpacked
     */
    downloadAndUnpackTemplates(installer, url, destDir) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            // Add codeready-operator folder for operator templates
            if (installer === 'operator') {
                destDir = path.join(destDir, constants_1.OPERATOR_TEMPLATE_DIR);
            }
            // No need to add kubernetes folder for Helm installer as it already present in the archive
            const tempDir = path.join(os.tmpdir(), Date.now().toString());
            yield fs.mkdirp(tempDir);
            const zipFile = path.join(tempDir, `che-templates-${installer}.zip`);
            yield util_1.downloadFile(url, zipFile);
            yield this.unzipTemplates(zipFile, destDir);
            // Clean up zip. Do not wait when finishes.
            rimraf(tempDir, () => { });
        });
    }
    /**
     * Unpacks repository deploy templates into specified folder
     * @param zipFile path to zip archive with source code
     * @param destDir target directory into which templates should be unpacked
     */
    unzipTemplates(zipFile, destDir) {
        var e_1, _a;
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            // Gets path from: repo-name/deploy/path
            const deployDirRegex = new RegExp('(?:^[\\\w-]*\\\/deploy\\\/)(.*)');
            const zip = fs.createReadStream(zipFile).pipe(unzipper.Parse({ forceStream: true }));
            try {
                for (var zip_1 = tslib_1.__asyncValues(zip), zip_1_1; zip_1_1 = yield zip_1.next(), !zip_1_1.done;) {
                    const entry = zip_1_1.value;
                    const entryPathInZip = entry.path;
                    const templatesPathMatch = entryPathInZip.match(deployDirRegex);
                    if (templatesPathMatch && templatesPathMatch.length > 1 && templatesPathMatch[1]) {
                        // Remove prefix from in-zip path
                        const entryPathWhenExtracted = templatesPathMatch[1];
                        // Path to the item in target location
                        const dest = path.join(destDir, entryPathWhenExtracted);
                        // Extract item
                        if (entry.type === 'File') {
                            const parentDirName = path.dirname(dest);
                            if (!fs.existsSync(parentDirName)) {
                                yield fs.mkdirp(parentDirName);
                            }
                            entry.pipe(fs.createWriteStream(dest));
                        }
                        else if (entry.type === 'Directory') {
                            if (!fs.existsSync(dest)) {
                                yield fs.mkdirp(dest);
                            }
                            // The folder is created above
                            entry.autodrain();
                        }
                        else {
                            // Ignore the item as we do not need to handle links and etc.
                            entry.autodrain();
                        }
                    }
                    else {
                        // No need to extract this item
                        entry.autodrain();
                    }
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (zip_1_1 && !zip_1_1.done && (_a = zip_1.return)) yield _a.call(zip_1);
                }
                finally { if (e_1) throw e_1.error; }
            }
        });
    }
}
exports.CheHelper = CheHelper;
//# sourceMappingURL=che.js.map